/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.deploy.actions;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.profileservice.management.actions.AbstractTwoPhaseModificationAction;
import org.jboss.profileservice.plugins.deploy.DeploymentConstants;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.action.deployment.DeploymentAction;
import org.jboss.profileservice.spi.action.deployment.DeploymentActionContext;
import org.jboss.profileservice.spi.managed.ManagedProfile;

/**
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public abstract class AbstractDeploymentLifeCycleAction extends AbstractTwoPhaseModificationAction<DeploymentActionContext>
   implements DeploymentAction<DeploymentActionContext>
{

   static final Logger log = Logger.getLogger(DeploymentConstants.DEPLOYMENT_LOGGER_CATEGORY);
   
   private final Collection<String> deploymentNames;
   private final ManagedProfile managed;
   private final List<DeploymentState> deployments = new ArrayList<DeploymentState>();
   
   public AbstractDeploymentLifeCycleAction(ManagedProfile managed, Collection<String> names, DeploymentActionContext modificationContext)
   {
      super(modificationContext);
      this.deploymentNames = names;
      this.managed = managed;
   }

   /**
    * Add a deployment.
    * 
    * @param deployment the deployment
    * @return the deployment state
    */
   protected DeploymentState addDeployment(ProfileDeployment deployment)
   {
      DeploymentState state = new DeploymentState(deployment);
      this.deployments.add(state);
      return state;
   }
   
   public Collection<String> getDeploymentNames()
   {
      return deploymentNames;
   }
   
   /**
    * Get the deployment states.
    * 
    * @return the deployment states
    */
   protected List<DeploymentState> getDeployments()
   {
      return deployments;
   }
   
   /**
    * Get the managed profile
    * 
    * @return the profile
    */
   protected ManagedProfile getManagedProfile()
   {
      return managed;
   }
   
   /**
    * Resolve the deployment names.
    * 
    * @return the deployment names
    */
   protected String[] resolvedNames()
   {
      Set<String> set = new HashSet<String>();
      for(DeploymentState state : deployments)
      {
         set.add(state.getName());
      }
      return set.toArray(new String[set.size()]);
   }
 
   static class DeploymentState
   {

      private final ProfileDeployment deployment;
      private final boolean deployed;
      
      public DeploymentState(ProfileDeployment deployment)
      {
         this.deployment = deployment;
         this.deployed = deployment.getDeploymentInfo().isDeployed();
      }
      
      public ProfileDeployment getDeployment()
      {
         return deployment;
      }
      
      public String getName()
      {
         return this.deployment.getName();
      }

      public boolean isDeployed()
      {
         return deployed;
      }
      
   }
   
}

