/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.deploy.actions;

import java.net.URL;

import org.jboss.deployers.spi.management.deploy.DeploymentID;
import org.jboss.profileservice.management.event.ProfileModificationEvent;
import org.jboss.profileservice.plugins.deploy.TransientDeploymentMetaData;
import org.jboss.profileservice.plugins.deploy.TransientProfileActivator;
import org.jboss.profileservice.repository.ProfileDeploymentFactory;
import org.jboss.profileservice.spi.MutableProfile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.action.ProfileModificationResponse;
import org.jboss.profileservice.spi.action.ProfileModificationType;
import org.jboss.profileservice.spi.action.deployment.DeploymentAction;
import org.jboss.profileservice.spi.action.deployment.DeploymentActionContext;
import org.jboss.profileservice.spi.deployment.ProfileDeploymentFlag;
import org.jboss.vfs.VFS;
import org.jboss.vfs.VirtualFile;

/**
 * Add a deployment to a profile. This action only targets to add transient deployments, which is handled
 * by the profile created through the {@code TransientProfileActivator}. This profile does not provide any
 * persistence, so references to deployments will be lost after a restart.  
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class DeploymentAddAction extends AbstractDeploymentAddAction implements DeploymentAction<DeploymentActionContext>
{
   
   /** The {@code ProfileDeployment} factory. */
   private static final ProfileDeploymentFactory deploymentFactory = ProfileDeploymentFactory.getInstance();

   public DeploymentAddAction(DeploymentID dtID, MutableProfile profile, DeploymentActionContext modificationContext)
   {
      super(dtID, profile, modificationContext);
   }

   protected void doCancel()
   {
      //
   }

   protected void doCommit(ProfileModificationResponse response)
   {
      //
   }

   protected void doComplete(ProfileModificationResponse response) throws Exception
   {
      //
   }

   protected boolean doPrepare(ProfileModificationResponse response)
   {
      try
      {
         final String name = getDeploymentName();
         final URL url = getDeploymentID().getContentURL();
         if(url == null)
         {
            return false;
         }
         //
         final VirtualFile vf = VFS.getChild(url);
         // Create the deployment directly
         final ProfileDeployment deployment = deploymentFactory.createDeployment(vf, new TransientDeploymentMetaData(name));
         deployment.getDeploymentInfo().setFlag(ProfileDeploymentFlag.LOCKED);
         // Attach original deployment url for resolving deployment names
         deployment.getTransientAttachments().putAttachment(TransientProfileActivator.TRANSIENT_DEPLOYMENT_URL_KEY, url.toExternalForm());
         getProfile().addDeployment(deployment);
         
         // Notify
         response.fireModificationEvent(ProfileModificationEvent.create(ProfileModificationType.ADD, getProfile().getKey()));
      }
      catch(Exception e)
      {
         response.setFailure(e);
         return false;
      }
      return true;
   }

   protected void doRollbackFromActive()
   {
      //       
   }

   protected void doRollbackFromCancelled()
   {
      //       
   }

   protected void doRollbackFromComplete()
   {
      // 
   }

   protected void doRollbackFromPrepared()
   {
      final String name = getDeploymentName();
      if(getProfile().hasDeployment(name))
      {
         try
         {
            getProfile().removeDeployment(name);
         }
         catch(Exception e)
         {
            log.error("failed to rollback changes " + name);
         }
      }
   }

   protected void doRollbackFromRollbackOnly()
   {
      // 
   }

}

