/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.deployers.spi.management.NameMatcher;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.profileservice.management.event.ProfileLifeCycleEvent;
import org.jboss.profileservice.management.event.ProfileModificationEvent;
import org.jboss.profileservice.management.event.ProfileLifeCycleEvent.LifeCycleState;
import org.jboss.profileservice.persistence.repository.PersistenceRepository;
import org.jboss.profileservice.plugins.management.actions.ProfileViewUpdateAction;
import org.jboss.profileservice.plugins.management.actions.RemoveComponentAction;
import org.jboss.profileservice.plugins.management.actions.UpdateComponentAction;
import org.jboss.profileservice.plugins.management.util.AbstractManagementProxyFactory;
import org.jboss.profileservice.plugins.management.view.RegisteredProfileView;
import org.jboss.profileservice.plugins.spi.ProfileView;
import org.jboss.profileservice.plugins.spi.ProfileViewWrapper;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.action.ActionController;
import org.jboss.profileservice.spi.action.EventBus;
import org.jboss.profileservice.spi.action.ModificationEvent;
import org.jboss.profileservice.spi.action.ModificationStatus;
import org.jboss.profileservice.spi.action.ProfileModificationAction;
import org.jboss.profileservice.spi.action.ProfileModificationContext;
import org.jboss.profileservice.spi.action.ProfileModificationType;
import org.jboss.profileservice.spi.managed.ManagedProfile;
import org.jboss.util.collection.ConcurrentSet;


/**
 * A wrapper to handle multiple registered <code>ProfileView</code>s.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class RegisteredProfileViewsWrapper implements ProfileViewWrapper
{
   
   /** The profiles. */
   private Set<ProfileKey> registeredKeys = new ConcurrentSet<ProfileKey>();
   private Map<ProfileKey, RegisteredProfileView> viewsByName = new ConcurrentHashMap<ProfileKey, RegisteredProfileView>();  
   
   /** The managed proxy factory. */
   private AbstractManagementProxyFactory proxyFactory;
   
   /** The persistence repository. */
   private PersistenceRepository persistenceRepository;
   
   /** The action controller. */
   private final ActionController controller;
   
   public RegisteredProfileViewsWrapper(ActionController controller, PersistenceRepository persistenceRepository)
   {
      if(controller == null)
      {
         throw new IllegalArgumentException("null action controller");
      }
      if(persistenceRepository == null)
      {
         throw new IllegalArgumentException("null persistence repository");
      }
      this.controller = controller;
      this.persistenceRepository = persistenceRepository;
   }
   
   public synchronized boolean load()
   {
      boolean changed = false;
      
      Set<ProfileKey> have = registeredKeys;
      this.registeredKeys = new ConcurrentSet<ProfileKey>();
      List<ProfileKey> keys = controller.getActiveProfiles();
      // Additions
      for(ProfileKey key : keys)
      {
         boolean exists = have.remove(key);
         registeredKeys.add(key);
         
         if(exists == false)
         {
            RegisteredProfileView view = createProfileView(key); 
            viewsByName.put(key, view);
            getEventBus().addListener(view);
         }
      }
      // Removals
      for(ProfileKey key : have)
      {
         ProfileView view = viewsByName.remove(key);
         if(view != null)
         {
            getEventBus().removeListener(view);
            changed = true;            
         }
      }
      // Load and process
      for(ProfileKey key : keys)
      {
         RegisteredProfileView view = viewsByName.get(key);
         if(view.load())
         {
            ManagedProfile managedProfile = controller.getManagedProfile(key);
            ProfileModificationAction<ProfileModificationContext> action = new ProfileViewUpdateAction(managedProfile, view, proxyFactory);

            controller.perform(key, ProfileModificationType.GET, Collections.singletonList(action));

            changed = true;
         }
      }
      //
      return changed;
   }
   
   RegisteredProfileView createProfileView(ProfileKey key)
   {
      return new RegisteredProfileView(key);
   }

   protected Collection<RegisteredProfileView> getViews()
   {
      return this.viewsByName.values();
   }
   
   protected EventBus getEventBus()
   {
      // TODO
      return EventBus.class.cast(controller);
   }
   
   /**
    * {@inheritDoc}
    */
   public void setManagementProxyFactory(AbstractManagementProxyFactory proxyFactory)
   {
      this.proxyFactory = proxyFactory;
   }

   /**
    * {@inheritDoc}
    */
   public ManagedComponent getComponent(String name, ComponentType type)
   {
      ManagedComponent component = null;
      for(final ProfileView view : getViews())
      {
         ManagedComponent resolved = view.getComponent(name, type);
         if(resolved != null)
         {
            component = resolved;
         }
         // TODO fail on duplicates ?
         // maybe should be done by the top level view?
      }
      return component;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ComponentType> getComponentTypes()
   {
      Set<ComponentType> types = new HashSet<ComponentType>();
      for(final ProfileView view : getViews())
      {
         types.addAll(view.getComponentTypes());
      }
      return types;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getComponentsForType(ComponentType type)
   {
      Set<ManagedComponent> components = new HashSet<ManagedComponent>();
      for(final ProfileView view : getViews())
      {
         components.addAll(view.getComponentsForType(type));
      }
      return components;
   }

   /**
    * {@inheritDoc}
    */
   public Collection<String> getDeploymentNames()
   {
      Set<String> deployments = new TreeSet<String>();
      for(final ProfileView view : getViews())
      {
         deployments.addAll(view.getDeploymentNames());
      }
      return deployments;
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getDeploymentNamesForType(String type)
   {
      Set<String> deployments = new TreeSet<String>();
      for(final ProfileView view : getViews())
      {
         deployments.addAll(view.getDeploymentNamesForType(type));
      }
      return deployments;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getDeploymentsForType(String type)
   {
      Set<ManagedDeployment> deployments = new HashSet<ManagedDeployment>();
      for(final ProfileView view : getViews())
      {
         deployments.addAll(view.getDeploymentsForType(type));
      }
      return deployments;
   }

   /**
    * {@inheritDoc}
    */
   public ManagedDeployment getManagedDeployment(String name)
   {
      ManagedDeployment deployment = null;
      for(final ProfileView view : getViews())
      {
         final ManagedDeployment resolved = view.getManagedDeployment(name);
         if(resolved != null)
         {
            deployment = resolved;
         }
      }
      return deployment;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getMatchingComponents(String name, ComponentType type,
         NameMatcher<ManagedComponent> matcher)
   {
      Set<ManagedComponent> components = new HashSet<ManagedComponent>();
      for(final ProfileView view : getViews())
      {
         components.addAll(view.getMatchingComponents(name, type, matcher));
      }
      return components;
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getMatchingDeploymentName(String regex)
   {
      Set<String> deployments = new TreeSet<String>();
      for(final ProfileView view : getViews())
      {
         deployments.addAll(view.getMatchingDeploymentName(regex));
      }
      return deployments;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getMatchingDeployments(String name, NameMatcher<ManagedDeployment> matcher)
   {
      Set<ManagedDeployment> deployments = new HashSet<ManagedDeployment>();
      for(final ProfileView view : getViews())
      {
         deployments.addAll(view.getMatchingDeployments(name, matcher));
      }
      return deployments;
   }

   /**
    * {@inheritDoc}
    */
   public void notify(final ModificationEvent event)
   {
      if(event instanceof ProfileLifeCycleEvent)
      {
         ProfileLifeCycleEvent lifeCycle = ProfileLifeCycleEvent.class.cast(event);
         if(lifeCycle.getState() == LifeCycleState.UNINSTALLED)
         {
            // TODO remove ?
         }
      }
   }

   public ProfileKey resolveProfile(String deploymentName)
   {
      for(RegisteredProfileView view : getViews())
      {
         if(view.getDeploymentNames().contains(deploymentName))
         {
            return view.getKey();
         }
      }
      return null;     
   }
   
   public void removeComponent(ManagedComponent update, ManagedComponent original) throws Exception
   {
      ManagedDeployment deployment = original.getDeployment();
      ProfileKey key = resolveProfile(deployment.getName());
      
      ProfileModificationAction updateComponentAction = new RemoveComponentAction(original, persistenceRepository, null);
      List<ProfileModificationAction<ProfileModificationContext>> actions = new ArrayList<ProfileModificationAction<ProfileModificationContext>>();
      actions.add(updateComponentAction);
      ModificationStatus status = this.controller.perform(key, ProfileModificationType.UPDATE, actions);
      
      getEventBus().fireModificationEvent(new ProfileModificationEvent(ProfileModificationType.DELETE, key));
      
      if(status.isFailed())
      {
         rethrowAsException(status.getFailure());
      }
   }
   
   public void updateComponent(ManagedComponent update, ManagedComponent original) throws Exception
   {
      ManagedDeployment deployment = original.getDeployment();
      ProfileKey key = resolveProfile(deployment.getName());
      
      ProfileModificationAction updateComponentAction = new UpdateComponentAction(update, original, proxyFactory, persistenceRepository, null);
      List<ProfileModificationAction<ProfileModificationContext>> actions = new ArrayList<ProfileModificationAction<ProfileModificationContext>>();
      actions.add(updateComponentAction);
      ModificationStatus status = this.controller.perform(key, ProfileModificationType.UPDATE, actions);
      
      getEventBus().fireModificationEvent(new ProfileModificationEvent(ProfileModificationType.UPDATE, key));
      
      if(status.isFailed())
      {
         rethrowAsException(status.getFailure());
      }
   }
 
   static void rethrowAsException(Throwable t) throws Exception
   {
      if(t instanceof Exception)
      {
         throw Exception.class.cast(t);
      }
      else
      {
         throw new UndeclaredThrowableException(t);
      }
   }
   
}

