/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.deployers.spi.management.ContextStateMapper;
import org.jboss.deployers.spi.management.ManagedComponentRuntimeDispatcher;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedOperation;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedComponent;
import org.jboss.managed.api.MutableManagedObject;
import org.jboss.managed.api.RunState;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementObject;
import org.jboss.managed.api.annotation.ViewUse;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.profileservice.plugins.spi.ProfileViewProcessingContext;

/**
 * A helper to process runtime managed objects.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public abstract class ManagedObjectRuntimeProcessor extends ManagedObjectProcessor
{

   /** The logger. */
   private static final Logger log = Logger.getLogger(ManagedObjectRuntimeProcessor.class);

   /** A map of runtime ManagedObjects needing to be merged with their matching ManagedObject. */
   private Map<String, ManagedObject> runtimeMOs = new HashMap<String, ManagedObject>();

   /** The state mappings. */
   private static final ContextStateMapper<RunState> runStateMapper;

   /** The runtime component dispatcher. */
   private final ManagedComponentRuntimeDispatcher dispatcher;

   /** The proxy factory. */
   private final AbstractManagementProxyFactory proxyFactory;

   static
   {
      // Set default run state mappings for mc beans/mbeans
      Map<String, RunState> runStateMappings = new HashMap<String, RunState>();
      runStateMappings.put("**ERROR**", RunState.FAILED);
      runStateMappings.put("Not Installed", RunState.STOPPED);
      runStateMappings.put("PreInstall", RunState.STOPPED);
      runStateMappings.put("Described", RunState.STOPPED);
      runStateMappings.put("Instantiated", RunState.STOPPED);
      runStateMappings.put("Configured", RunState.STOPPED);
      runStateMappings.put("Create", RunState.STOPPED);
      runStateMappings.put("Start", RunState.STOPPED);
      runStateMappings.put("Installed", RunState.RUNNING);

      runStateMapper = new DefaultContextStateMapper<RunState>(runStateMappings, RunState.STARTING, RunState.STOPPED,
            RunState.FAILED, RunState.UNKNOWN);
   }

   public ManagedObjectRuntimeProcessor(AbstractManagementProxyFactory proxyFactory)
   {
      if (proxyFactory == null)
         throw new IllegalArgumentException("null proxy factory");
      if (proxyFactory.getDispatcher() == null)
         throw new IllegalArgumentException("null runtime component dispatcher");

      this.proxyFactory = proxyFactory;
      this.dispatcher = proxyFactory.getDispatcher();
   }
   
   protected void processManagedObject(String key, ManagedObject mo, ManagedDeployment md,
         ProfileViewProcessingContext context) throws Exception
   {
      boolean trace = log.isTraceEnabled();

      ManagementObject managementObject = getAnnotation(mo, ManagementObject.class);
      if (managementObject.isRuntime())
      {
         boolean merged = false;
         ManagementComponent mc = managementObject.componentType();
         boolean isMC = !(mc.type().length() == 0 && mc.subtype().length() == 0);

         // Merge this with the ManagedObject
         ManagedObject parentMO = getRegisteredManagedObject(key);
         if (parentMO == null && isMC == false)
         {
            if (trace)
            {
               log.trace("Deferring resolution of runtime ManagedObject: " + managementObject);
            }
            // Save the runtime mo for merging
            runtimeMOs.put(key, mo);
         }
         else
         {
            mergeRuntimeMO(parentMO, mo);
            merged = true;
            runtimeMOs.remove(key);
         }
         // Update the runtime state of any ManagedComponent associated with this runtime mo
         if (md != null && md.getComponents() != null)
         {
            ManagedComponent comp = md.getComponent(mo.getName());
            if (comp != null)
            {
               RunState state = updateRunState(mo, comp);
               if (trace)
               {
                  log.trace("Updated component: " + comp + " run state to: " + state);
               }

            }
         }

         // There is no further processing of runtime ManagedObjects, unless its marked as a component
         if (isMC == false)
            return;
         // 
         else if (merged == false)
         {
            Set<ManagedOperation> runtimeOps = mo.getOperations();
            runtimeOps = createOperationProxies(mo, runtimeOps);
            MutableManagedObject moi = (MutableManagedObject) mo;
            moi.setOperations(runtimeOps);
         }
      }
      else
      {
         // See if there is runtime info to merge
         ManagedObject runtimeMO = runtimeMOs.get(key);
         if (runtimeMO != null)
         {
            mergeRuntimeMO(mo, runtimeMO);
            runtimeMOs.remove(key);
            // Update the runtime state of any ManagedComponent associated with this runtime mo
            if (md != null && md.getComponents() != null)
            {
               ManagedComponent comp = md.getComponent(mo.getName());
               if (comp != null)
               {
                  RunState state = updateRunState(runtimeMO, comp);
                  if (trace)
                  {
                     log.trace("Updated component: " + comp + " run state to: " + state);
                  }
               }
            }
         }
      }
      // Process the managed object
      super.processManagedObject(key, mo, md, context);
   }

   /**
    * Merge the runtime ManagedObject
    * 
    * @param mo the managed object
    * @param runtimeMO the runtime managed object
    * @throws Exception
    */
   protected void mergeRuntimeMO(ManagedObject mo, ManagedObject runtimeMO) throws Exception
   {
      boolean trace = log.isTraceEnabled();

      Map<String, ManagedProperty> runtimeProps = runtimeMO.getProperties();
      Set<ManagedOperation> runtimeOps = runtimeMO.getOperations();
      // Get the runtime MO component name
      Object componentName = runtimeMO.getComponentName();
      if (trace)
      {
         log.trace("Merging runtime: " + runtimeMO.getName() + ", compnent name: " + componentName);
      }
      Map<String, ManagedProperty> moProps = null;
      Set<ManagedOperation> moOps = null;
      HashMap<String, ManagedProperty> props = null;
      HashSet<ManagedOperation> ops = null;
      // If mo is null, the merge target is the runtimeMO
      if (mo == null)
      {
         // Just proxy the runtime props/ops
         mo = runtimeMO;
         moProps = mo.getProperties();
         moOps = mo.getOperations();
         // These will be updated with the proxied values, don't duplicate props/ops
         props = new HashMap<String, ManagedProperty>();
         ops = new HashSet<ManagedOperation>();
      }
      else
      {
         // Merge the runtime props/ops
         moProps = mo.getProperties();
         moOps = mo.getOperations();
         props = new HashMap<String, ManagedProperty>(moProps);
         ops = new HashSet<ManagedOperation>(moOps);
      }

      if (runtimeProps != null && runtimeProps.size() > 0)
      {
         if (trace)
         {
            log.trace("Properties before:" + props);
         }
         // We need to pull the runtime values for stats
         for (ManagedProperty prop : runtimeProps.values())
         {
            if (prop.hasViewUse(ViewUse.STATISTIC))
            {
               String propName = prop.getMappedName();
               try
               {
                  // RuntimeComponentDispatcher.setActiveProperty(prop);
                  MetaValue propValue = dispatcher.get(componentName, prop);
                  if (propValue != null)
                     prop.setValue(propValue);
               }
               catch (Throwable t)
               {
                  log.debug("Failed to get stat value, " + componentName + ":" + propName);
               }
               ManagedProperty proxiedProp = createPropertyProxy(prop);
               props.put(prop.getName(), proxiedProp);
            }
            else
            {
               props.put(prop.getName(), prop);
            }
            // Keep the property associated with the runtime MO for invocations/updates
            if (prop.getTargetManagedObject() == null)
               prop.setTargetManagedObject(runtimeMO);
         }
         if (trace)
         {
            log.trace("Properties after:" + props);
         }
      }
      if (runtimeOps != null && runtimeOps.size() > 0)
      {
         if (trace)
         {
            log.trace("Ops before:" + ops);
         }
         runtimeOps = createOperationProxies(runtimeMO, runtimeOps);
         ops.addAll(runtimeOps);
         if (trace)
         {
            log.trace("Ops after:" + ops);
         }
      }
      MutableManagedObject moi = (MutableManagedObject) mo;
      moi.setProperties(props);
      moi.setOperations(ops);
   }

   protected MutableManagedComponent createManagedComponentProxy(Object componentName, MutableManagedComponent component)
   {
      return MutableManagedComponent.class.cast(proxyFactory.createComponentProxy(component, componentName));
   }
   
   /**
    * Create a proxy for managed operations.
    * 
    * @param mo the managed object
    * @param ops the managed operations
    * @return a set of proxied managed operations
    * @throws Exception
    */
   protected Set<ManagedOperation> createOperationProxies(ManagedObject mo, Set<ManagedOperation> ops) throws Exception
   {
      if (proxyFactory == null)
         throw new IllegalArgumentException("Missing RuntimeComponentDispatcher.");

      Object componentName = mo.getComponentName();
      return createOperationProxies(ops, componentName);
   }

   /**
    * Create a proxy for managed operations.
    * 
    * @param ops the managed operations
    * @param componentName the runtime component name 
    * @return a set of proxied managed operations
    * @throws Exception
    */
   protected Set<ManagedOperation> createOperationProxies(Set<ManagedOperation> ops, Object componentName)
         throws Exception
   {
      // Create the delegate operation
      Set<ManagedOperation> operations = new HashSet<ManagedOperation>();
      for (ManagedOperation op : ops)
      {
         ManagedOperation operation = proxyFactory.createOperationProxy(op, componentName);
         operations.add(operation);
      }
      return operations;
   }

   /**
    * Create a proxy for a managed property
    * 
    * @param prop the managed property
    * @return the proxied managed property
    * @throws Exception
    */
   private ManagedProperty createPropertyProxy(ManagedProperty prop) throws Exception
   {
      if (proxyFactory == null)
         throw new IllegalArgumentException("Missing RuntimeComponentDispatcher.");

      // Create the delegate property
      Object componentName = prop.getManagedObject().getComponentName();
      return proxyFactory.createPropertyProxy(prop, componentName);
   }

   /**
    * {@inheritDoc} 
    */
   protected RunState updateRunState(ManagedObject runtimeMO, ManagedComponent comp)
   {
      RunState state = comp.getRunState();
      if (state == RunState.UNKNOWN && dispatcher != null)
      {
         Object name = comp.getComponentName();
         if (name == null && runtimeMO != null)
            name = runtimeMO.getComponentName();
         if (name != null)
         {
            state = getMappedState(name, runStateMapper);
            if (comp instanceof MutableManagedComponent)
            {
               MutableManagedComponent mcomp = MutableManagedComponent.class.cast(comp);
               mcomp.setRunState(state);
            }
         }
      }
      return state;
   }

   /**
    * Get the mapped state.
    * 
    * @param <T> the enum type
    * @param name the runtime component name
    * @param mapper the context mapper
    * @return the state
    */
   protected <T extends Enum<?>> T getMappedState(Object name, ContextStateMapper<T> mapper)
   {
      T state = null;
      if (dispatcher != null)
      {
         try
         {
            state = dispatcher.mapControllerState(name, mapper);
         }
         catch (Exception e)
         {
            state = mapper.getErrorState();
         }
      }
      return state;
   }

}
