/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.view;

import java.util.Collection;

import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.factory.ManagedObjectFactory;
import org.jboss.managed.plugins.jmx.ManagementFactoryUtils;
import org.jboss.profileservice.plugins.management.util.ManagedDeploymentProcessor;
import org.jboss.profileservice.plugins.spi.ProfileView;

/**
 * A ProfileView exposing the Platform MBeans as <code>ManagedComponents</code>.
 * 
 * This view is immutable, it will only get created the first time
 * the view gets loaded. Therefore all containing <code>ManagedObject</code>s
 * and/or <code>ManagedComponent</code>s need to declare dynamic properties,
 * as <code>ViewUse.STATISTIC</code> and <code>runtime=true</code>.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class PlatformMBeanView extends AbstractProfileViewWrapper
{
   
   /** The managed object factory. */
   private static final ManagedObjectFactory managedObjFactory = ManagedObjectFactory.getInstance();

   /** The logger. */
   private static final Logger log = Logger.getLogger(PlatformMBeanView.class);
   
   /** The profile view. */
   private final AbstractProfileView view;

   public PlatformMBeanView()
   {
      this.view = new AbstractProfileView();
   }
   
   @Override
   protected ProfileView getDelegate()
   {
      return view;
   }

   public boolean load()
   {
      if(view.load())
      {
         try
         {
            // JDK PlatformMBeans
            process(ManagementFactoryUtils.getPlatformMBeanMOs(managedObjFactory).values());
            // GarbageCollectorMXBeans
            process(ManagementFactoryUtils.getGarbageCollectorMXBeans(managedObjFactory));
            // MemoryManagerMXBeans
            process(ManagementFactoryUtils.getMemoryManagerMXBeans(managedObjFactory));
            // MemoryPoolMXBeans
            process(ManagementFactoryUtils.getMemoryPoolMXBeans(managedObjFactory));
         }
         catch(Exception e)
         {
            log.debug("failed to create platform MBean management interfaces", e);
         }
         return true;
      }
      return false;
   }
   
   protected void process(Collection<ManagedObject> managedObjects) throws Exception
   {
      if(managedObjects != null && managedObjects.isEmpty() == false)
      {
         ManagedDeploymentProcessor processor = new ManagedDeploymentProcessor(getProxyFactory());
         for(ManagedObject mo : managedObjects)
         {
            processor.processManagedObject(mo, null, view);
         }
      }
   }

   public void removeComponent(ManagedComponent update, ManagedComponent original) throws Exception
   {
      // we cannot remove platform mbeans
   }
   
   public void updateComponent(ManagedComponent update, ManagedComponent original) throws Exception
   {
      // we cannot update a platform mbean
   }
   
   
}

