/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.spi;

import java.util.Collection;
import java.util.Set;

import org.jboss.deployers.spi.management.NameMatcher;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.profileservice.spi.action.ModificationListener;

/**
 * A management view of a single profile and extension point for integrating
 * different views.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public interface ProfileView extends ModificationListener
{

   /**
    * Load the profile view.
    * 
    * @return if there were changes
    */
   boolean load();
   
   /**
    * Get the deployment names.
    * 
    * @return the deployment names
    */
   Collection<String> getDeploymentNames();
   
   /**
    * Get the managed deployment.
    * 
    * @param name the deployment name
    * @return the managed deployment, null if there is no such deployment
    */
   ManagedDeployment getManagedDeployment(String name);
   
   /**
    * Get the names of the deployment that have the given deployment type.
    * 
    * @param type - the deployment type
    * @return the names
    */
   Set<String> getDeploymentNamesForType(String type);

   /**
    * Search for all deployment names with a name matching the regex expression.
    * 
    * @param regex - the regex to query deployment name 
    * @return the deployment name. 
    */
   Set<String> getMatchingDeploymentName(String regex);
   
   /**
    * Search for all deployments matched by the matcher.
    * 
    * @param name - the name to input to the matcher
    * @param matcher - the matcher 
    * @return a possibly empty set of ManagedDeployment accepted by the matcher
    */
   Set<ManagedDeployment> getMatchingDeployments(String name, NameMatcher<ManagedDeployment> matcher);

   /**
    * Get the deployments of a type.
    * 
    * @param type - the deployment or module type.
    * @return the possibly empty set of deployment with the given type.
    * @throws Exception
    */
   Set<ManagedDeployment> getDeploymentsForType(String type);

   /**
    * Get a set of the component types in use in the profiles
    * 
    * @return set of component types in use
    */
   Set<ComponentType> getComponentTypes();

   /**
    * Get the components of a type. 
    * 
    * @param type - the component type.
    * @return the possibly empty set of components with the given type.
    */
   Set<ManagedComponent> getComponentsForType(ComponentType type);

   /**
    * Obtain the ManagedComponent for the given name/type
    * 
    * @see #getComponent(String, ComponentType, NameMatcher)
    * @param name - the ManagedComponent#getName value to match
    * @param type - the component type
    * @return the possibly null ManagedComponent
    */
   ManagedComponent getComponent(String name, ComponentType type);
   
   /**
    * Obtain the ManagedComponents for the given name/type and matcher.
    * 
    * @param name - the ManagedComponent#getName value to match
    * @param type - the component type
    * @param matcher - a matcher that compares name to candidate ManagedComponent
    * @return the possibly empty set of ManagedComponent matches.
    */
   Set<ManagedComponent> getMatchingComponents(String name, ComponentType type,
         NameMatcher<ManagedComponent> matcher);
   
}

