/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.deploy.actions;

import java.io.InputStream;
import java.io.SyncFailedException;
import java.util.Arrays;

import org.jboss.deployers.spi.management.deploy.DeploymentID;
import org.jboss.profileservice.management.event.ProfileModificationEvent;
import org.jboss.profileservice.profile.metadata.BasicProfileDeploymentMetaData;
import org.jboss.profileservice.repository.artifact.CopyStreamArtifactTransformer;
import org.jboss.profileservice.repository.artifact.ExplodedArtifactTransformer;
import org.jboss.profileservice.repository.artifact.file.FileArtifactId;
import org.jboss.profileservice.spi.DeploymentOption;
import org.jboss.profileservice.spi.MutableProfile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.VirtualDeploymentRepository;
import org.jboss.profileservice.spi.VirtualDeploymentRepositoryExt;
import org.jboss.profileservice.spi.action.ProfileModificationResponse;
import org.jboss.profileservice.spi.action.ProfileModificationType;
import org.jboss.profileservice.spi.action.deployment.DeploymentAction;
import org.jboss.profileservice.spi.action.deployment.DeploymentActionContext;
import org.jboss.profileservice.spi.deployment.ProfileDeploymentFlag;
import org.jboss.profileservice.spi.repository.artifact.ArtifactRepository;
import org.jboss.profileservice.spi.repository.artifact.ArtifactTransformer;
import org.jboss.profileservice.spi.repository.artifact.MutableArtifactRepository;


/**
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class DeploymentDistributeAction extends AbstractDeploymentAddAction implements DeploymentAction<DeploymentActionContext>
{

   private static final FileArtifactId noopArtifactId = FileArtifactId.EMPTY_ARTIFACT;

   private final InputStream contentIS;
   
   public DeploymentDistributeAction(MutableProfile profile, DeploymentID dtId, InputStream is, DeploymentActionContext modificationContext)
   {
      super(dtId, profile, modificationContext);
      this.contentIS = is;
   }

   protected boolean doPrepare(ProfileModificationResponse response)
   {
      try
      {
         String deploymentName = getDeploymentName();
         // in case there is already a deployment, lock the contents.
         // TODO this might should not be done here ?
         if(getProfile().hasDeployment(deploymentName))
         {
            ProfileDeployment deployment = getProfile().getDeployment(deploymentName);
            deployment.getDeploymentInfo().setFlag(ProfileDeploymentFlag.LOCKED);
         }
         //
         VirtualDeploymentRepository deploymentRepository = resolveDeploymentRepository(getProfile());
         if(deploymentRepository == null)
         {
            response.setFailure(new IllegalStateException("failed to resolve deployment repository"));
            return false;
         }
         MutableArtifactRepository<FileArtifactId> artifactRepository = resolveRepository(deploymentRepository);
         // Create the artifact based on the deployment name
         String artifactName = deploymentName;
         FileArtifactId artifactId = new FileArtifactId(artifactName);
         
         // Check if the artifact is already there
         if(isFailIfExists() && artifactRepository.containsArtifact(artifactId))
         {
            response.setFailure(new SyncFailedException("Deployment content already exists: "+ artifactName));
            return false;
         }
         
         // Check if we need unpack this deployment
         ArtifactTransformer<FileArtifactId> transformer = null;
         if(isExplode())
         {
            transformer = ExplodedArtifactTransformer.getInstance();
         }
         else
         {
            transformer = CopyStreamArtifactTransformer.getInstance();
         }
         // Add the content
         artifactRepository.addArtifact(artifactId, contentIS, transformer);
         
         // Create a new deployment
         ProfileDeployment deployment = deploymentRepository.createDeployment(new BasicProfileDeploymentMetaData(artifactName));
         // Mark it as locked, so we can add contents after
         deployment.getDeploymentInfo().setFlag(ProfileDeploymentFlag.LOCKED);
         // Add the deployment
         getProfile().addDeployment(deployment);
         
         // Notify
         response.fireModificationEvent(ProfileModificationEvent.create(ProfileModificationType.ADD, getProfile().getKey()));
         
      }
      catch(Exception e)
      {
         response.setFailure(e);
         return false;
      }
      return true;
   }

   protected ArtifactTransformer<FileArtifactId> createTransformer()
   {
      if(isExplode())
      {
         return ExplodedArtifactTransformer.getInstance();
      }
      else
      {
         return CopyStreamArtifactTransformer.getInstance();
      }      
   }

   protected void doCancel()
   {
      // FIXME doCancel
      
   }
   protected void doCommit(ProfileModificationResponse response)
   {
      // FIXME doCommit
      
   }
   protected void doComplete(ProfileModificationResponse response) throws Exception
   {
      // FIXME doComplete
      
   }
   protected void doRollbackFromCancelled()
   {
      // FIXME doRollbackFromCancelled
      
   }
   protected void doRollbackFromComplete()
   {
      // FIXME doRollbackFromComplete
      
   }
   protected void doRollbackFromActive()
   {
      // FIXME doRollbackFromOpen
      
   }
   protected void doRollbackFromPrepared()
   {
      // FIXME doRollbackFromPrepared
      
   }
   protected void doRollbackFromRollbackOnly()
   {
      // FIXME doRollbackFromRollbackOnly
      
   }
   
   protected boolean isExplode()
   {
      return hasDeploymentOption(DeploymentOption.Explode);
   }
   
   protected boolean isFailIfExists()
   {
      return hasDeploymentOption(DeploymentOption.FailIfExists);
   }
   
   protected boolean hasDeploymentOption(DeploymentOption option)
   {
      if(getDeploymentID().getDeploymentOptions() != null)
      {
         return Arrays.binarySearch(getDeploymentID().getDeploymentOptions(), option) != -1;
      }
      return false;      
   }
   
   /**
    * Cast to VirtualDeploymentRepositoryExt
    * 
    * @param profile the mutable profile
    * @return the virtual deployment repository
    */
   static VirtualDeploymentRepository resolveDeploymentRepository(MutableProfile profile)
   {
      if(profile instanceof VirtualDeploymentRepositoryExt == false)
      {
         throw new IllegalStateException("Failed to get the deployment repository for profile " + profile.getKey());
      }
      return VirtualDeploymentRepositoryExt.class.cast(profile).getDeploymentRepository();
   }
   
   /**
    * Resolve and cast to mutable artifact repository
    * 
    * @param deploymentRepository the virtual deployment repository
    * @return the mutable artifact repository
    */
   static MutableArtifactRepository<FileArtifactId> resolveRepository(VirtualDeploymentRepository deploymentRepository)
   {
      ArtifactRepository<FileArtifactId> repo = deploymentRepository.resolveArtifactRepository(noopArtifactId);
      if(repo instanceof MutableArtifactRepository<?> == false)
      {
         throw new IllegalStateException("no mutable artifact repository");
      }
      return MutableArtifactRepository.class.cast(repo);
   }
   
}

