/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.jboss.deployers.spi.management.DeploymentTemplate;
import org.jboss.deployers.spi.management.ManagementView;
import org.jboss.deployers.spi.management.NameMatcher;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.DeploymentTemplateInfo;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.profileservice.plugins.management.util.AbstractManagementProxyFactory;
import org.jboss.profileservice.plugins.spi.ProfileView;
import org.jboss.profileservice.plugins.spi.ProfileViewWrapper;
import org.jboss.profileservice.spi.NoSuchDeploymentException;

/**
 * The aggregating management view. This represents the local view of
 * the server. The view on the domain will most probably aggregate
 * the <code>ManagedObject</code>s defined in the <code>DomainMetaData</code>
 * and expose them over a <code>DomainView</code>
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class AggregatingLocalManagementView extends AbstractTemplateCreator
   implements ManagementView
{

   /** The logger. */
   private static final Logger log = Logger.getLogger(AggregatingLocalManagementView.class);
   
   /** The bundle name. */
   private static final String BUNDLE_NAME = "org.jboss.profileservice.plugins.management.messages";
   
   /** The internationalization resource bundle. */
   private ResourceBundle i18n;
   /** the Locale for the i18n messages. */
   private Locale currentLocale;
   /** The formatter used for i18n messages. */
   private MessageFormat formatter = new MessageFormat("");
   
   /** The profile views. */
   private List<ProfileViewWrapper> views = new CopyOnWriteArrayList<ProfileViewWrapper>();
  
   /** The deployment templates that have been registered with the MV. */
   private Map<String, DeploymentTemplate> templates = new HashMap<String, DeploymentTemplate>();
   
   /** The proxy factory. */
   private AbstractManagementProxyFactory proxyFactory;
   
   private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
   
   public AggregatingLocalManagementView()
   {
      currentLocale = Locale.getDefault();
      formatter.setLocale(currentLocale);
      i18n = ResourceBundle.getBundle(BUNDLE_NAME, currentLocale);
   }
   
   public AbstractManagementProxyFactory getProxyFactory()
   {
      return proxyFactory;
   }
   
   public void setProxyFactory(AbstractManagementProxyFactory proxyFactory)
   {
      if(proxyFactory == null)
      {
         throw new IllegalArgumentException("null proxy factory");
      }
      this.proxyFactory = proxyFactory;
   }

   public void start()
   {
      if(proxyFactory == null)
      {
         throw new IllegalStateException("null proxy factory");
      }
   }
   
   /**
    * {@inheritDoc}
    */
   public boolean load()
   {
      lockWrite();
      try
      {
         // Clear any thread interrupt
         boolean wasInterrupted = Thread.interrupted();
         if(wasInterrupted)
         {
            log.debug("Cleared interrupted state of calling thread");
         }
         boolean changed = false;
         for(final ProfileView view : views)
         {
            if(view.load())
               changed = true;
         }
         // 
         if(wasInterrupted)
         {
            Thread.currentThread().interrupt();
            log.debug("Restored interrupted state of calling thread");
         }
         return changed;
      }
      finally
      {
         unlockWrite();
      }
   }
   
   /**
    * Add a profile view wrapper.
    * 
    * @param view the profile view wrapper.
    */
   public void addProfileView(ProfileViewWrapper view)
   {
      if(view == null)
      {
         throw new IllegalArgumentException("null profile view");
      }
      // FIXME
      view.setManagementProxyFactory(proxyFactory);
      this.views.add(view);
   }
   
   /**
    * Remove profile view wrapper.
    * 
    * @param view the profile view wrapper
    */
   public void removeProfileView(ProfileViewWrapper view)
   {
      if(view == null)
      {
         throw new IllegalArgumentException("null profile view");
      }
      this.views.remove(view);
   }
   
   /**
    * Add a deployment template.
    * 
    * @param template the deployment template
    */
   public void addTemplate(DeploymentTemplate template)
   {
      if(template == null)
      {
         throw new IllegalArgumentException("null deployment template");
      }
      if(template.getInfo() == null)
      {
         throw new IllegalArgumentException("null deployment template info");
      }
      if(template.getInfo().getName() == null)
      {
         throw new IllegalArgumentException("null deployment template info name");
      }
      this.templates.put(template.getInfo().getName(), template);
   }
   
   /**
    * Remove a deployment template.
    * 
    * @param template the deployment template
    */
   public void removeTemplate(DeploymentTemplate template)
   {
      if(template == null)
      {
         throw new IllegalArgumentException("null deployment template");
      }
      if(template.getInfo() == null)
      {
         throw new IllegalArgumentException("null deployment template info");
      }
      if(template.getInfo().getName() == null)
      {
         throw new IllegalArgumentException("null deployment template info name");
      }
      this.templates.remove(template.getInfo().getName());
   }

   /**
    * {@inheritDoc}
    */
   public void applyTemplate(String deploymentBaseName, DeploymentTemplateInfo info) throws Exception
   {
      if(deploymentBaseName == null)
      {
         throw new IllegalArgumentException("Null deployment base name.");
      }
      if(info == null)
      {
         throw new IllegalArgumentException("Null template info.");
      }

      final DeploymentTemplate template = templates.get(info.getName());
      if( template == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchTemplate"));
         Object[] args = {info.getName()};
         String msg = formatter.format(args);
         throw new IllegalStateException(msg);
      }
      
      // Create a deployment base from the template
      if( log.isTraceEnabled() )
         log.trace("applyTemplate, deploymentBaseName="+deploymentBaseName +", info="+info);

      // Create, distribute and start a deployment template
      super.applyTemplate(template, deploymentBaseName, info);
      // TODO reload view
   }

   /**
    * {@inheritDoc}
    */
   public ManagedComponent getComponent(String name, ComponentType type) throws Exception
   {
      if(name == null)
      {
         throw new IllegalArgumentException("null component name");
      }
      lockRead();
      try
      {
         ManagedComponent component = null;
         for(final ProfileView view : views)
         {
            ManagedComponent resolved = view.getComponent(name, type); 
            if(resolved != null)
            {
               component = resolved;
            }
            // TODO fail on multiple results
         }
         if(component == null)
         {
            // TODO throw localized exception
            // TODO throw new IllegalStateException("failed to find component for name " + name);
         }
         return component;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<ComponentType> getComponentTypes()
   {
      lockRead();
      try
      {
         Set<ComponentType> types = new HashSet<ComponentType>();
         for(final ProfileView view : views)
         {
            types.addAll(view.getComponentTypes());
         }
         return types;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getComponentsForType(ComponentType type)
   {
      if(type == null)
      {
         throw new IllegalArgumentException("null component type");
      }
      lockRead();
      try
      {
         Set<ManagedComponent> components = new HashSet<ManagedComponent>();
         for(final ProfileView view : views)
         {
            components.addAll(view.getComponentsForType(type));
         }
         return components;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public ManagedDeployment getDeployment(String name) throws NoSuchDeploymentException
   {
      if(name == null)
      {
         throw new IllegalArgumentException("null deployment name");
      }
      lockRead();
      try
      {
         ManagedDeployment deployment = null;
         for(final ProfileView view : views)
         {
            ManagedDeployment resolved = view.getManagedDeployment(name); 
            if(resolved != null)
            {
               deployment = resolved;
            }
            // TODO fail on multiple results
         }
         if(deployment == null)
         {
            // TODO localized message
            throw new NoSuchDeploymentException("Managed deployment: " + name + " not found.");
         }
         return deployment;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getDeploymentNames()
   {
      lockRead();
      try
      {
         final Set<String> deploymentNames = new TreeSet<String>();
         for(final ProfileView view : views)
         {
            deploymentNames.addAll(view.getDeploymentNames());
         }
         return deploymentNames;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getDeploymentNamesForType(String type)
   {
      if(type == null)
      {
         throw new IllegalArgumentException("null deployment type");
      }
      lockRead();
      try
      {
         Set<String> deployments = new TreeSet<String>();
         for(final ProfileView view: views)
         {
            deployments.addAll(view.getDeploymentNamesForType(type));
         }
         return deployments;
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getDeploymentsForType(String type)
   {
      if(type == null)
      {
         throw new IllegalArgumentException("null deployment type");
      }
      lockRead();
      try
      {
         Set<ManagedDeployment> deployments = new HashSet<ManagedDeployment>();
         for(final ProfileView view : views)
         {
            deployments.addAll(view.getDeploymentsForType(type));
         }
         return deployments;         
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getMatchingComponents(String name, ComponentType type,
         NameMatcher<ManagedComponent> matcher)
   {
      if(name == null)
      {
         throw new IllegalArgumentException("null component name");
      }
      if(type == null)
      {
         throw new IllegalArgumentException("null component type");
      }
      if(matcher == null)
      {
         throw new IllegalArgumentException("null component name matcher");
      }
      lockRead();
      try
      {
         Set<ManagedComponent> components = new HashSet<ManagedComponent>();
         for(final ProfileView view : views)
         {
            components.addAll(view.getMatchingComponents(name, type, matcher));
         }
         return components;         
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getMatchingDeploymentName(String regex) throws NoSuchDeploymentException
   {
      if(regex == null)
      {
         throw new IllegalArgumentException("null regex");
      }
      lockRead();
      try
      {
         Set<String> deployments = new TreeSet<String>();
         for(final ProfileView view : views)
         {
            deployments.addAll(view.getMatchingDeploymentName(regex));
         }
         return deployments;         
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getMatchingDeployments(String name, NameMatcher<ManagedDeployment> matcher)
   {
      if(name == null)
      {
         throw new IllegalArgumentException("null deployment name");
      }
      if(matcher == null)
      {
         throw new IllegalArgumentException("null deployment matcher");
      }
      lockRead();
      try
      {
         Set<ManagedDeployment> deployments = new HashSet<ManagedDeployment>();
         for(final ProfileView view : views)
         {
            deployments.addAll(view.getMatchingDeployments(name, matcher));
         }
         return deployments;         
      }
      finally
      {
         unlockRead();
      }
   }

   /**
    * {@inheritDoc}
    */
   public DeploymentTemplateInfo getTemplate(String name) throws NoSuchDeploymentException
   {
      if(name == null)
      {
         throw new IllegalArgumentException("null template name");
      }
      final DeploymentTemplate template = this.templates.get(name);
      if( template == null )
      {
         formatter.applyPattern(i18n.getString("ManagementView.NoSuchTemplate")); //$NON-NLS-1$
         Object[] args = {name};
         String msg = formatter.format(args);
         throw new NoSuchDeploymentException(msg);
      }
      // Make sure to return a copy to avoid call by reference uses modifying the template values
      final DeploymentTemplateInfo info = template.getInfo();
      return info.copy();
   }

   /**
    * {@inheritDoc}
    */
   public Set<String> getTemplateNames()
   {
      return new TreeSet<String>(this.templates.keySet());
   }

   /**
    * {@inheritDoc}
    */
   public void process() throws Exception
   {
      // nothing ...   
   }

   /**
    * {@inheritDoc}
    */
   public void reload()
   {
      load();
   }

   /**
    * {@inheritDoc}
    */
   public void removeComponent(ManagedComponent comp) throws Exception
   {
      if(comp == null)
      {
         throw new IllegalArgumentException("null managed component");
      }
      ManagedComponent serverComponent = getComponent(comp.getName(), comp.getType());
      if(serverComponent == null)
      {
         formatter.applyPattern(i18n.getString("ManagementView.InvalidComponentName")); //$NON-NLS-1$
         Object[] args = {comp.getName(), comp.getType()};
         String msg = formatter.format(args);
         throw new IllegalArgumentException(msg);
      }
      lockWrite();
      try
      {
         ProfileViewWrapper target = resolveWrapper(serverComponent.getDeployment().getName());
         if(target == null)
         {
            throw new IllegalStateException("failed to resolve target profile for component " + serverComponent);
         }
         // Delegate request to handling wrapper.
         target.removeComponent(comp, serverComponent);         
      }
      finally
      {
         unlockWrite();
      }
   }

   /**
    * {@inheritDoc}
    */
   public void updateComponent(ManagedComponent comp) throws Exception
   {
      if(comp == null)
      {
         throw new IllegalArgumentException("null managed component");
      }
      ManagedComponent serverComponent = getComponent(comp.getName(), comp.getType());
      if(serverComponent == null)
      {
         formatter.applyPattern(i18n.getString("ManagementView.InvalidComponentName")); //$NON-NLS-1$
         Object[] args = {comp.getName(), comp.getType()};
         String msg = formatter.format(args);
         throw new IllegalArgumentException(msg);
      }
      lockWrite();
      try
      {
         ProfileViewWrapper target = resolveWrapper(serverComponent.getDeployment().getName());
         if(target == null)
         {
            throw new IllegalStateException("failed to resolve target profile for component " + serverComponent);
         }
         // Delegate request to handling wrapper.
         target.updateComponent(comp, serverComponent);         
      }
      finally
      {
         unlockWrite();
      }
   }

   /**
    * Try to resolve the target profile.
    * 
    * @param deploymentName the deployment name
    * @return the profile
    */
   protected ProfileViewWrapper resolveWrapper(String deploymentName)
   {
      for(ProfileViewWrapper view : views)
      {
         if(view.getDeploymentNames().contains(deploymentName))
         {
            return view;
         }
      }
      return null;
   }

   void lockRead()
   {
      lock.readLock().lock();
   }
   void unlockRead()
   {
      lock.readLock().unlock();
   }
   
   void lockWrite()
   {
      lock.writeLock().lock();
   }
   void unlockWrite()
   {
      lock.writeLock().unlock();
   }
   
}

