/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.actions;

import java.util.Collection;
import java.util.HashSet;

import org.jboss.deployers.spi.management.KnownDeploymentTypes;
import org.jboss.managed.api.DeploymentState;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.plugins.ManagedDeploymentImpl;
import org.jboss.profileservice.plugins.management.util.AbstractManagementProxyFactory;
import org.jboss.profileservice.plugins.management.util.ManagedDeploymentProcessor;
import org.jboss.profileservice.plugins.spi.ProfileViewProcessingContext;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileDeployment;
import org.jboss.profileservice.spi.action.ProfileModificationAction;
import org.jboss.profileservice.spi.action.ProfileModificationContext;
import org.jboss.profileservice.spi.action.ProfileModificationResponse;
import org.jboss.profileservice.spi.managed.ManagedProfile;

/**
 * The <code>ProfileView</code> update action.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class ProfileViewUpdateAction implements ProfileModificationAction<ProfileModificationContext>
{
   
   /** The deployment types. */
   static final String DEPLOYMENT_TYPES = KnownDeploymentTypes.class.getName();
   
   /** is complete. */
   private boolean complete = false;
   
   /** The managed profile. */
   private final ManagedProfile managedProfile;

   /** The proxy factory. */
   private final AbstractManagementProxyFactory proxyFactory;
   
   /** The context. */
   private final ProfileViewProcessingContext context;

   public ProfileViewUpdateAction(ManagedProfile managedProfile, ProfileViewProcessingContext context,
         AbstractManagementProxyFactory proxyFactory)
   {
      this.managedProfile = managedProfile;
      this.context = context;
      this.proxyFactory = proxyFactory;
   }

   // -----
   
   @Override
   public ProfileModificationContext getContext()
   {
      return null;
   }
   
   public void complete(ProfileModificationResponse response)
   {
      ManagedDeploymentProcessor processor = new ManagedDeploymentProcessor(proxyFactory);
      Profile profile = managedProfile.getProfile();
      for(ProfileDeployment deployment : profile.getDeployments())
      {
         try
         {
            if(deployment.getDeploymentInfo().isDeployed())
            {
               // Deployed
               ManagedDeployment md = managedProfile.getManagedDeployment(deployment);
               processor.processRootManagedDeployment(md, context);
               if(md.getTypes() != null && md.getTypes().isEmpty() == false)
               {
                  if(deployment.getTransientAttachments().hasAttachment(DEPLOYMENT_TYPES) == false)
                  {
                     deployment.getTransientAttachments().putAttachment(DEPLOYMENT_TYPES, md.getTypes());
                  }
               }
            }
            else
            {
               // Not deployed
               ManagedDeployment md = new ManagedDeploymentImpl(deployment.getName(), deployment.getSimpleName());
               updateStoppedDeploymentTypes(deployment, md);
               processor.processManagedDeployment(md, DeploymentState.STOPPED, 0, context);
            }            
         }
         catch(Exception e)
         {
            response.setFailure(e);
         }
      }
      this.complete = true;
   }

   @Override
   public void cancel()
   {
      // 
   }

   @Override
   public boolean isCancelled()
   {
      return false;
   }

   @Override
   public boolean isComplete()
   {
      return complete;
   }

   @SuppressWarnings("unchecked")
   protected void updateStoppedDeploymentTypes(ProfileDeployment deployment, ManagedDeployment md)
   {
      String deploymentName = deployment.getName();
      // Try to get the cached deployment type
      Collection<String> deploymentTypes = deployment
            .getTransientAttachments().getAttachment(DEPLOYMENT_TYPES, Collection.class);

      if(deploymentTypes != null)
      {
         md.setTypes(new HashSet<String>(deploymentTypes));
      }
      else
      {
         int i = deploymentName.lastIndexOf(".");
         if(i != -1 && (i + 1) < deploymentName.length())
         {
            String guessedType = deploymentName.substring(i + 1, deploymentName.length());
            if(guessedType.endsWith("/"))
               guessedType = guessedType.substring(0, guessedType.length() -1 );
            md.setTypes(new HashSet<String>(1));
            md.addType(guessedType);
         }
      }
   }
   
   
}

