/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.util;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.Arrays;

import org.jboss.deployers.spi.management.ManagedComponentRuntimeDispatcher;
import org.jboss.managed.api.ManagedOperation;
import org.jboss.managed.api.ManagedParameter;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.MetaValueFactory;
import org.jboss.metatype.plugins.values.MetaValueFactoryBuilder;
import org.jboss.metatype.spi.values.MetaMapper;

/**
 * The abstract managed component runtime dispatcher.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public abstract class AbstractManagedComponentRuntimeDispatcher implements ManagedComponentRuntimeDispatcher
{

   /** The meta value factory. */
   private static final MetaValueFactory valueFactory = MetaValueFactoryBuilder.create();
   
   /**
    * Getter property / attribute
    *
    * @param name entry name
    * @param getter property / attribute name
    * @return target's property / attribute instance
    * @throws Throwable for any error
    */
   protected abstract Object get(Object name, String getter) throws Throwable;

   /**
    * Setter property / attribute
    *
    * @param name entry name
    * @param setter property / attribute name
    * @param value set target's property / attribute instance
    * @throws Throwable for any error
    */
   protected  abstract void set(Object name, String setter, Object value) throws Throwable;

   /**
    * Invoke method / operation
    *
    * @param name entry name
    * @param methodName method name
    * @param parameters parameter values
    * @param signature method's parameter types / signatures
    * @return inovocation's return object
    * @throws Throwable for any error
    */
   protected abstract Object invoke(Object name, String methodName, Object[] parameters, String[] signature) throws Throwable;
   
   /**
    * {@inheritDoc}
    */
   public MetaValue get(Object componentName, ManagedProperty property)
   {
      Object value = null;
      try
      {
         value = get(componentName, property.getName());
      }
      catch(Throwable t)
      {
         throw new UndeclaredThrowableException(t, "Failed to get property '" + property.getName() + "' on component '" + componentName + "'.");
      }
      MetaMapper mapper = property.getTransientAttachment(MetaMapper.class);
      return create(value, property.getMetaType(), mapper);
   }

   /**
    * {@inheritDoc}
    */
   public void set(Object componentName, ManagedProperty property, MetaValue metaValue)
   {
      MetaMapper mapper = property.getTransientAttachment(MetaMapper.class);
      Object value = unwrap(metaValue, mapper);
      try
      {
         set(componentName, property.getName(), value);
      }
      catch(Throwable t)
      {
         throw new UndeclaredThrowableException(t, "Failed to set property '" + property.getName() + "' on component '" + componentName + "' to value [" + value + "].");
      }
   }
   
   /**
    * {@inheritDoc}
    */
   public MetaValue invoke(Object componentName, ManagedOperation operation, MetaValue... param)
   {
      String[] sig = new String[param.length];
      Object[] args = new Object[param.length];
      ManagedParameter[] params = operation.getParameters();
      for(int i=0; i < param.length; i++)
      {
         ManagedParameter mp = params[i];
         MetaMapper<?> mapper = mp.getTransientAttachment(MetaMapper.class);
         //
         args[i] = unwrap(param[i], mapper);
         sig[i] = mp.getMetaType().getTypeName();
      }
      try
      {
         // Invoke
         Object value = invoke(componentName, operation.getName(), args, sig);
         MetaValue mvalue = null;
         if (value != null)
         {
            MetaType type = operation.getReturnType();
            // Look for a return type MetaMapper
            MetaMapper returnTypeMapper = operation.getTransientAttachment(MetaMapper.class);
            mvalue = create(value, type, returnTypeMapper);
         }
         return mvalue;
      }
      catch(Throwable t)
      {
         throw new UndeclaredThrowableException(t, "Failed to invoke method '" + operation.getName() + "' on component '" + componentName + "' with parameters " + Arrays.asList(param) + ".");
      }
   }

   /**
    * Create a meta value.
    * 
    * @param <T> 
    * @param value the value
    * @param metaType the meta type
    * @param mapper the meta mapper
    * @return the meta value
    */
   protected <T> MetaValue create(T value, MetaType metaType, MetaMapper<T> mapper)
   {
      if(mapper != null)
      {
         return mapper.createMetaValue(metaType, value);
      }
      else
      {
         return create(value);
      }
   }
   
   /**
    * Create meta value.
    *
    * @param value the value
    * @return meta value instance
    */
   protected MetaValue create(Object value)
   {
      return valueFactory.create(value);
   }

   /**
    * Unwrap meta value.
    * 
    * @param <T>
    * @param metaValue the meta value
    * @param metaMapper the meta mapper
    * @return unwrapped value
    */
   protected <T> Object unwrap(MetaValue metaValue, MetaMapper<T> metaMapper)
   {
      if(metaMapper != null)
      {
         return metaMapper.unwrapMetaValue(metaValue);
      }
      else
      {
         return unwrap(metaValue);
      }
   }
   
   /**
    * Unwrap meta value.
    *
    * @param metaValue the meta value
    * @return unwrapped value
    */
   protected Object unwrap(MetaValue metaValue)
   {
      return valueFactory.unwrap(metaValue);
   }
   
}

