/*
* JBoss, Home of Professional Open Source
* Copyright 2010, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.util;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.deployers.spi.management.ContextStateMapper;
import org.jboss.logging.Logger;
import org.jboss.managed.api.DeploymentState;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.plugins.ManagedDeploymentImpl;
import org.jboss.profileservice.plugins.spi.ProfileViewProcessingContext;

/**
 * The profile view processor helper.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class ManagedDeploymentProcessor extends ManagedObjectRuntimeProcessor
{

   /** The logger. */
   private static final Logger log = Logger.getLogger(ManagedDeploymentProcessor.class);
   
   /** The deployment states. */
   private static final ContextStateMapper<DeploymentState> deploymentStateMapper;
   
   static
   {
      Map<String, DeploymentState> deploymentMappings = new HashMap<String, DeploymentState>();
      deploymentMappings.put("**ERROR**", DeploymentState.FAILED);
      deploymentMappings.put("Not Installed", DeploymentState.STOPPED);
      deploymentMappings.put("Installed", DeploymentState.STARTED);

      deploymentStateMapper = new DefaultContextStateMapper<DeploymentState>(deploymentMappings,
            DeploymentState.STARTING, DeploymentState.STOPPING, DeploymentState.FAILED, DeploymentState.UNKNOWN);
   }
   
   public ManagedDeploymentProcessor(AbstractManagementProxyFactory proxyFactory)
   {
      super(proxyFactory);
   }
   
   /**
    * Process a root managed deployment.
    * 
    * @param md the managed deployment
    * @param context the profile view context
    * @throws Exception
    */
   public void processRootManagedDeployment(ManagedDeployment md, ProfileViewProcessingContext context) throws Exception
   {
      DeploymentState state = getDeploymentState(md);
      processManagedDeployment(md, state, 0, context);      
   }

   /**
    * Process a root managed deployment.
    * 
    * @param md the managed deployment
    * @param state the deployment state
    * @param context the profile view context
    * @throws Exception
    */
   public void processRootManagedDeployment(ManagedDeployment md, DeploymentState state, ProfileViewProcessingContext context) throws Exception
   {
      processManagedDeployment(md, state, 0, context);
   }
   
   /**
    * Process a managed deployment.
    * 
    * @param md the managed deployment
    * @param state the deployment state
    * @param level the level
    * @param context the profile view context
    * @throws Exception
    */
   public void processManagedDeployment(ManagedDeployment md, DeploymentState state, int level, ProfileViewProcessingContext context) throws Exception
   {
      boolean trace = log.isTraceEnabled();
      
      String name = md.getName();
      if (trace)
      {
         log.trace(name + " ManagedDeployment_" + level + ": " + md);
      }

      // Set the deployment state
      if(state != null && md instanceof ManagedDeploymentImpl)
         ManagedDeploymentImpl.class.cast(md).setDeploymentState(state);

      // Map any existing ManagedComponent types
      if(md != null && md.getComponents() != null)
      {
         for(ManagedComponent comp : md.getComponents().values())
         {
            context.addManagedComponent(comp);
         }
      }
      // Process managed objects
      Map<String, ManagedObject> mos = md.getManagedObjects();
      for(ManagedObject mo : mos.values())
      {
         processManagedObject(mo, md, context);
      }
      // Add the deployment
      context.addManagedDeployment(md);
      
      // Process children
      List<ManagedDeployment> mdChildren = md.getChildren();
      if(mdChildren != null && mdChildren.isEmpty() == false)
      {
         for(ManagedDeployment mdChild : mdChildren)
         {
            // process the child deployments, with the state of the parent.
            processManagedDeployment(mdChild, state, level + 1, context);
         }
      }
   }
   
   /**
    * Get the deployment state.
    * 
    * @param md the managed deployment
    * @return the deployment state
    */
   protected DeploymentState getDeploymentState(ManagedDeployment md)
   {
      DeploymentState state = md.getDeploymentState();
      if (state == DeploymentState.UNKNOWN)
      {
         Object name = md.getName();
         if (name != null)
         {
            state = getMappedState(name, deploymentStateMapper);
         }
      }
      return state;
   }
   
}
