/*
* JBoss, Home of Professional Open Source
* Copyright 2009, Red Hat Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.util;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.logging.Logger;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;
import org.jboss.managed.api.MutableManagedComponent;
import org.jboss.managed.api.RunState;
import org.jboss.managed.api.annotation.ManagementComponent;
import org.jboss.managed.api.annotation.ManagementObjectID;
import org.jboss.managed.api.annotation.ManagementObjectRef;
import org.jboss.managed.plugins.ManagedComponentImpl;
import org.jboss.managed.plugins.factory.AbstractManagedObjectFactory;
import org.jboss.metatype.api.types.ArrayMetaType;
import org.jboss.metatype.api.types.CollectionMetaType;
import org.jboss.metatype.api.types.MetaType;
import org.jboss.metatype.api.values.ArrayValue;
import org.jboss.metatype.api.values.CollectionValue;
import org.jboss.metatype.api.values.GenericValue;
import org.jboss.metatype.api.values.MetaValue;
import org.jboss.metatype.api.values.SimpleValue;
import org.jboss.profileservice.plugins.spi.ProfileViewProcessingContext;

/**
 * A helper to process managed deployments.
 * 
 * TODO we need separate the registry out of the processor. 
 * 
 * TODO (long term) we need to change how ManagedDeployments/ManagedObjects/ManagedComponents
 * are created. This is a temporary solution to make things work.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public abstract class ManagedObjectProcessor
{
   
   /** The logger. */
   private static final Logger log = Logger.getLogger(ManagedObjectProcessor.class);
   
   /** id/type key to ManagedObject map */
   private Map<String, ManagedObject> moRegistry = new HashMap<String, ManagedObject>();
   
   /** The ManagedPropertys with unresolved ManagementObjectRefs */
   private Map<String, Set<ManagedProperty>> unresolvedRefs = new HashMap<String, Set<ManagedProperty>>();

   // runtime functionality 

   /**
    * Update the RunState of a managed component
    * 
    * @param runtimeMO the runtime ManagedObject
    * @param comp the managed component
    * @return the run state
    */
   protected abstract RunState updateRunState(ManagedObject runtimeMO, ManagedComponent comp);
   
   /**
    * Create a managed component proxy.
    * 
    * @param componentName the component name
    * @param component the component
    * @return the component proxy
    */
   protected abstract MutableManagedComponent createManagedComponentProxy(Object componentName, MutableManagedComponent component);
   
   /**
    * Process a managed object
    * 
    * @param mo the managed object
    * @param md the managed deployment
    * @param context the profile view
    * @throws Exception
    */
   public void processManagedObject(ManagedObject mo, ManagedDeployment md, ProfileViewProcessingContext context)
      throws Exception
   {
      boolean trace = log.isTraceEnabled();
      String key = mo.getName() + "/" + mo.getNameType();
      if(trace)
      {
         log.trace("ID for ManagedObject: "+key+", attachmentName: "+mo.getAttachmentName());
      }
      
      // Create ManagedComponents for ManagedObjects annotated with ManagementComponent
      ManagementComponent mc = getAnnotation(mo, ManagementComponent.class);
      if (mc != null)
      {
         ComponentType type = new ComponentType(mc.type(), mc.subtype());
         MutableManagedComponent comp = new ManagedComponentImpl(type, md, mo);
         if(comp.getComponentName() != null)
         {
            // create proxy
            comp = createManagedComponentProxy(comp.getComponentName(), comp);
         }
         if(trace)
         {
            log.trace("Processing ManagementComponent("+mo.getName()+"): "+comp);
         }
         if(md != null && md.getComponent(mo.getName()) == null)
         {
            md.addComponent(mo.getName(), comp);
         }
         context.addManagedComponent(comp);
         updateRunState(null, comp);
      }
      
      processManagedObject(key, mo, md, context);
   }

   /**
    * Get a registered managed object
    * 
    * @param key the managed object key
    * @return the managed object
    */
   protected ManagedObject getRegisteredManagedObject(String key)
   {
      return this.moRegistry.get(key);
   }
   
   /**
    * Process managed object.
    *
    * @param mo the managed object
    * @param md the managed deployment
    */
   protected void processManagedObject(String key, ManagedObject mo, ManagedDeployment md, ProfileViewProcessingContext context)
      throws Exception
   {
      boolean trace = log.isTraceEnabled(); 

      // Check for unresolved refs
      checkForReferences(key, mo);

      // Scan for @ManagementObjectRef
      // Process nested ManagedObjects
      for(ManagedProperty prop : mo.getProperties().values())
      {
         // See if this is a ManagementObjectID
         Map<String, Annotation> pannotations = prop.getAnnotations();
         if (pannotations != null && pannotations.isEmpty() == false)
         {
            ManagementObjectID id = (ManagementObjectID) pannotations.get(ManagementObjectID.class.getName());
            if (id != null)
            {
               Object refName = getRefName(prop.getValue());
               if (refName == null)
               {
                  refName = id.name();
               }
               String propKey = refName + "/" + id.type();
               if(trace)
               {
                  log.trace("ManagedProperty level ID for ManagedObject: "+propKey+", attachmentName: "+mo.getAttachmentName());
               }
               moRegistry.put(propKey, mo);
               checkForReferences(propKey, mo);
            }
            // See if this is a ManagementObjectRef
            ManagementObjectRef ref = (ManagementObjectRef) pannotations.get(ManagementObjectRef.class.getName());
            if ( ref != null )
            {
               // The reference key is the prop value + ref.type()
               if(trace)
               {
                  log.trace("Property("+prop.getName()+") references: "+ref);
               }
               Object refName = getRefName(prop.getValue());
               if (refName == null)
               {
                  refName = ref.name();
               }
               String targetKey = refName + "/" + ref.type();
               ManagedObject target = moRegistry.get(targetKey);
               if (target != null)
               {
                  if(trace)
                  {
                     log.trace("Resolved property("+prop.getName()+") reference to: "+targetKey);
                  }
                  prop.setTargetManagedObject(target);
               }
               else
               {
                  Set<ManagedProperty> referers =  unresolvedRefs.get(targetKey);
                  if (referers == null)
                  {
                     referers = new HashSet<ManagedProperty>();
                     unresolvedRefs.put(targetKey, referers);
                  }
                  referers.add(prop);
               }
            }
         }

         // Process generic values and extract nested ManagedObjects
         MetaType propType = prop.getMetaType();
         if (propType == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
         {
            processGenericValue (GenericValue.class.cast(prop.getValue()), md, context);
         }
         else if (propType.isArray())
         {
            ArrayMetaType amt = (ArrayMetaType) propType;
            MetaType etype = amt.getElementType();
            if (etype == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
            {
               ArrayValue avalue = (ArrayValue) prop.getValue();
               int length = avalue != null ? avalue.getLength() : 0;
               for(int n = 0; n < length; n ++)
               {
                  processGenericValue(GenericValue.class.cast(avalue.getValue(n)), md, context);
               }
            }
         }
         else if (propType.isCollection())
         {
            CollectionMetaType amt = (CollectionMetaType) propType;
            MetaType etype = amt.getElementType();
            if (etype == AbstractManagedObjectFactory.MANAGED_OBJECT_META_TYPE)
            {
               CollectionValue avalue = (CollectionValue) prop.getValue();
               if(avalue != null)
               {
                  MetaValue[] elements = avalue.getElements();
                  for(int n = 0; n < avalue.getSize(); n ++)
                  {
                     processGenericValue(GenericValue.class.cast(elements[n]), md, context);
                  }
               }
            }
         }
      }
   }

   /**
    * Get ref name.
    *
    * @param value property value
    * @return plain value
    */
   protected Object getRefName(Object value)
   {
      if (value instanceof MetaValue)
      {
         MetaValue metaValue = MetaValue.class.cast(value);
         if (metaValue.getMetaType().isSimple() == false)
         {
            throw new IllegalArgumentException("Can only get ref from simple value: " + value);
         }
         return SimpleValue.class.cast(metaValue).getValue();
      }
      return value;
   }

   /**
    * Check for references.
    * 
    * @param key the property key
    * @param mo the managed object
    */
   protected void checkForReferences(String key, ManagedObject mo)
   {
      Set<ManagedProperty> referers =  unresolvedRefs.get(key);
      log.trace("checkForReferences, "+key+" has referers: "+referers);
      if (referers != null)
      {
         for(ManagedProperty prop : referers)
         {
            prop.setTargetManagedObject(mo);
         }
         unresolvedRefs.remove(key);
      }      
   }

   /**
    * Process generic value.
    *
    * @param genericValue the generic value
    * @param md the managed deployment
    * @throws Exception for any error
    */
   protected void processGenericValue(GenericValue genericValue, ManagedDeployment md, ProfileViewProcessingContext context) throws Exception
   {
      // TODO: a null is probably an error condition
      if (genericValue != null)
      {
         ManagedObject propMO = ManagedObject.class.cast(genericValue.getValue());
         // TODO: a null is probably an error condition
         if (propMO != null)
         {
            processManagedObject(propMO, md, context);
         }
      }
   }
   
   /**
    * Get the annotation.
    * 
    * @param <T> the expected type
    * @param mo the managed object
    * @param annotationType the annotation type
    * @return the annotation
    */
   protected <T> T getAnnotation(ManagedObject mo, Class<T> annotationType)
   {
      return annotationType.cast(mo.getAnnotations().get(annotationType.getName()));
   }
   
}

