/*
* JBoss, Home of Professional Open Source
* Copyright 2009, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.profileservice.plugins.management.view;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jboss.deployers.spi.management.KnownComponentTypes;
import org.jboss.deployers.spi.management.NameMatcher;
import org.jboss.managed.api.ComponentType;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.profileservice.plugins.spi.ProfileView;
import org.jboss.profileservice.plugins.spi.ProfileViewProcessingContext;
import org.jboss.profileservice.spi.action.ModificationEvent;
import org.jboss.profileservice.spi.action.ModificationListener;

/**
 * The abstract profile view.
 * 
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class AbstractProfileView implements ProfileView, ProfileViewProcessingContext, ModificationListener
{

   /** The managed components by ComponentType. */
   private Map<ComponentType, Set<ManagedComponent>> compByCompType = new HashMap<ComponentType, Set<ManagedComponent>>();
   
   /** The managed deployments by name.  */
   private Map<String, ManagedDeployment> managedDeployments = new HashMap<String, ManagedDeployment>();
   
   /** Is modified. */
   private final AtomicBoolean modified;

   public AbstractProfileView()
   {
      // Set true for the first load()
      this.modified = new AtomicBoolean(true);
   }
   
   public boolean load()
   {
      if(this.modified.getAndSet(false) == false)
      {
         return false;
      }
      // Clear the stuff
      compByCompType.clear();
      managedDeployments.clear();

      return true;
   }
   
   /**
    * {@inheritDoc}
    */
   public Collection<String> getDeploymentNames()
   {
      return managedDeployments.keySet();
   }
   
   /**
    * {@inheritDoc}
    */
   public ManagedDeployment getManagedDeployment(final String name)
   {
      ManagedDeployment deployment = managedDeployments.get(name);
      if(deployment == null)
      {
         for(final ManagedDeployment d : managedDeployments.values())
         {
            // Only check top-level deployments for the simple name
            if(d.getParent() == null && name.equals(d.getSimpleName()))
            {
               deployment = d;
               break;
            }
         }
      }
      return deployment;
   }
   
   /**
    * {@inheritDoc}
    */
   public Set<String> getMatchingDeploymentName(final String regex)
   {
      final Collection<String> names = getDeploymentNames();
      final HashSet<String> matches = new HashSet<String>();
      final Pattern p = Pattern.compile(regex);
      for(final String name : names)
      {
         final Matcher m = p.matcher(name);
         if( m.matches() )
         {
            matches.add(name);
         }
      }
      return matches;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getMatchingDeployments(final String name, final NameMatcher<ManagedDeployment> matcher)
   {
      final Set<ManagedDeployment> matches = new HashSet<ManagedDeployment>();
      for(final ManagedDeployment deployment : this.managedDeployments.values())
      {
         if(matcher.matches(deployment, name))
         {
            matches.add(deployment);
         }
      }
      return matches;
   }
   
   /**
    * {@inheritDoc}
    */
   public Set<String> getDeploymentNamesForType(String type)
   {
      final Set<String> matches = new TreeSet<String>();
      for(final ManagedDeployment md : managedDeployments.values())
      {
         final String name = md.getName();
         final Set<String> types = md.getTypes();
         if(types != null)
         {
            if(types.contains(type))
            {
               matches.add(name);
            }
         }
      }
      return matches;
   }

   /**
    * {@inheritDoc}
    */
   public ManagedComponent getComponent(final String name, final ComponentType type)
   {
      final Set<ManagedComponent> components = compByCompType.get(type);
      if(components != null)
      {
         for(final ManagedComponent mc : components)
         {
            if(mc.getName().equals(name))
            {
               return mc;
            }
         }
      }
      return null;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ComponentType> getComponentTypes()
   {
      return compByCompType.keySet();
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getComponentsForType(ComponentType type)
   {
      Set<ManagedComponent> comps = null;
      // Check the any component type
      if(type.equals(KnownComponentTypes.ANY_TYPE))
      {
         HashSet<ManagedComponent> all = new HashSet<ManagedComponent>();
         for(Set<ManagedComponent> typeComps : compByCompType.values())
         {
            for(ManagedComponent comp : typeComps)
            {
               all.add(comp);
            }
         }
         comps = all;
      }
      else
      {
        comps = compByCompType.get(type);
      }
      if(comps == null)
      {
         comps = Collections.emptySet();
      }
      return comps;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedDeployment> getDeploymentsForType(String type)
   {
      final Set<String> names = getDeploymentNamesForType(type);
      final HashSet<ManagedDeployment> mds = new HashSet<ManagedDeployment>();
      for(String name : names)
      {
         mds.add(getManagedDeployment(name));
      }
      return mds;
   }

   /**
    * {@inheritDoc}
    */
   public Set<ManagedComponent> getMatchingComponents(final String name, final ComponentType type,
         final NameMatcher<ManagedComponent> matcher)
   {
      Set<ManagedComponent> components = compByCompType.get(type);
      Set<ManagedComponent> matched = new HashSet<ManagedComponent>();
      if(components != null)
      {
         for(ManagedComponent mc : components)
         {
            if(matcher.matches(mc, name))
               matched.add(mc);
         }
      }
      return matched;
   }

   // ProfileViewProcessingContext
   
   /**
    * {@inheritDoc}
    */
   public void addManagedComponent(final ManagedComponent component)
   {
      final ComponentType type = component.getType();
      Set<ManagedComponent> components = compByCompType.get(type);
      if(components == null)
      {
         components = new HashSet<ManagedComponent>();
         compByCompType.put(type, components);
      }
      components.add(component);
   }
   
   /**
    * {@inheritDoc}
    */
   public void addManagedDeployment(ManagedDeployment deployment)
   {
      this.managedDeployments.put(deployment.getName(), deployment);
   }
   
   /**
    * {@inheritDoc}
    */
   public void addManagedObject(ManagedObject managedObject)
   {
      throw new IllegalStateException("unsupported operation");
   }

   // ModificationListener
   
   /**
    * {@inheritDoc}
    */
   public void notify(ModificationEvent event)
   {
      // nothing here
   }

   public void markAsModified()
   {
      this.modified.set(true);
   }
   
}

