/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.scanning.plugins.filter;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.classloading.spi.visitor.ResourceFilter;
import org.jboss.scanning.spi.metadata.PathEntryMetaData;
import org.jboss.scanning.spi.metadata.PathMetaData;
import org.jboss.scanning.spi.metadata.ScanningMetaData;
import org.jboss.vfs.util.PathTokenizer;

/**
 * Simple recurse filter.
 *
 * It searches for path substring in url string,
 * and tries to match the tree structure as far as it goes.
 */
public class ScanningMetaDataRecurseFilter implements ResourceFilter
{
   /** Path tree roots */
   private Map<String, RootNode> roots;

   public ScanningMetaDataRecurseFilter(ScanningMetaData smd)
   {
      if (smd == null)
         throw new IllegalArgumentException("Null metadata");

      List<PathMetaData> paths = smd.getPaths();
      if (paths != null && paths.isEmpty() == false)
      {
         roots = new HashMap<String, RootNode>();
         for (PathMetaData pmd : paths)
         {
            RootNode pathNode = new RootNode();
            roots.put(pmd.getPathName(), pathNode);
            Set<PathEntryMetaData> includes = pmd.getIncludes();
            if (includes != null && includes.isEmpty() == false)
            {
               pathNode.explicitInclude = true;
               for (PathEntryMetaData pemd : includes)
               {
                  String name = pemd.getName();
                  String[] tokens = name.split("\\.");
                  Node current = pathNode;
                  for (String token : tokens)
                     current = current.addChild(token);
                  if (pemd.isRecurse())
                     current.recurse = true; // mark last one as recurse
               }
            }
         }
      }
   }

   public boolean accepts(ResourceContext resource)
   {
      if (roots == null)
         return false;

      URL url = resource.getUrl();
      String urlString = url.toExternalForm();
      for (Map.Entry<String, RootNode> root : roots.entrySet())
      {
         if (urlString.contains(root.getKey()))
         {
            RootNode rootNode = root.getValue();
            if (rootNode.explicitInclude) // we have explicit includes in path, try tree path
            {
               String resourceName = resource.getResourceName();
               List<String> tokens = PathTokenizer.getTokens(resourceName);
               Node current = rootNode;
               // let's try to walk some tree path
               for (String token : tokens)
               {
                  // if we're here, the rest is recursively matched
                  if (current.recurse)
                     break;

                  current = current.getChild(token);
                  // no fwd path
                  if (current == null)
                     return false;
               }
            }
            return true;
         }
      }
      return false;
   }

   private static class Node
   {
      private Map<String, Node> children;
      private boolean recurse;

      public Node addChild(String value)
      {
         if (children == null)
            children = new HashMap<String, Node>();

         Node child = children.get(value);
         if (child == null)
         {
            child = new Node();
            children.put(value, child);
         }
         return child;
      }

      public Node getChild(String child)
      {
         return children != null ? children.get(child) : null;
      }
   }

   private static class RootNode extends Node
   {
      private boolean explicitInclude;
   }
}
