/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.visitor;

import java.io.InputStream;
import java.util.Map;
import java.util.WeakHashMap;

import org.jboss.classloading.spi.visitor.ResourceContext;
import org.jboss.reflect.plugins.javassist.JavassistTypeInfoFactoryImpl;
import org.jboss.reflect.plugins.javassist.JavassistUtil;
import org.jboss.reflect.plugins.javassist.classpool.ClassPoolFactory;
import org.jboss.reflect.spi.TypeInfo;

import javassist.ClassPath;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.LoaderClassPath;

/**
 * Javassist based reflect provider.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class JavassistReflectProvider implements ReflectProvider
{
   /** The weak class pool map */
   private Map<ClassLoader, ClassPool> poolMap = new WeakHashMap<ClassLoader, ClassPool>();

   /**
    * Get class pool.
    *
    * @param cl the class loader
    * @return class pool
    */
   private ClassPool getClassPool(ClassLoader cl)
   {
      ClassPool pool = poolMap.get(cl);
      if (pool == null)
      {
         ClassPoolFactory factory = JavassistTypeInfoFactoryImpl.getPoolFactory();
         pool = factory.getPoolForLoader(cl);

         ClassPool defaultCP = ClassPool.getDefault();
         if (pool == null || pool == defaultCP)
         {
            pool = new ClassPool();
            ClassPath path = new LoaderClassPath(cl);
            pool.appendClassPath(path);
         }
         poolMap.put(cl, pool);
      }
      return pool;
   }

   public TypeInfo getTypeInfo(ResourceContext resource) throws Throwable
   {
      InputStream stream = resource.getInputStream();
      if (stream == null)
         throw new IllegalArgumentException("Null stream: " + resource);

      try
      {
         CtClass ctClass = getClassPool(resource.getClassLoader()).makeClassIfNew(stream);
         return JavassistUtil.toTypeInfo(ctClass);
      }
      finally
      {
         stream.close();
      }
   }
}