/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2010, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.scanning.plugins.helpers;

import java.util.HashSet;
import java.util.Set;

import org.jboss.classloading.spi.dependency.Module;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.scanning.spi.ScanningHandle;
import org.jboss.scanning.spi.ScanningPlugin;

/**
 * Deployment scanning handle aggregator.
 *
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class DeploymentScanningHandleAggregator<T extends ScanningHandle, U> extends AbstractScanningHandleAggregator<T, U>
{
   /** The deployment unit */
   private DeploymentUnit unit;

   public DeploymentScanningHandleAggregator(DeploymentUnit unit)
   {
      if (unit == null)
         throw new IllegalArgumentException("Null unit");
      this.unit = unit;
   }

   public U getAggregatedHandle(ScanningPlugin<T, U> plugin)
   {
      Set<U> handles = findHandles(plugin);
      return aggregate(plugin, handles);
   }

   private Set<U> findHandles(ScanningPlugin<T, U> plugin)
   {
      DeploymentUnit top = unit;
      while(top != null && top.isAttachmentPresent(Module.class) == false)
         top = top.getParent();

      if (top == null)
         throw new IllegalArgumentException("No CL module: " + unit);

      Set<U> handles = new HashSet<U>();
      recurse(top, plugin, handles, true);
      return handles;
   }

   private void recurse(DeploymentUnit current, ScanningPlugin<T, U> plugin, Set<U> handles, boolean allowModule)
   {
      // do not recurse into sub-deployments that have their own Module
      if (allowModule == false && current.isAttachmentPresent(Module.class))
         return;

      U handle = current.getAttachment(plugin.getAttachmentKey(), plugin.getHandleInterface());
      if (handle != null)
         handles.add(handle);

      for (DeploymentUnit child : current.getChildren())
      {
         recurse(child, plugin, handles, false);
      }
   }
}