/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat, Inc. and/or its affiliates, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,  
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.weld.bean.proxy;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.Set;

import javassist.NotFoundException;
import javassist.bytecode.Bytecode;
import javassist.bytecode.ClassFile;
import javassist.bytecode.Opcode;
import javassist.util.proxy.MethodHandler;

import javax.enterprise.inject.spi.Bean;

import org.jboss.weld.util.bytecode.BytecodeUtils;
import org.jboss.weld.util.bytecode.DescriptorUtils;
import org.jboss.weld.util.bytecode.MethodUtils;

/**
 * Proxy factory that generates client proxies, it uses optimizations that 
 * are not valid for other proxy types.
 * 
 * @author Stuart Douglas
 */
public class ClientProxyFactory<T> extends ProxyFactory<T>
{

   public final static String CLIENT_PROXY_SUFFIX = "ClientProxy";

   public ClientProxyFactory(Class<?> proxiedBeanType, Set<? extends Type> typeClosure, Bean<?> bean)
   {
      super(proxiedBeanType, typeClosure, bean);
   }

   public ClientProxyFactory(Class<?> proxiedBeanType, Set<? extends Type> typeClosure, String proxyName)
   {
      super(proxiedBeanType, typeClosure, proxyName);
   }

   /**
    * Calls methodHandler.invoke with a null method parameter in order to 
    * get the underlying instance. The invocation is then forwarded to 
    * this instance with generated bytecode.
    * 
    */
   protected Bytecode createForwardingMethodBody(ClassFile file, Method method) throws NotFoundException
   {
      // we can only use bytecode based invocation for some methods
      // at the moment we restrict it solely to public methods with public
      // return and parameter types
      boolean bytecodeInvocationAllowed = Modifier.isPublic(method.getModifiers()) && Modifier.isPublic(method.getReturnType().getModifiers());
      for (Class<?> paramType : method.getParameterTypes())
      {
         if (!Modifier.isPublic(paramType.getModifiers()))
         {
            bytecodeInvocationAllowed = false;
            break;
         }
      }
      if (!bytecodeInvocationAllowed)
      {
         return createInterceptorBody(file, method);
      }
      Bytecode b = new Bytecode(file.getConstPool());
      int localCount = MethodUtils.calculateMaxLocals(method) + 1;

      b.add(Opcode.ALOAD_0);
      b.addGetfield(file.getName(), "methodHandler", DescriptorUtils.classToStringRepresentation(MethodHandler.class));
      //pass null arguments to methodHandler.invoke
      b.add(Opcode.ALOAD_0);
      b.add(Opcode.ACONST_NULL);
      b.add(Opcode.ACONST_NULL);
      b.add(Opcode.ACONST_NULL);

      // now we have all our arguments on the stack
      // lets invoke the method
      b.addInvokeinterface(MethodHandler.class.getName(), "invoke", "(Ljava/lang/Object;Ljava/lang/reflect/Method;Ljava/lang/reflect/Method;[Ljava/lang/Object;)Ljava/lang/Object;", 5);

      b.addCheckcast(method.getDeclaringClass().getName());

      //now we should have the target bean instance on top of the stack
      //lets create the method invocation
      String methodDescriptor = DescriptorUtils.getMethodDescriptor(method);
      BytecodeUtils.loadParameters(b, methodDescriptor);
      if (method.getDeclaringClass().isInterface())
      {
         b.addInvokeinterface(method.getDeclaringClass().getName(), method.getName(), methodDescriptor, method.getParameterTypes().length + 1);
      }
      else
      {
         b.addInvokevirtual(method.getDeclaringClass().getName(), method.getName(), methodDescriptor);
      }
      // now we need to return the appropriate type
      BytecodeUtils.addReturnInstruction(b, method.getReturnType());

      if (b.getMaxLocals() < localCount)
      {
         b.setMaxLocals(localCount);
      }
      return b;
   }

   @Override
   protected String getProxyNameSuffix()
   {
      return CLIENT_PROXY_SUFFIX;
   }

}
