/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.test.ws.jaxws.cxf.fastinfoset;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintWriter;
import java.net.URL;

import javax.xml.namespace.QName;
import jakarta.xml.ws.Service;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.cxf.ext.logging.LoggingInInterceptor;
import org.apache.cxf.ext.logging.LoggingOutInterceptor;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.OperateOnDeployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit5.ArquillianExtension;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.FileAsset;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.jboss.ws.api.configuration.ClientConfigUtil;
import org.jboss.ws.api.configuration.ClientConfigurer;
import org.jboss.wsf.stack.cxf.client.UseThreadBusFeature;
import org.jboss.wsf.test.JBossWSTest;
import org.jboss.wsf.test.JBossWSTestHelper;
import org.jboss.wsf.test.WrapThreadContextClassLoader;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(ArquillianExtension.class)
public class FastInfosetTestCase extends JBossWSTest
{
   public static final String DEP1 = "dep1";
   public static final String DEP2 = "dep2";
   
   @ArquillianResource
   private URL baseURL;
   
   @Deployment(testable = false, name = DEP1)
   public static WebArchive createDeployment1() {
      WebArchive archive = ShrinkWrap.create(WebArchive.class, "jaxws-cxf-fastinfoset.war");
      archive.setManifest(new StringAsset("Manifest-Version: 1.0\n"
                  + "Dependencies: org.apache.cxf\n"))
            .addClass(org.jboss.test.ws.jaxws.cxf.fastinfoset.HelloWorldImpl.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.fastinfoset.HelloWorldFIImpl.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.fastinfoset.HelloWorldFeatureImpl.class)
            .add(new FileAsset(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/fastinfoset/WEB-INF/jaxws-endpoint-config.xml")), "jaxws-endpoint-config.xml")
            .setWebXML(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/fastinfoset/WEB-INF/web.xml"));
      return archive;
   }

   @Deployment(testable = false, name = DEP2)
   public static WebArchive createDeployment2() {
      WebArchive archive = ShrinkWrap.create(WebArchive.class, "jaxws-cxf-fastinfoset2.war");
      archive.setManifest(new StringAsset("Manifest-Version: 1.0\n"
                  + "Dependencies: org.apache.cxf\n"))
            .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/fastinfoset/WEB-INF/jboss-webservices.xml"), "jboss-webservices.xml")
            .addClass(org.jboss.test.ws.jaxws.cxf.fastinfoset.HelloWorldImpl.class);
      return archive;
   }

   @Override
   public String getClientJarPaths() {
      return JBossWSTestHelper.writeToFile(new JBossWSTestHelper.JarDeployment("jaxws-cxf-fastinfoset-client.jar") { {
         archive
               .addManifest()
               .addAsManifestResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/fastinfoset/META-INF-client/jaxws-client-config.xml"), "jaxws-client-config.xml");
         }
      });
   }

   @Test
   @RunAsClient
   @OperateOnDeployment(DEP1)
   public void testInfosetUsingFastInfosetAnnotation() throws Exception
   {
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      ByteArrayOutputStream in = new ByteArrayOutputStream();
      PrintWriter pwIn = new PrintWriter(in);
      PrintWriter pwOut = new PrintWriter(out);
      Bus bus = BusFactory.newInstance().createBus();
      BusFactory.setThreadDefaultBus(bus);
      try {
         bus.getInInterceptors().add(new LoggingInInterceptor(pwIn));
         bus.getOutInterceptors().add(new LoggingOutInterceptor(pwOut));
   
         URL wsdlURL = new URL(baseURL + "HelloWorldService/HelloWorldFIImpl?wsdl");
         QName serviceName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldFIService");
         Service service = Service.create(wsdlURL, serviceName, new UseThreadBusFeature());
         QName portQName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldFIImplPort");
         HelloWorldFI port = (HelloWorldFI) service.getPort(portQName, HelloWorldFI.class);
         assertEquals("helloworld", port.echo("helloworld"));
         assertTrue(out.toString().indexOf("application/fastinfoset") > -1, "request is expected fastinfoset");
         assertTrue(in.toString().indexOf("application/fastinfoset") > -1, "response is expected fastinfoset");
      } finally {
         bus.shutdown(true);
         pwOut.close();
         pwIn.close();
      }
   }
   
   @Test
   @RunAsClient
   @OperateOnDeployment(DEP1)
   public void testInfosetUsingFeature() throws Exception
   {
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      ByteArrayOutputStream in = new ByteArrayOutputStream();
      PrintWriter pwIn = new PrintWriter(in);
      PrintWriter pwOut = new PrintWriter(out);
      Bus bus = BusFactory.newInstance().createBus();
      BusFactory.setThreadDefaultBus(bus);
      try {
         bus.getInInterceptors().add(new LoggingInInterceptor(pwIn));
         bus.getOutInterceptors().add(new LoggingOutInterceptor(pwOut));
   
         URL wsdlURL = new URL(baseURL + "HelloWorldService/HelloWorldFeatureImpl?wsdl");
         QName serviceName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldFeatureService");
         Service service = Service.create(wsdlURL, serviceName, new UseThreadBusFeature());
         QName portQName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldFeatureImplPort");
         HelloWorldFeature port = (HelloWorldFeature) service.getPort(portQName, HelloWorldFeature.class);
         assertEquals("helloworldFeature", port.echo("helloworldFeature"));
         assertTrue(out.toString().indexOf("application/fastinfoset") > -1, "request is expected fastinfoset");
         assertTrue(in.toString().indexOf("application/fastinfoset") > -1, "response is expected fastinfoset");
      } finally {
         bus.shutdown(true);
         pwOut.close();
         pwIn.close();
      }
   }
   
   @Test
   @RunAsClient
   @WrapThreadContextClassLoader
   @OperateOnDeployment(DEP1)
   public void testInfosetUsingFeatureProperties() throws Exception
   {
      URL wsdlURL = new URL(baseURL + "HelloWorldService/HelloWorldImpl?wsdl");
      QName serviceName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldService");
      QName portQName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldImplPort");
      internalTestInfosetUsingFeatureProperties(wsdlURL, serviceName, portQName);
   }
   
   @Test
   @RunAsClient
   @WrapThreadContextClassLoader
   @OperateOnDeployment(DEP2)
   public void testInfosetUsingFeatureProperties2() throws Exception
   {
      URL wsdlURL = new URL(baseURL + "HelloWorldService?wsdl");
      QName serviceName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldService");
      QName portQName = new QName("http://org.jboss.ws/jaxws/cxf/fastinfoset", "HelloWorldImplPort");
      internalTestInfosetUsingFeatureProperties(wsdlURL, serviceName, portQName);
   }
   
   private void internalTestInfosetUsingFeatureProperties(URL wsdlURL, QName serviceName, QName portQName) throws Exception {
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      ByteArrayOutputStream in = new ByteArrayOutputStream();
      PrintWriter pwIn = new PrintWriter(in);
      PrintWriter pwOut = new PrintWriter(out);
      Bus bus = BusFactory.newInstance().createBus();
      BusFactory.setThreadDefaultBus(bus);
      try {
         bus.getInInterceptors().add(new LoggingInInterceptor(pwIn));
         bus.getOutInterceptors().add(new LoggingOutInterceptor(pwOut));
   
         Service service = Service.create(wsdlURL, serviceName, new UseThreadBusFeature());
         HelloWorld port = (HelloWorld) service.getPort(portQName, HelloWorld.class);
         
         ClientConfigurer configurer = ClientConfigUtil.resolveClientConfigurer();
         configurer.setConfigProperties(port, "META-INF/jaxws-client-config.xml", "Custom Client Config");
         
         assertEquals("helloworld", port.echo("helloworld"));
         assertTrue(out.toString().indexOf("application/fastinfoset") > -1, "request is expected fastinfoset");
         assertTrue(in.toString().indexOf("application/fastinfoset") > -1, "response is expected fastinfoset");
      } finally {
         bus.shutdown(true);
         pwOut.close();
         pwIn.close();
      }
   }
   
}
