/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.test.ws.jaxws.cxf.jbws3713;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit5.ArquillianExtension;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.jboss.wsf.stack.cxf.client.Constants;
import org.jboss.wsf.stack.cxf.client.ProviderImpl;
import org.jboss.wsf.test.JBossWSTest;
import org.jboss.wsf.test.JBossWSTestHelper;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(ArquillianExtension.class)
public class ClientBusStrategyTestCase extends JBossWSTest
{
   private final String FS = System.getProperty("file.separator"); // '/' on unix, '\' on windows

   @ArquillianResource
   private URL baseURL;
   
   @Deployment(testable = false)
   public static WebArchive createDeployment() {
      WebArchive archive = ShrinkWrap.create(WebArchive.class, "jaxws-cxf-jbws3713.war");
      archive.addManifest()
         .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloRequest.class)
         .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloResponse.class)
         .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloWSImpl.class)
         .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloWs.class)
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello.wsdl"), "wsdl/Hello.wsdl")
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello_schema1.xsd"), "wsdl/Hello_schema1.xsd")
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello_schema2.xsd"), "wsdl/Hello_schema2.xsd")
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello_schema3.xsd"), "wsdl/Hello_schema3.xsd")
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello_schema4.xsd"), "wsdl/Hello_schema4.xsd")
         .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/wsdl/Hello_schema5.xsd"), "wsdl/Hello_schema5.xsd");
      return archive;
   }

   public static final String CLIENT_JAR = JBossWSTestHelper.writeToFile(new JBossWSTestHelper.JarDeployment("jaxws-cxf-jbws3713-client.jar") { {
         archive
            .setManifest(new StringAsset("Manifest-Version: 1.0\n"
                  + "Main-Class: org.jboss.test.ws.jaxws.cxf.jbws3713.TestClient\n"
                    //TODO:review this jbossws-cxf-client export services dependency after jaf factory finder
                    //issue is fixed https://github.com/eclipse-ee4j/jaf/pull/91
                  + "Dependencies: org.jboss.ws.cxf.jbossws-cxf-client export services,org.apache.cxf.impl,org.jboss.ws.jaxws-client\n"))
             .addAsManifestResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3713/WEB-INF/client-permissions.xml"), "permissions.xml")
             .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.BusCounter.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloRequest.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloResponse.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelloWs.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.Helper.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.HelperUsignThreadLocal.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3713.TestClient.class);
      }
   });

   @Test
   @RunAsClient
   public void testClientWithNewBusStrategy() throws Exception
   {
      final int threadPoolSize = 4;
      final int invocations = 5;
      List<Integer> list = runJBossModulesClient(Constants.NEW_BUS_STRATEGY, baseURL + "/HelloService?wsdl", threadPoolSize, invocations);
      assertEquals(threadPoolSize, list.get(0).intValue());
      assertEquals(invocations, list.get(1).intValue());
   }

   @Test
   @RunAsClient
   public void testClientWithTCCLBusStrategy() throws Exception
   {
      final int threadPoolSize = 4;
      final int invocations = 5;
      List<Integer> list = runJBossModulesClient(Constants.TCCL_BUS_STRATEGY, baseURL + "/HelloService?wsdl", threadPoolSize, invocations);
      assertEquals(1, list.get(0).intValue());
      assertEquals(1, list.get(1).intValue());
   }

   @Test
   @RunAsClient
   public void testClientWithThreadBusStrategy() throws Exception
   {
      final int threadPoolSize = 4;
      final int invocations = 5;
      List<Integer> list = runJBossModulesClient(Constants.THREAD_BUS_STRATEGY, baseURL + "/HelloService?wsdl", threadPoolSize, invocations);
      assertEquals(threadPoolSize, list.get(0).intValue());
      assertEquals(threadPoolSize, list.get(1).intValue());
   }
   
   /**
    * Verifies jaxws client bus selection strategy controlled by system properties; in order for checking that,
    * starting a new process is required, as the system property is read once and cached in JBossWS. 
    * 
    * @param strategy
    * @param wsdlAddress
    * @param threadPoolSize
    * @param invocations
    * @return
    * @throws Exception
    */
   protected List<Integer> runJBossModulesClient(final String strategy,
                                                 final String wsdlAddress,
                                                 final int threadPoolSize,
                                                 final int invocations) throws Exception {
      StringBuilder sbuf = new StringBuilder();

      // java cmd
      File javaFile = new File (System.getProperty("java.home") + FS + "bin" + FS + "java");
      String javaCmd = javaFile.exists() ? javaFile.getCanonicalPath() : "java";
      sbuf.append(javaCmd);

      //properties
      String additionalJVMArgs = System.getProperty("additionalJvmArgs", "");
      additionalJVMArgs =  additionalJVMArgs.replace('\n', ' ');
      sbuf.append(" ").append(additionalJVMArgs);
      sbuf.append(" -Djakarta.xml.ws.spi.Provider=").append(ProviderImpl.class.getName());
      sbuf.append(" -D").append(Constants.JBWS_CXF_JAXWS_CLIENT_BUS_STRATEGY).append("=").append(strategy);

      // ref to jboss-modules jar
      final String jbh = System.getProperty("jboss.home");
      final String jbm = jbh + FS + "modules";
      final String jbmjar = jbh + FS + "jboss-modules.jar";
      sbuf.append(" -jar ").append(jbmjar);

      // input arguments to jboss-module's main
      sbuf.append(" -mp ").append(jbm);

      // wildfly9 security manage flag changed from -Djava.security.manager to -secmgr.
      // Can't pass -secmgr arg through arquillian because it breaks arquillian's
      // config of our tests.
      // the -secmgr flag MUST be provided as an input arg to jboss-modules so it must
      // come after the jboss-modules.jar ref.
      if (additionalJVMArgs.contains("-Djava.security.manager")) {
         sbuf.append(" ").append("-secmgr");
      }

      // our client jar is an input param to jboss-module
      final File f = new File(JBossWSTestHelper.getTestArchiveDir(), CLIENT_JAR);
      sbuf.append(" -jar ").append(f.getAbsolutePath());

      // input args to our client.jar main
      sbuf.append(" ").append(wsdlAddress).append(" ").append(threadPoolSize).append(" ").append(invocations);

      final String command = sbuf.toString();
      ByteArrayOutputStream bout = new ByteArrayOutputStream();
      executeCommand(command, bout);
      StringTokenizer st = new StringTokenizer(readFirstLine(bout), " ");
      List<Integer> list = new LinkedList<Integer>();
      while (st.hasMoreTokens()) {
         list.add(Integer.parseInt(st.nextToken()));
      }
      return list;
   }

   private static String readFirstLine(ByteArrayOutputStream bout) throws IOException {
      bout.flush();
      final byte[] bytes = bout.toByteArray();
      if (bytes != null) {
          BufferedReader reader = new BufferedReader(new java.io.StringReader(new String(bytes)));
          return reader.readLine();
      } else {
         return null;
      }
   }
}
