/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.test.ws.jaxws.jbws2074.handler;

import jakarta.annotation.PostConstruct;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import jakarta.xml.soap.SOAPElement;
import jakarta.xml.soap.SOAPException;
import jakarta.xml.soap.SOAPMessage;
import jakarta.xml.ws.WebServiceException;
import jakarta.xml.ws.handler.MessageContext;
import jakarta.xml.ws.handler.soap.SOAPMessageContext;

import org.jboss.logging.Logger;
import org.jboss.ws.api.handler.GenericSOAPHandler;

/**
 * This handler is initialized using manual JNDI lookup
 * and refer to the ejb-jar.xml or web.xml defined
 * environment entries.
 *
 * @author ropalka@redhat.com
 */
public class ManualResourcesHandler extends GenericSOAPHandler<SOAPMessageContext>
{
   // provide logging
   private static final Logger log = Logger.getLogger(ManualResourcesHandler.class);

   /**
    * java.lang.Boolean
    */

   private Boolean boolean0;

   /**
    * java.lang.Byte
    */

   private Byte byte0;

   /**
    * java.lang.Character
    */

   private Character character0;

   /**
    * java.lang.Short
    */

   private Short short0;

   /**
    * java.lang.Integer
    */

   private Integer integer0;

   /**
    * java.lang.Long
    */

   private Long long0;

   /**
    * java.lang.Float
    */

   private Float float0;

   /**
    * java.lang.Double
    */

   private Double double0;

   /**
    * java.lang.String
    */

   private String string;

   /**
    * Indicates whether handler is in correct state.
    */
   private boolean correctState;

   @PostConstruct
   private void init()
   {
      doManualJndiLookup();
   }

   private void doManualJndiLookup() {
      this.boolean0 = null;
      this.byte0 = null;
      this.character0 = null;
      this.short0 = null;
      this.integer0 = null;
      this.long0 = null;
      this.float0 = null;
      this.double0 = null;
      this.string = null;
      boolean correctInitialization = true;
      try {
         final InitialContext env = new InitialContext();

         // java.lang.Boolean
         this.boolean0 = (Boolean)env.lookup("java:comp/env/boolean");
         if (this.boolean0 == null || this.boolean0 != true)
         {
            log.error("Manual JNDI lookup for boolean0 failed");
            correctInitialization = false;
         }

         // java.lang.Byte
         this.byte0 = (Byte)env.lookup("java:comp/env/byte");
         if (this.byte0 == null || this.byte0 != (byte)1)
         {
            log.error("Manual JNDI lookup for byte0 failed");
            correctInitialization = false;
         }

         // java.lang.Character
         this.character0 = (Character)env.lookup("java:comp/env/character");
         if (this.character0 == null || this.character0 != 'c')
         {
            log.error("Manual JNDI lookup for character0 failed");
            correctInitialization = false;
         }

         // java.lang.Short
         this.short0 = (Short)env.lookup("java:comp/env/short");
         if (this.short0 == null || this.short0 != (short)5)
         {
            log.error("Manual JNDI lookup for short0 failed");
            correctInitialization = false;
         }

         // java.lang.Integer
         this.integer0 = (Integer)env.lookup("java:comp/env/integer");
         if (this.integer0 == null || this.integer0 != 7)
         {
            log.error("Manual JNDI lookup for integer0 failed");
            correctInitialization = false;
         }

         // java.lang.Long
         this.long0 = (Long)env.lookup("java:comp/env/long");
         if (this.long0 == null || this.long0 != 11L)
         {
            log.error("Manual JNDI lookup for long0 failed");
            correctInitialization = false;
         }

         // java.lang.Float
         this.float0 = (Float)env.lookup("java:comp/env/float");
         if (this.float0 == null || this.float0 != 13.0f)
         {
            log.error("Manual JNDI lookup for float0 failed");
            correctInitialization = false;
         }

         // java.lang.Double
         this.double0 = (Double)env.lookup("java:comp/env/double");
         if (this.double0 == null || this.double0 != 17.0)
         {
            log.error("Manual JNDI lookup for double0 failed");
            correctInitialization = false;
         }

         // java.lang.String
         this.string = (String)env.lookup("java:comp/env/string");
         if ("s".equals(this.string) == false)
         {
            log.error("Manual JNDI lookup for string0 failed");
            correctInitialization = false;
         }

         this.correctState = correctInitialization;
      } catch (final NamingException e) {
          this.correctState = false;
      }
   }

   @Override
   public boolean handleOutbound(SOAPMessageContext msgContext)
   {
      return ensureInjectionsAndInitialization(msgContext, "Outbound");
   }

   @Override
   public boolean handleInbound(SOAPMessageContext msgContext)
   {
      return ensureInjectionsAndInitialization(msgContext, "Inbound");
   }

   private boolean ensureInjectionsAndInitialization(MessageContext msgContext, String direction)
   {
      if (!this.correctState)
      {
         throw new WebServiceException("Unfunctional manual JNDI lookups in @PostConstruct annotated methods");
      }
      doManualJndiLookup();
      if (!this.correctState)
      {
         throw new WebServiceException("Unfunctional manual JNDI lookups in handler execution methods");
      }

      try
      {
         SOAPMessage soapMessage = ((SOAPMessageContext)msgContext).getMessage();
         SOAPElement soapElement = (SOAPElement)soapMessage.getSOAPBody().getChildElements().next();
         soapElement = (SOAPElement)soapElement.getChildElements().next();

         String oldValue = soapElement.getValue();
         String newValue = oldValue + ":" + direction + ":ManualResourcesHandler";
         soapElement.setValue(newValue);

         log.debug("oldValue: " + oldValue);
         log.debug("newValue: " + newValue);

         return true;
      }
      catch (SOAPException ex)
      {
         throw new WebServiceException(ex);
      }
   }

}
