/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.jboss.test.ws.jaxws.cxf.jbws3648;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.PrintWriter;
import java.net.URL;

import javax.xml.namespace.QName;
import jakarta.xml.ws.BindingProvider;
import jakarta.xml.ws.Service;

import org.apache.cxf.Bus;
import org.apache.cxf.BusFactory;
import org.apache.cxf.ext.logging.LoggingInInterceptor;
import org.apache.cxf.message.Message;
import org.apache.cxf.ws.security.SecurityConstants;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit5.ArquillianExtension;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.jboss.wsf.stack.cxf.client.UseThreadBusFeature;
import org.jboss.wsf.test.CryptoCheckHelper;
import org.jboss.wsf.test.JBossWSTest;
import org.jboss.wsf.test.JBossWSTestHelper;
import org.jboss.wsf.test.WrapThreadContextClassLoader;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;

/**
 * 
 *
 * @author alessio.soldano@jboss.com
 * @since 13-Jun-2013
 */
@ExtendWith(ArquillianExtension.class)
public class PolicyAttachmentTestCase extends JBossWSTest
{
   @ArquillianResource
   private URL baseURL;
   
   @Deployment(testable = false)
   public static WebArchive createDeployment() {
      WebArchive archive = ShrinkWrap.create(WebArchive.class, "jaxws-cxf-jbws3648-b.war");
      archive.setManifest(new StringAsset("Manifest-Version: 1.0\n"
                  + "Dependencies: org.jboss.ws.cxf.jbossws-cxf-client, org.apache.ws.security\n"))
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3648.EndpointFour.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3648.EndpointFourImpl.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3648.EndpointThree.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3648.EndpointThreeImpl.class)
            .addClass(org.jboss.test.ws.jaxws.cxf.jbws3648.KeystorePasswordCallback.class)
            .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/WEB-INF/bob.jks"), "classes/bob.jks")
            .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/WEB-INF/bob.properties"), "classes/bob.properties")
            .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/policies/My-WSSE-conf-BINDING.xml"),
                  "classes/META-INF/policies/My-WSSE-conf-BINDING.xml")
            .addAsWebInfResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/policies/org.jboss.wsf.stack.cxf.extensions.policy.PolicyAttachmentStore"),
                  "classes/META-INF/policies/org.jboss.wsf.stack.cxf.extensions.policy.PolicyAttachmentStore");
      return archive;
   }

   @Override
   public String getClientJarPaths() {
      return JBossWSTestHelper.writeToFile(new JBossWSTestHelper.JarDeployment("jaxws-cxf-jbws3648-b-client.jar") { {
         archive
               .addManifest()
               .addAsManifestResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/META-INF/alice.jks"), "alice.jks")
               .addAsManifestResource(new File(JBossWSTestHelper.getTestResourcesDir() + "/jaxws/cxf/jbws3648-b/META-INF/alice.properties"), "alice.properties");
         }
      });
   }

   @Test
   @RunAsClient
   @WrapThreadContextClassLoader
   public void testEndpointWithWSSEAndWSA() throws Exception {
      Bus bus = BusFactory.newInstance().createBus();
      BusFactory.setThreadDefaultBus(bus);
      ByteArrayOutputStream bos = new ByteArrayOutputStream();
      PrintWriter pw = new PrintWriter(bos);
      try {
         bus.getInInterceptors().add(new LoggingInInterceptor(pw));
         
         URL wsdlURL = new URL(baseURL + "/ServiceThree?wsdl");
         QName serviceName = new QName("http://org.jboss.ws.jaxws.cxf/jbws3648", "ServiceThree");
         Service service = Service.create(wsdlURL, serviceName, new UseThreadBusFeature());
         EndpointThree proxy = (EndpointThree)service.getPort(EndpointThree.class);
         setupWsse((BindingProvider)proxy);
         
         assertEquals("Foo3", proxy.echo("Foo3"));
         final String m = bos.toString();
         assertTrue(m.contains("http://www.w3.org/2005/08/addressing") && m.contains("http://www.w3.org/2005/08/addressing/anonymous"), "WS-Addressing was not enabled!");
         assertTrue(m.contains("http://www.w3.org/2001/04/xmlenc#rsa-1_5") && m.contains("http://www.w3.org/2001/04/xmlenc#tripledes-cbc"), "WS-Security was not enabled!");
      } finally {
         bus.shutdown(true);
         pw.close();
         bos.close();
      }
   }
   
   @Test
   @RunAsClient
   @WrapThreadContextClassLoader
   public void testEndpointWithCustomWSSEAndWSA() throws Exception {
      Bus bus = BusFactory.newInstance().createBus();
      BusFactory.setThreadDefaultBus(bus);
      ByteArrayOutputStream bos = new ByteArrayOutputStream();
      PrintWriter pw = new PrintWriter(bos);
      try {
         bus.getInInterceptors().add(new LoggingInInterceptor(pw));
         
         URL wsdlURL = new URL(baseURL + "/ServiceFour?wsdl");
         QName serviceName = new QName("http://org.jboss.ws.jaxws.cxf/jbws3648", "ServiceFour");
         Service service = Service.create(wsdlURL, serviceName, new UseThreadBusFeature());
         EndpointFour proxy = (EndpointFour)service.getPort(EndpointFour.class);
         setupWsse((BindingProvider)proxy);
         
         try {
            assertEquals("Foo4", proxy.echo("Foo4"));
         } catch (Exception e) {
            throw CryptoCheckHelper.checkAndWrapException(e);
         }
         final String m = bos.toString();
         assertTrue(m.contains("http://www.w3.org/2005/08/addressing") && m.contains("http://www.w3.org/2005/08/addressing/anonymous"), "WS-Addressing was not enabled!");
         assertTrue(m.contains("http://www.w3.org/2001/04/xmlenc#rsa-1_5") && m.contains("http://www.w3.org/2001/04/xmlenc#aes256-cbc"), "WS-Security was not enabled!");
      } finally {
         bus.shutdown(true);
         pw.close();
         bos.close();
      }
   }
   
   private void setupWsse(BindingProvider proxy)
   {
      proxy.getRequestContext().put(SecurityConstants.CALLBACK_HANDLER, new KeystorePasswordCallback());
      proxy.getRequestContext().put(SecurityConstants.SIGNATURE_PROPERTIES, Thread.currentThread().getContextClassLoader().getResource("META-INF/alice.properties"));
      proxy.getRequestContext().put(SecurityConstants.ENCRYPT_PROPERTIES, Thread.currentThread().getContextClassLoader().getResource("META-INF/alice.properties"));
      proxy.getRequestContext().put(SecurityConstants.SIGNATURE_USERNAME, "alice");
      proxy.getRequestContext().put(SecurityConstants.ENCRYPT_USERNAME, "bob");
      proxy.getRequestContext().put(Message.RECEIVE_TIMEOUT, 120000);
   }
}
