/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.test.ws.projectGenerator;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;

import javax.xml.namespace.QName;
import javax.xml.ws.Service;

import org.jboss.wsf.test.JBossWSTest;
import org.jboss.wsf.test.JBossWSTestHelper;

/**
 * A test case for the user project generator:
 * - creates a user-project.properties file with the configuration (i.e. the output users get with the interactive setup-new-project Ant target)
 * - calls the project generator (create-new-project Ant target)
 * - writes a simple endpoint implementation in the project workspace
 * - calls the deploy target of the generated build file
 * - invokes the deployed simple endpoint
 * 
 * @author alessio.soldano@jboss.com
 * @since 28-Apr-2008
 */
public class ProjectGeneratorTestCase extends JBossWSTest
{
   private static final String FS = System.getProperty("file.separator"); // '/' on unix, '\' on windows
   private String jbossHome;
   private File workspaceHome;
   private File binDistroDir;
   private String projectName = "GeneratorTestProject";
   
   private String endpointURL = "http://" + getServerHost() + ":8080/" + projectName + "/EndpointImpl";
   private String targetNS = "http://projectGenerator.ws.test.jboss.org/";
   
   protected void setUp() throws Exception
   {
      super.setUp();
      jbossHome = System.getProperty("jboss.home");
      workspaceHome = new File(".");
      binDistroDir = new File("..");
   }
   
   public void testGenerator() throws Exception
   {
      if (!isDistroTest())
      {
         System.out.println("Skipping this test since it is meant to be run on binary distribution only.");
         return;
      }
      File userPrjProp = new File(binDistroDir, "user-project.properties");
      writeUserProjectProperties(userPrjProp);
      File distroBuild = new File(binDistroDir, "build.xml");
      String integrationTarget = System.getProperty("jbossws.integration.target");
      assertStatusCode(executeCommand("ant -f " + distroBuild.getAbsolutePath() + " -D" + integrationTarget + ".home=" + jbossHome + " -Djbossws.integration.target="
            + integrationTarget + " create-project"), "Error while creating the user project!");
      File projectHomeDir = new File(workspaceHome, projectName);
      File packageDir = new File(projectHomeDir.getCanonicalPath() + FS + "src" + FS + "main" + FS + "java" + FS + "org" + FS + "jboss" + FS + "test" + FS + "ws" + FS + "projectGenerator");
      packageDir.mkdirs();
      File endpointImpl = new File(packageDir, "EndpointImpl.java");
      writeEndpointImpl(endpointImpl);
      File endpointInterface = new File(packageDir, "Endpoint.java");
      writeEndpointInterface(endpointInterface);
      File projectBuild = new File(projectHomeDir.getCanonicalPath(), "build.xml");
      try
      {
         assertStatusCode(executeCommand("ant -f " + projectBuild + " deploy"), "Error while compiling / deploying the user project!");
         //Sleeping 15 sec to let the deployment scanner pick up the user project jar
         Thread.sleep(15000);
         //Running the actual test
         URL wsdlURL = new URL(endpointURL + "?wsdl");
         QName serviceName = new QName(targetNS, "EndpointService");
         Service service = Service.create(wsdlURL, serviceName);
         Endpoint port = (Endpoint)service.getPort(Endpoint.class);
         Object retObj = port.echo("Hello");
         assertEquals("Hello", retObj);
      }
      finally
      {
         assertStatusCode(executeCommand("ant -f " + projectBuild + " undeploy"), "Error while undeploying the user project");
         assertStatusCode(executeCommand("ant -f " + distroBuild.getCanonicalPath() + " delete-project"), "Error while deleting the user project");
         userPrjProp.delete();
      }
   }
   
   private void writeEndpointImpl(File file) throws Exception
   {
      StringBuffer sb = new StringBuffer();
      sb.append("package org.jboss.test.ws.projectGenerator;\n");
      sb.append("import javax.jws.WebService;\n");
      sb.append("import javax.ejb.Stateless;\n");
      sb.append("import org.jboss.logging.Logger;\n");
      sb.append("@Stateless\n");
      sb.append("@WebService(name=\"Endpoint\", serviceName=\"EndpointService\", endpointInterface = \"org.jboss.test.ws.projectGenerator.Endpoint\")\n");
      sb.append("public class EndpointImpl {\n");
      sb.append("  private static Logger log = Logger.getLogger(EndpointImpl.class);\n");
      sb.append("  public String echo(String input) {\n");
      sb.append("    log.info(\"echo: \" + input);\n");
      sb.append("    return input;\n");
      sb.append("  }\n");
      sb.append("}\n");
      BufferedWriter out = new BufferedWriter(new FileWriter(file));
      out.write(sb.toString());
      out.close();
   }
   
   private void writeEndpointInterface(File file) throws Exception
   {
      StringBuffer sb = new StringBuffer();
      sb.append("package org.jboss.test.ws.projectGenerator;\n");
      sb.append("import javax.jws.WebMethod;\n");
      sb.append("import javax.jws.WebService;\n");
      sb.append("import javax.jws.soap.SOAPBinding;\n");
      sb.append("@WebService (name=\"Endpoint\")\n");
      sb.append("@SOAPBinding(style = SOAPBinding.Style.RPC)\n");
      sb.append("public interface Endpoint {\n");
      sb.append("  @WebMethod(operationName = \"echoString\", action = \"urn:EchoString\")\n");
      sb.append("  String echo(String input);\n");
      sb.append("}\n");
      BufferedWriter out = new BufferedWriter(new FileWriter(file));
      out.write(sb.toString());
      out.close();
   }
   
   private void writeUserProjectProperties(File file) throws Exception
   {
      StringBuffer sb = new StringBuffer();
      sb.append("#JBossWS user project generator test\n");
      sb.append("project.name=" + projectName + "\n");
      sb.append("project.jboss.home=" + jbossHome + "\n");
      sb.append("project.type=jar\n");
      sb.append("project.jboss.conf=default\n");
      sb.append("workspace.home=" + workspaceHome.getCanonicalPath() + "\n");
      BufferedWriter out = new BufferedWriter(new FileWriter(file));
      out.write(sb.toString());
      out.close();
   }
   
   private Process executeCommand(String command) throws IOException
   {
      // be verbose      
      System.out.println("cmd: " + command);

      Process p = Runtime.getRuntime().exec(command);
      return p;
   }
   
   private void assertStatusCode(Process p, String error) throws Exception
   {
      // check status code
      int status = p.waitFor();
      if (p.exitValue() != 0)
      {
         printStream(p.getInputStream());
         printStream(p.getErrorStream());
      }
      assertTrue(error, status == 0);
   }
   
   private void printStream(InputStream is) throws IOException
   {
      BufferedReader in = new BufferedReader(new InputStreamReader(is));
      StringBuffer buffer = new StringBuffer();
      String line;
      while ((line = in.readLine()) != null) {
        buffer.append(line + "\n");
      }
      System.out.println(buffer.toString() + "\n");
   }
   
   private boolean isDistroTest() throws Exception
   {
      File build = new File(".." + FS + "build.xml");
      if (!build.exists())
      {
         String testResDir = JBossWSTestHelper.getTestResourcesDir();
         build = new File(testResDir.substring(0, testResDir.lastIndexOf("modules")) + "build.xml");
      }
      if (!build.exists())
         throw new Exception("Unable to find build.xml!");
      Process p = executeCommand("ant -f " + build.getAbsolutePath() + " -p");
      p.waitFor();
      BufferedReader in = new BufferedReader(new InputStreamReader(p.getInputStream()));
      String line;
      while ((line = in.readLine()) != null) {
        if (line.contains("build-bin-dist"))
           return false;
      }
      return true;
   }
}
