/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.framework.deployment;

import org.jboss.wsf.spi.ComposableRuntime;
import org.jboss.wsf.spi.WSFRuntime;
import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.DeploymentAspect;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.transport.*;

import java.util.Map;
import java.util.HashMap;

/**
 * Invokes a HTTP {@link org.jboss.wsf.spi.transport.TransportManager} 
 *
 * @author Thomas.Diesler@jboss.org
 * @author Heiko.Braun@jboss.com
 * 
 * @since 19-May-2007
 */
public class HttpTransportDeploymentAspect extends DeploymentAspect
{
   @Override
   public void create(Deployment dep, WSFRuntime runtime)
   {
      TransportManagerFactory tmf = ((ComposableRuntime)runtime).getTransportManagerFactory();
      TransportManager httpTransport = tmf.createTransportManager(Protocol.HTTP);

      // Context root
      String contextRoot = dep.getService().getContextRoot();
      if (contextRoot == null)
         throw new IllegalStateException("Cannot obtain context root");

      // Setup transport for each endpoint
      for (Endpoint ep : dep.getService().getEndpoints())
      {
         String urlPattern = ep.getURLPattern();
         if (urlPattern == null)
            throw new IllegalStateException("Cannot obtain url pattern");
         if (urlPattern.endsWith("/*"))
            urlPattern = urlPattern.substring(0, urlPattern.length() - 2);

         HttpSpec spec = new HttpSpec(contextRoot, urlPattern);

         // Propagate stack provided properties through TransportSpec (ServletClass and InitParameter)
         String servletClass = (String)dep.getProperty(HttpSpec.PROPERTY_WEBAPP_SERVLET_CLASS);
         if (servletClass == null) throw new IllegalStateException("Cannot obtain context property: " + HttpSpec.PROPERTY_WEBAPP_SERVLET_CLASS);

         Object ctxProps = dep.getProperty(HttpSpec.PROPERTY_WEBAPP_CONTEXT_PARAMETERS);
         Map ctxParams = ctxProps != null ?  (Map)ctxProps:new HashMap();

         spec.setServletClass(servletClass);
         spec.getContextParameter().putAll(ctxParams);
         
         ListenerRef ref = httpTransport.createListener(ep, spec);
         ep.addAttachment(ListenerRef.class, ref);
         ep.setAddress(ref.getAddress().toString());
      }
   }
   
   @Override
   public void destroy(Deployment dep, WSFRuntime runtime)
   {
      TransportManagerFactory tmf = ((ComposableRuntime)runtime).getTransportManagerFactory();
      TransportManager httpTransport = tmf.createTransportManager(Protocol.HTTP);

      for (Endpoint ep : dep.getService().getEndpoints())
      {
         ListenerRef listenerRef = ep.getAttachment(ListenerRef.class);
         httpTransport.destroyListener(listenerRef);
      }
   }
   
}
