/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro.client;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.ws.handler.PortInfo;

import org.jboss.logging.Logger;

import com.sun.xml.ws.api.streaming.XMLStreamReaderFactory;
import com.sun.xml.ws.handler.HandlerChainsModel;
import com.sun.xml.ws.streaming.XMLStreamReaderUtil;

/**
 * Handler resolver for Metro integration
 * @author richard.opalka@jboss.com
 */
public final class HandlerResolverImpl implements HandlerResolver
{
   private static final Logger log = Logger.getLogger(HandlerResolverImpl.class);
   private final String handlerFile;
   private final Class<?> clazz;
   private HandlerChainsModel handlerChainsModel;
   
   public HandlerResolverImpl(String handlerFile, Class<?> clazz)
   {
      super();
      this.handlerFile = handlerFile;
      this.clazz = clazz;
      this.init();
   }
   
   public List<Handler> getHandlerChain(PortInfo portInfo)
   {
      return this.handlerChainsModel.getHandlersForPortInfo(portInfo).getHandlers();
   }

   private void init()
   {
      final InputStream is = this.getInputStream(this.handlerFile, this.clazz);
      this.handlerChainsModel = this.buildHandlerChainsModel(is, clazz);
   }
   
   private InputStream getInputStream(String filename, Class<?> wsClass)
   {
      URL fileURL = null;
      log.debug("processHandlerChain [" + filename + "] on: " + wsClass.getName());

      // Try the filename as URL
      try
      {
         fileURL = new URL(filename);
      }
      catch (MalformedURLException ex)
      {
         // ignore
      }

      // Try the filename as File
      if (fileURL == null)
      {
         try
         {
            File file = new File(filename);
            if (file.exists())
               fileURL = file.toURI().toURL();
         }
         catch (MalformedURLException e)
         {
            // ignore
         }
      }

      // Try the filename as Resource
      if (fileURL == null)
      {
         log.debug(wsClass.getProtectionDomain().getCodeSource());
         log.debug(wsClass.getClassLoader());
         fileURL = wsClass.getClassLoader().getResource(filename);
      }

      // Try the filename relative to class
      if (fileURL == null)
      {
         String filepath = filename;
         String packagePath = wsClass.getPackage().getName().replace('.', '/');
         String resourcePath = packagePath + "/" + filepath;
         while (filepath.startsWith("../"))
         {
            packagePath = packagePath.substring(0, packagePath.lastIndexOf("/"));
            filepath = filepath.substring(3);
            resourcePath = packagePath + "/" + filepath;
         }
         fileURL = wsClass.getClassLoader().getResource(resourcePath);
      }

      if (fileURL == null)
         throw new WebServiceException("Cannot resolve handler file '" + filename + "' on " + wsClass.getName());

      try
      {
         return fileURL.openStream();
      }
      catch (IOException ioe)
      {
         throw new WebServiceException(ioe);
      }
   }
   
   private HandlerChainsModel buildHandlerChainsModel(InputStream is, Class<?> target)
   {
      XMLStreamReader reader = XMLStreamReaderFactory.create(null, is, true);
      XMLStreamReaderUtil.nextElementContent(reader);
      HandlerChainsModel handlerChainsModel = HandlerChainsModel.parseHandlerConfigFile(target, reader);
      try
      {
         reader.close();
         is.close();
      }
      catch (XMLStreamException e)
      {
         log.error("Error while closing reader!", e);
         throw new WebServiceException(e.getMessage());
      }
      catch (IOException e)
      {
         log.error("Error while closing input stream!", e);
         throw new WebServiceException(e.getMessage());
      }
      return handlerChainsModel;
   }

}
