/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.wsf.stack.metro;

import org.jboss.wsf.spi.deployment.Deployment;
import org.jboss.wsf.spi.deployment.DeploymentAspect;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.WSFRuntime;
import org.jboss.wsf.stack.metro.metadata.sunjaxws.DDEndpoint;
import org.jboss.wsf.stack.metro.metadata.sunjaxws.DDEndpoints;
import org.jboss.wsf.stack.metro.metadata.RuntimeModelDeploymentAspect;

import javax.xml.ws.BindingType;
import javax.xml.ws.soap.SOAPBinding;

import java.net.URL;
import java.util.HashMap;
import java.util.Map;

/**
 * An aspect that generates sun-jaxws.xml 
 *
 * @author Thomas.Diesler@jboss.org
 * @since 10-May-2007
 */
public class SunJaxwsDeploymentAspect extends DeploymentAspect
{
   @Override
   @SuppressWarnings("unchecked")
   public void create(Deployment dep, WSFRuntime runtime)
   {
      DDEndpoints ddEndpoints = dep.getAttachment(DDEndpoints.class);
      if (ddEndpoints == null)
      {
         ddEndpoints = new DDEndpoints();
         for (Endpoint endpoint : dep.getService().getEndpoints())
         {
            String endpointName = endpoint.getShortName();
            String targetBean = endpoint.getTargetBeanName();
            String urlPattern = endpoint.getURLPattern();

            DDEndpoint ddEndpoint = new DDEndpoint(endpointName, targetBean, urlPattern);
            Class beanClass = endpoint.getTargetBeanClass();
            BindingType bindingType = (BindingType)beanClass.getAnnotation(BindingType.class);
            
            if (bindingType != null && bindingType.value().length() > 0)
            {
               String binding = bindingType.value();
               ddEndpoint.setBinding(binding);
               ddEndpoint.setEnableMTOM(isMtomEnabled(binding));
            }

            log.info("Add " + ddEndpoint);
            ddEndpoints.addEndpoint(ddEndpoint);
         }

         dep.addAttachment(DDEndpoints.class, ddEndpoints);
         putSunJaxwsConfigToDeployment(dep, ddEndpoints.createFileURL());
      }
   }

   @Override
   public void destroy(Deployment dep, WSFRuntime runtime)
   {
      DDEndpoints dd = dep.getAttachment(DDEndpoints.class);
      if (dd != null)
      {
         dd.destroyFileURL();
      }
   }
   
   /**
    * Puts SUN JAX-WS config file to deployment property <b>org.jboss.ws.webapp.ContextParameterMap</b> map
    * @param dep deployment where to put
    * @param dd beans to be put
    */
   private static void putSunJaxwsConfigToDeployment(Deployment dep, URL jaxwsConfigURL)
   {
      // get property map
      String propKey = "org.jboss.ws.webapp.ContextParameterMap";
      Map<String, String> contextParams = (Map<String, String>)dep.getProperty(propKey);
      if (contextParams == null)
      {
         // if there's no associated map with the property create it now
         contextParams = new HashMap<String, String>();
         dep.setProperty(propKey, contextParams);
      }
      // put sun jaxws config URL to the property map
      contextParams.put(RuntimeModelDeploymentAspect.PARAM_SUN_JAXWS_URL, jaxwsConfigURL.toExternalForm());
   }
   
   private static boolean isMtomEnabled(String binding)
   {
      return binding.equals(SOAPBinding.SOAP11HTTP_MTOM_BINDING) || binding.equals(SOAPBinding.SOAP12HTTP_MTOM_BINDING);
   }
}