"use strict";
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (Object.hasOwnProperty.call(mod, k)) result[k] = mod[k];
    result["default"] = mod;
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
const typescript_estree_1 = require("@typescript-eslint/typescript-estree");
const util = __importStar(require("../util"));
exports.default = util.createRule({
    name: 'explicit-function-return-type',
    meta: {
        type: 'problem',
        docs: {
            description: 'Require explicit return types on functions and class methods',
            category: 'Stylistic Issues',
            recommended: 'warn',
        },
        messages: {
            missingReturnType: 'Missing return type on function.',
        },
        schema: [
            {
                type: 'object',
                properties: {
                    allowExpressions: {
                        type: 'boolean',
                    },
                },
                additionalProperties: false,
            },
        ],
    },
    defaultOptions: [
        {
            allowExpressions: true,
        },
    ],
    create(context, [options]) {
        /**
         * Checks if the parent of a function expression is a constructor.
         * @param parent The parent of a function expression node
         */
        function isConstructor(parent) {
            return (parent.type === typescript_estree_1.AST_NODE_TYPES.MethodDefinition &&
                parent.kind === 'constructor');
        }
        /**
         * Checks if the parent of a function expression is a setter.
         * @param parent The parent of a function expression node
         */
        function isSetter(parent) {
            return (parent.type === typescript_estree_1.AST_NODE_TYPES.MethodDefinition && parent.kind === 'set');
        }
        /**
         * Checks if a function declaration/expression has a return type.
         * @param node The node representing a function.
         */
        function checkFunctionReturnType(node) {
            if (!node.returnType &&
                node.parent &&
                !isConstructor(node.parent) &&
                !isSetter(node.parent) &&
                util.isTypeScriptFile(context.getFilename())) {
                context.report({
                    node,
                    messageId: 'missingReturnType',
                });
            }
        }
        /**
         * Checks if a function declaration/expression has a return type.
         * @param {ASTNode} node The node representing a function.
         */
        function checkFunctionExpressionReturnType(node) {
            if (options.allowExpressions &&
                node.parent &&
                node.parent.type !== typescript_estree_1.AST_NODE_TYPES.VariableDeclarator &&
                node.parent.type !== typescript_estree_1.AST_NODE_TYPES.MethodDefinition) {
                return;
            }
            checkFunctionReturnType(node);
        }
        return {
            ArrowFunctionExpression: checkFunctionExpressionReturnType,
            FunctionDeclaration: checkFunctionReturnType,
            FunctionExpression: checkFunctionExpressionReturnType,
        };
    },
});
//# sourceMappingURL=explicit-function-return-type.js.map