/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.utils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.cert.Certificate;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.jboss.logging.Logger;
import org.keycloak.common.util.BouncyIntegration;
import org.keycloak.models.KeycloakSession;
import org.keycloak.truststore.TruststoreProvider;

public final class CRLUtils {
    private static final Logger log = Logger.getLogger(CRLUtils.class);
    private static final String CRL_DISTRIBUTION_POINTS_OID = "2.5.29.31";

    public static List<String> getCRLDistributionPoints(X509Certificate cert) throws IOException {
        CRLDistPoint crlDP;
        DEROctetString octetString;
        byte[] data = cert.getExtensionValue(CRL_DISTRIBUTION_POINTS_OID);
        if (data == null) {
            return Collections.emptyList();
        }
        LinkedList<String> distributionPointUrls = new LinkedList<String>();
        try (ASN1InputStream crldpExtensionInputStream = new ASN1InputStream((InputStream)new ByteArrayInputStream(data));){
            octetString = (DEROctetString)crldpExtensionInputStream.readObject();
        }
        byte[] octets = octetString.getOctets();
        try (ASN1InputStream crldpInputStream = new ASN1InputStream((InputStream)new ByteArrayInputStream(octets));){
            crlDP = CRLDistPoint.getInstance((Object)crldpInputStream.readObject());
        }
        for (DistributionPoint dp : crlDP.getDistributionPoints()) {
            GeneralName[] names;
            DistributionPointName dpn = dp.getDistributionPoint();
            if (dpn == null || dpn.getType() != 0) continue;
            for (GeneralName gn : names = GeneralNames.getInstance((Object)dpn.getName()).getNames()) {
                if (gn.getTagNo() != 6) continue;
                String url = DERIA5String.getInstance((Object)gn.getName()).getString();
                distributionPointUrls.add(url);
            }
        }
        return distributionPointUrls;
    }

    public static void check(X509Certificate[] certs, X509CRL crl, KeycloakSession session) throws GeneralSecurityException {
        if (certs.length < 2) {
            throw new GeneralSecurityException("Not possible to verify signature on CRL. X509 certificate doesn't have CA chain available on it");
        }
        X500Principal crlIssuerPrincipal = crl.getIssuerX500Principal();
        Certificate crlSignatureCertificate = null;
        for (int i = 1; i < certs.length; ++i) {
            X509Certificate currentCACert = certs[i];
            if (!crlIssuerPrincipal.equals(currentCACert.getSubjectX500Principal())) continue;
            crlSignatureCertificate = currentCACert;
            log.tracef("Found certificate used to sign CRL in the CA chain of the certificate. CRL issuer: %s", (Object)crlIssuerPrincipal);
            break;
        }
        if (crlSignatureCertificate == null) {
            log.tracef("Not found CRL issuer '%s' in the CA chain of the certificate. Fallback to lookup CRL issuer in the truststore", (Object)crlIssuerPrincipal);
            crlSignatureCertificate = CRLUtils.findCRLSignatureCertificateInTruststore(session, certs, crlIssuerPrincipal);
        }
        crl.verify(crlSignatureCertificate.getPublicKey());
        if (crl.isRevoked(certs[0])) {
            String message = String.format("Certificate has been revoked, certificate's subject: %s", certs[0].getSubjectDN().getName());
            log.debug((Object)message);
            throw new GeneralSecurityException(message);
        }
    }

    private static X509Certificate findCRLSignatureCertificateInTruststore(KeycloakSession session, X509Certificate[] certs, X500Principal crlIssuerPrincipal) throws GeneralSecurityException {
        X500Principal currentCRLAnchorPrincipal;
        X509Certificate crlSignatureCertificate;
        block4: {
            TruststoreProvider truststoreProvider = (TruststoreProvider)session.getProvider(TruststoreProvider.class);
            if (truststoreProvider == null || truststoreProvider.getTruststore() == null) {
                throw new GeneralSecurityException("Truststore not available");
            }
            Map rootCerts = truststoreProvider.getRootCertificates();
            Map intermediateCerts = truststoreProvider.getIntermediateCertificates();
            crlSignatureCertificate = (X509Certificate)intermediateCerts.get(crlIssuerPrincipal);
            if (crlSignatureCertificate == null) {
                crlSignatureCertificate = (X509Certificate)rootCerts.get(crlIssuerPrincipal);
            }
            if (crlSignatureCertificate == null) {
                throw new GeneralSecurityException("Not available certificate for CRL issuer '" + crlIssuerPrincipal + "' in the truststore, nor in the CA chain");
            }
            log.tracef("Found CRL issuer certificate with subject '%s' in the truststore. Verifying trust anchor", (Object)crlIssuerPrincipal);
            Set certificateCAPrincipals = Arrays.asList(certs).stream().map(X509Certificate::getSubjectX500Principal).collect(Collectors.toSet());
            certificateCAPrincipals.remove(certs[0].getSubjectX500Principal());
            X509Certificate currentCRLAnchorCertificate = crlSignatureCertificate;
            currentCRLAnchorPrincipal = crlIssuerPrincipal;
            do {
                if (certificateCAPrincipals.contains(currentCRLAnchorPrincipal)) break block4;
                currentCRLAnchorPrincipal = currentCRLAnchorCertificate.getIssuerX500Principal();
                if ((currentCRLAnchorCertificate = (X509Certificate)intermediateCerts.get(currentCRLAnchorPrincipal)) != null) continue;
                currentCRLAnchorCertificate = (X509Certificate)rootCerts.get(currentCRLAnchorPrincipal);
            } while (currentCRLAnchorCertificate != null);
            throw new GeneralSecurityException("Certificate for CRL issuer '" + crlIssuerPrincipal + "' available in the truststore, but doesn't have trust anchors with the CA chain.");
        }
        log.tracef("Found trust anchor of the CRL issuer '%s' in the CA chain. Anchor is '%s'", (Object)crlIssuerPrincipal, (Object)currentCRLAnchorPrincipal);
        return crlSignatureCertificate;
    }

    static {
        BouncyIntegration.init();
    }
}

