/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage.ldap.idm.store.ldap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.TreeSet;
import javax.naming.AuthenticationException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchResult;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Base64;
import org.keycloak.models.ModelException;
import org.keycloak.storage.ldap.LDAPConfig;
import org.keycloak.storage.ldap.idm.model.LDAPDn;
import org.keycloak.storage.ldap.idm.model.LDAPObject;
import org.keycloak.storage.ldap.idm.query.Condition;
import org.keycloak.storage.ldap.idm.query.EscapeStrategy;
import org.keycloak.storage.ldap.idm.query.internal.EqualCondition;
import org.keycloak.storage.ldap.idm.query.internal.LDAPQuery;
import org.keycloak.storage.ldap.idm.store.IdentityStore;
import org.keycloak.storage.ldap.idm.store.ldap.LDAPOperationManager;
import org.keycloak.storage.ldap.mappers.LDAPOperationDecorator;

public class LDAPIdentityStore
implements IdentityStore {
    private static final Logger logger = Logger.getLogger(LDAPIdentityStore.class);
    private final LDAPConfig config;
    private final LDAPOperationManager operationManager;

    public LDAPIdentityStore(LDAPConfig config) {
        this.config = config;
        try {
            this.operationManager = new LDAPOperationManager(config);
        }
        catch (NamingException e) {
            throw new ModelException("Couldn't init operation manager", (Throwable)e);
        }
    }

    @Override
    public LDAPConfig getConfig() {
        return this.config;
    }

    @Override
    public void add(LDAPObject ldapObject) {
        if (ldapObject.getUuid() != null) {
            throw new ModelException("Can't add object with already assigned uuid");
        }
        String entryDN = ldapObject.getDn().toString();
        BasicAttributes ldapAttributes = this.extractAttributes(ldapObject, true);
        this.operationManager.createSubContext(entryDN, ldapAttributes);
        ldapObject.setUuid(this.getEntryIdentifier(ldapObject));
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and dn [%s] successfully added to LDAP store.", (Object)ldapObject.getUuid(), (Object)entryDN);
        }
    }

    @Override
    public void update(LDAPObject ldapObject) {
        this.checkRename(ldapObject);
        BasicAttributes updatedAttributes = this.extractAttributes(ldapObject, false);
        NamingEnumeration<Attribute> attributes = updatedAttributes.getAll();
        String entryDn = ldapObject.getDn().toString();
        this.operationManager.modifyAttributes(entryDn, attributes);
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and DN [%s] successfully updated to LDAP store.", (Object)ldapObject.getUuid(), (Object)entryDn);
        }
    }

    protected void checkRename(LDAPObject ldapObject) {
        String rdnAttrName = ldapObject.getRdnAttributeName();
        if (ldapObject.getReadOnlyAttributeNames().contains(rdnAttrName.toLowerCase())) {
            return;
        }
        String rdnAttrVal = ldapObject.getAttributeAsString(rdnAttrName);
        if (rdnAttrVal == null) {
            return;
        }
        String oldRdnAttrVal = ldapObject.getDn().getFirstRdnAttrValue();
        if (!oldRdnAttrVal.equals(rdnAttrVal)) {
            LDAPDn newLdapDn = ldapObject.getDn().getParentDn();
            newLdapDn.addFirst(rdnAttrName, rdnAttrVal);
            String oldDn = ldapObject.getDn().toString();
            String newDn = newLdapDn.toString();
            if (logger.isDebugEnabled()) {
                logger.debugf("Renaming LDAP Object. Old DN: [%s], New DN: [%s]", (Object)oldDn, (Object)newDn);
            }
            newDn = this.operationManager.renameEntry(oldDn, newDn, true);
            ldapObject.setDn(LDAPDn.fromString(newDn));
        }
    }

    @Override
    public void remove(LDAPObject ldapObject) {
        this.operationManager.removeEntry(ldapObject.getDn().toString());
        if (logger.isDebugEnabled()) {
            logger.debugf("Type with identifier [%s] and DN [%s] successfully removed from LDAP store.", (Object)ldapObject.getUuid(), (Object)ldapObject.getDn().toString());
        }
    }

    @Override
    public List<LDAPObject> fetchQueryResults(LDAPQuery identityQuery) {
        if (identityQuery.getSorting() != null && !identityQuery.getSorting().isEmpty()) {
            throw new ModelException("LDAP Identity Store does not yet support sorted queries.");
        }
        ArrayList<LDAPObject> results = new ArrayList<LDAPObject>();
        try {
            String baseDN = identityQuery.getSearchDn();
            for (Condition condition : identityQuery.getConditions()) {
                EqualCondition equalCondition;
                String uuidAttrName = this.getConfig().getUuidLDAPAttributeName();
                if (!(condition instanceof EqualCondition) || !(equalCondition = (EqualCondition)condition).getParameterName().equalsIgnoreCase(uuidAttrName)) continue;
                SearchResult search = this.operationManager.lookupById(baseDN, equalCondition.getValue().toString(), identityQuery.getReturningLdapAttributes());
                if (search != null) {
                    results.add(this.populateAttributedType(search, identityQuery));
                }
                return results;
            }
            StringBuilder filter = this.createIdentityTypeSearchFilter(identityQuery);
            List<SearchResult> search = this.getConfig().isPagination() && identityQuery.getLimit() > 0 ? this.operationManager.searchPaginated(baseDN, filter.toString(), identityQuery) : this.operationManager.search(baseDN, filter.toString(), identityQuery.getReturningLdapAttributes(), identityQuery.getSearchScope());
            for (SearchResult result : search) {
                if (result.getNameInNamespace().equalsIgnoreCase(baseDN)) continue;
                results.add(this.populateAttributedType(result, identityQuery));
            }
        }
        catch (Exception e) {
            throw new ModelException("Querying of LDAP failed " + identityQuery, (Throwable)e);
        }
        return results;
    }

    @Override
    public int countQueryResults(LDAPQuery identityQuery) {
        int limit = identityQuery.getLimit();
        int offset = identityQuery.getOffset();
        identityQuery.setLimit(0);
        identityQuery.setOffset(0);
        int resultCount = identityQuery.getResultList().size();
        identityQuery.setLimit(limit);
        identityQuery.setOffset(offset);
        return resultCount;
    }

    @Override
    public void validatePassword(LDAPObject user, String password) throws AuthenticationException {
        String userDN = user.getDn().toString();
        if (logger.isTraceEnabled()) {
            logger.tracef("Using DN [%s] for authentication of user", (Object)userDN);
        }
        this.operationManager.authenticate(userDN, password);
    }

    @Override
    public void updatePassword(LDAPObject user, String password, LDAPOperationDecorator passwordUpdateDecorator) {
        String userDN = user.getDn().toString();
        if (logger.isDebugEnabled()) {
            logger.debugf("Using DN [%s] for updating LDAP password of user", (Object)userDN);
        }
        if (this.getConfig().isActiveDirectory()) {
            this.updateADPassword(userDN, password, passwordUpdateDecorator);
        } else {
            ModificationItem[] mods = new ModificationItem[1];
            try {
                BasicAttribute mod0 = new BasicAttribute("userpassword", password);
                mods[0] = new ModificationItem(2, mod0);
                this.operationManager.modifyAttributes(userDN, mods, passwordUpdateDecorator);
            }
            catch (ModelException me) {
                throw me;
            }
            catch (Exception e) {
                throw new ModelException("Error updating password.", (Throwable)e);
            }
        }
    }

    private void updateADPassword(String userDN, String password, LDAPOperationDecorator passwordUpdateDecorator) {
        try {
            String newQuotedPassword = "\"" + password + "\"";
            byte[] newUnicodePassword = newQuotedPassword.getBytes("UTF-16LE");
            BasicAttribute unicodePwd = new BasicAttribute("unicodePwd", newUnicodePassword);
            ArrayList<ModificationItem> modItems = new ArrayList<ModificationItem>();
            modItems.add(new ModificationItem(2, unicodePwd));
            this.operationManager.modifyAttributes(userDN, modItems.toArray(new ModificationItem[0]), passwordUpdateDecorator);
        }
        catch (ModelException me) {
            throw me;
        }
        catch (Exception e) {
            throw new ModelException((Throwable)e);
        }
    }

    protected StringBuilder createIdentityTypeSearchFilter(LDAPQuery identityQuery) {
        StringBuilder filter = new StringBuilder();
        for (Condition condition : identityQuery.getConditions()) {
            condition.applyCondition(filter);
        }
        filter.insert(0, "(&");
        filter.append((CharSequence)this.getObjectClassesFilter(identityQuery.getObjectClasses()));
        filter.append(")");
        if (logger.isTraceEnabled()) {
            logger.tracef("Using filter for LDAP search: %s . Searching in DN: %s", (Object)filter, (Object)identityQuery.getSearchDn());
        }
        return filter;
    }

    private StringBuilder getObjectClassesFilter(Collection<String> objectClasses) {
        StringBuilder builder = new StringBuilder();
        if (!objectClasses.isEmpty()) {
            for (String objectClass : objectClasses) {
                builder.append("(").append("objectclass").append("=").append(objectClass).append(")");
            }
        } else {
            builder.append("(").append("objectclass").append("=").append("*").append(")");
        }
        return builder;
    }

    private LDAPObject populateAttributedType(SearchResult searchResult, LDAPQuery ldapQuery) {
        Set<String> readOnlyAttrNames = ldapQuery.getReturningReadOnlyLdapAttributes();
        TreeSet<String> lowerCasedAttrNames = new TreeSet<String>();
        for (String attrName : ldapQuery.getReturningLdapAttributes()) {
            lowerCasedAttrNames.add(attrName.toLowerCase());
        }
        try {
            String entryDN = searchResult.getNameInNamespace();
            Attributes attributes = searchResult.getAttributes();
            LDAPObject ldapObject = new LDAPObject();
            LDAPDn dn = LDAPDn.fromString(entryDN);
            ldapObject.setDn(dn);
            ldapObject.setRdnAttributeName(dn.getFirstRdnAttrName());
            NamingEnumeration<? extends Attribute> ldapAttributes = attributes.getAll();
            while (ldapAttributes.hasMore()) {
                Attribute ldapAttribute = ldapAttributes.next();
                try {
                    ldapAttribute.get();
                }
                catch (NoSuchElementException nsee) {
                    continue;
                }
                String ldapAttributeName = ldapAttribute.getID();
                if (ldapAttributeName.equalsIgnoreCase(this.getConfig().getUuidLDAPAttributeName())) {
                    Object uuidValue = ldapAttribute.get();
                    ldapObject.setUuid(this.operationManager.decodeEntryUUID(uuidValue));
                }
                if (ldapAttributeName.equalsIgnoreCase(this.getConfig().getUuidLDAPAttributeName()) && !lowerCasedAttrNames.contains(ldapAttributeName.toLowerCase())) continue;
                LinkedHashSet<String> attrValues = new LinkedHashSet<String>();
                NamingEnumeration<?> enumm = ldapAttribute.getAll();
                while (enumm.hasMoreElements()) {
                    String attrVal;
                    Object val = enumm.next();
                    if (val instanceof byte[]) {
                        attrVal = Base64.encodeBytes((byte[])((byte[])val));
                        attrValues.add(attrVal);
                        continue;
                    }
                    attrVal = val.toString().trim();
                    attrValues.add(attrVal);
                }
                if (ldapAttributeName.equalsIgnoreCase("objectclass")) {
                    ldapObject.setObjectClasses(attrValues);
                    continue;
                }
                ldapObject.setAttribute(ldapAttributeName, attrValues);
                if (!readOnlyAttrNames.contains(ldapAttributeName.toLowerCase())) continue;
                ldapObject.addReadOnlyAttributeName(ldapAttributeName);
            }
            if (logger.isTraceEnabled()) {
                logger.tracef("Found ldap object and populated with the attributes. LDAP Object: %s", (Object)ldapObject.toString());
            }
            return ldapObject;
        }
        catch (Exception e) {
            throw new ModelException("Could not populate attribute type " + searchResult.getNameInNamespace() + ".", (Throwable)e);
        }
    }

    protected BasicAttributes extractAttributes(LDAPObject ldapObject, boolean isCreate) {
        BasicAttributes entryAttributes = new BasicAttributes();
        for (Map.Entry<String, Set<String>> attrEntry : ldapObject.getAttributes().entrySet()) {
            String attrName = attrEntry.getKey();
            Set<String> attrValue = attrEntry.getValue();
            if (ldapObject.getReadOnlyAttributeNames().contains(attrName.toLowerCase()) || !isCreate && ldapObject.getRdnAttributeName().equalsIgnoreCase(attrName)) continue;
            if (attrValue == null) {
                logger.warnf("Attribute '%s' is null on LDAP object '%s' . Using empty value to be saved to LDAP", (Object)attrName, (Object)ldapObject.getDn().toString());
                attrValue = Collections.emptySet();
            }
            if (isCreate && attrValue.isEmpty()) continue;
            BasicAttribute attr = new BasicAttribute(attrName);
            for (String val : attrValue) {
                if (val == null || val.toString().trim().length() == 0) {
                    val = " ";
                }
                if (this.getConfig().getBinaryAttributeNames().contains(attrName)) {
                    try {
                        byte[] bytes = Base64.decode((String)val);
                        attr.add(bytes);
                    }
                    catch (IOException ioe) {
                        logger.warnf("Wasn't able to Base64 decode the attribute value. Ignoring attribute update. LDAP DN: %s, Attribute: %s, Attribute value: %s" + ldapObject.getDn(), (Object)attrName, attrValue);
                    }
                    continue;
                }
                attr.add(val);
            }
            entryAttributes.put(attr);
        }
        if (isCreate) {
            BasicAttribute objectClassAttribute = new BasicAttribute("objectclass");
            for (String objectClassValue : ldapObject.getObjectClasses()) {
                objectClassAttribute.add(objectClassValue);
                if (!objectClassValue.equalsIgnoreCase("groupOfNames") && !objectClassValue.equalsIgnoreCase("groupOfEntries") && !objectClassValue.equalsIgnoreCase("groupOfUniqueNames")) continue;
                entryAttributes.put("member", "cn=empty-membership-placeholder");
            }
            entryAttributes.put(objectClassAttribute);
        }
        return entryAttributes;
    }

    protected String getEntryIdentifier(LDAPObject ldapObject) {
        try {
            String uuidAttrName = this.getConfig().getUuidLDAPAttributeName();
            String rdn = ldapObject.getDn().getFirstRdn();
            String filter = "(" + EscapeStrategy.DEFAULT.escape(rdn) + ")";
            List<SearchResult> search = this.operationManager.search(ldapObject.getDn().toString(), filter, Arrays.asList(uuidAttrName), 0);
            Attribute id = search.get(0).getAttributes().get(this.getConfig().getUuidLDAPAttributeName());
            if (id == null) {
                throw new ModelException("Could not retrieve identifier for entry [" + ldapObject.getDn().toString() + "].");
            }
            return this.operationManager.decodeEntryUUID(id.get());
        }
        catch (NamingException ne) {
            throw new ModelException("Could not retrieve identifier for entry [" + ldapObject.getDn().toString() + "].");
        }
    }
}

