/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.jwt;

import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.ext.auth.JWTOptions;
import io.vertx.ext.jwt.Crypto;
import io.vertx.ext.jwt.CryptoMac;
import io.vertx.ext.jwt.CryptoNone;
import io.vertx.ext.jwt.CryptoSignature;
import io.vertx.ext.jwt.JWK;
import io.vertx.ext.jwt.NoSuchKeyIdException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import javax.crypto.Mac;

public final class JWT {
    private static final Random RND = new Random();
    private static final Map<String, String> ALGORITHM_ALIAS = new HashMap<String, String>(){
        {
            this.put("HS256", "HMacSHA256");
            this.put("HS384", "HMacSHA384");
            this.put("HS512", "HMacSHA512");
            this.put("RS256", "SHA256withRSA");
            this.put("RS384", "SHA384withRSA");
            this.put("RS512", "SHA512withRSA");
            this.put("ES256", "SHA256withECDSA");
            this.put("ES384", "SHA384withECDSA");
            this.put("ES512", "SHA512withECDSA");
        }
    };
    private static final Charset UTF8 = StandardCharsets.UTF_8;
    private static final Logger log = LoggerFactory.getLogger(JWT.class);
    private static final Base64.Encoder encoder = Base64.getUrlEncoder().withoutPadding();
    private static final Base64.Decoder decoder = Base64.getUrlDecoder();
    private final Map<String, List<Crypto>> cryptoMap = new ConcurrentHashMap<String, List<Crypto>>();

    public JWT() {
        this.cryptoMap.put("none", Collections.singletonList(new CryptoNone()));
    }

    @Deprecated
    public JWT(KeyStore keyStore, char[] keyStorePassword) {
        this();
        for (String alg : Arrays.asList("HS256", "HS384", "HS512")) {
            try {
                Mac mac = this.getMac(keyStore, keyStorePassword, alg);
                if (mac != null) {
                    List l = this.cryptoMap.computeIfAbsent(alg, k -> new ArrayList());
                    l.add(new CryptoMac(mac));
                    continue;
                }
                log.info(alg + " not available");
            }
            catch (RuntimeException e) {
                log.warn((Object)(alg + " not supported"), e);
            }
        }
        for (String alg : Arrays.asList("RS256", "RS384", "RS512", "ES256", "ES384", "ES512")) {
            try {
                X509Certificate certificate = this.getCertificate(keyStore, alg);
                PrivateKey privateKey = this.getPrivateKey(keyStore, keyStorePassword, alg);
                if (certificate != null && privateKey != null) {
                    List l = this.cryptoMap.computeIfAbsent(alg, k -> new ArrayList());
                    l.add(new CryptoSignature(ALGORITHM_ALIAS.get(alg), certificate, privateKey));
                    continue;
                }
                log.info(alg + " not available");
            }
            catch (RuntimeException e) {
                e.printStackTrace();
                log.warn(alg + " not supported");
            }
        }
    }

    @Deprecated
    public JWT(String key, boolean keyPrivate) {
        this();
        if (keyPrivate) {
            this.addSecretKey("RS256", key);
        } else {
            this.addPublicKey("RS256", key);
        }
    }

    public JWT addJWK(JWK jwk) {
        List current = this.cryptoMap.computeIfAbsent(jwk.getAlgorithm(), k -> new ArrayList());
        boolean replaced = false;
        for (int i = 0; i < current.size(); ++i) {
            if (!((Crypto)current.get(i)).getLabel().equals(jwk.getLabel())) continue;
            current.set(i, jwk);
            replaced = true;
        }
        if (!replaced) {
            current.add(jwk);
        }
        return this;
    }

    @Deprecated
    public JWT addPublicKey(String algorithm, String key) {
        return this.addJWK(new JWK(algorithm, key, null));
    }

    @Deprecated
    public JWT addKeyPair(String algorithm, String publicKey, String privateKey) {
        return this.addJWK(new JWK(algorithm, publicKey, privateKey));
    }

    @Deprecated
    public JWT addSecretKey(String algorithm, String key) {
        return this.addJWK(new JWK(algorithm, null, key));
    }

    @Deprecated
    public JWT addCertificate(String algorithm, String cert) {
        return this.addJWK(new JWK(algorithm, true, cert, null));
    }

    @Deprecated
    public JWT addSecret(String algorithm, String key) {
        return this.addJWK(new JWK(algorithm, key));
    }

    private Mac getMac(KeyStore keyStore, char[] keyStorePassword, String alias) {
        try {
            Key secretKey = keyStore.getKey(alias, keyStorePassword);
            if (secretKey == null) {
                return null;
            }
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            return mac;
        }
        catch (InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new RuntimeException(e);
        }
    }

    private X509Certificate getCertificate(KeyStore keyStore, String alias) {
        try {
            return (X509Certificate)keyStore.getCertificate(alias);
        }
        catch (KeyStoreException e) {
            throw new RuntimeException(e);
        }
    }

    private PrivateKey getPrivateKey(KeyStore keyStore, char[] keyStorePassword, String alias) {
        try {
            return (PrivateKey)keyStore.getKey(alias, keyStorePassword);
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            throw new RuntimeException(e);
        }
    }

    public JsonObject decode(String token) {
        String signatureSeg;
        String[] segments = token.split("\\.");
        if (segments.length != (this.isUnsecure() ? 2 : 3)) {
            throw new RuntimeException("Not enough or too many segments");
        }
        String headerSeg = segments[0];
        String payloadSeg = segments[1];
        String string = signatureSeg = this.isUnsecure() ? null : segments[2];
        if ("".equals(signatureSeg)) {
            throw new RuntimeException("Signature is required");
        }
        JsonObject header = new JsonObject(new String(JWT.base64urlDecode(headerSeg), UTF8));
        JsonObject payload = new JsonObject(new String(JWT.base64urlDecode(payloadSeg), UTF8));
        String alg = header.getString("alg");
        List<Crypto> cryptos = this.cryptoMap.get(alg);
        if (cryptos == null || cryptos.size() == 0) {
            throw new NoSuchKeyIdException(alg);
        }
        if (!this.isUnsecure() && "none".equals(alg)) {
            throw new RuntimeException("Algorithm \"none\" not allowed");
        }
        if (!this.isUnsecure()) {
            byte[] payloadInput = JWT.base64urlDecode(signatureSeg);
            byte[] signingInput = (headerSeg + "." + payloadSeg).getBytes(UTF8);
            String kid = header.getString("kid");
            boolean hasKey = false;
            for (Crypto c : cryptos) {
                if (kid != null && c.getId() != null && !kid.equals(c.getId())) continue;
                hasKey = true;
                if (!c.verify(payloadInput, signingInput)) continue;
                return payload;
            }
            if (hasKey) {
                throw new RuntimeException("Signature verification failed");
            }
            throw new NoSuchKeyIdException(alg, kid);
        }
        return payload;
    }

    public boolean isExpired(JsonObject jwt, JWTOptions options) {
        Long nbf;
        Long iat;
        if (jwt == null) {
            return false;
        }
        long now = System.currentTimeMillis() / 1000L;
        if (jwt.containsKey("exp") && !options.isIgnoreExpiration() && now - (long)options.getLeeway() >= jwt.getLong("exp")) {
            throw new RuntimeException("Expired JWT token: exp <= now");
        }
        if (jwt.containsKey("iat") && (iat = jwt.getLong("iat")) > now + (long)options.getLeeway()) {
            throw new RuntimeException("Invalid JWT token: iat > now");
        }
        if (jwt.containsKey("nbf") && (nbf = jwt.getLong("nbf")) > now + (long)options.getLeeway()) {
            throw new RuntimeException("Invalid JWT token: nbf > now");
        }
        return false;
    }

    public String sign(JsonObject payload, JWTOptions options) {
        String algorithm = options.getAlgorithm();
        List<Crypto> cryptos = this.cryptoMap.get(algorithm);
        if (cryptos == null || cryptos.size() == 0) {
            throw new RuntimeException("Algorithm not supported");
        }
        JsonObject header = new JsonObject().mergeIn(options.getHeader()).put("typ", "JWT").put("alg", algorithm);
        long timestamp = System.currentTimeMillis() / 1000L;
        if (!options.isNoTimestamp()) {
            payload.put("iat", payload.getValue("iat", timestamp));
        }
        if (options.getExpiresInSeconds() > 0) {
            payload.put("exp", timestamp + (long)options.getExpiresInSeconds());
        }
        if (options.getAudience() != null && options.getAudience().size() >= 1) {
            if (options.getAudience().size() > 1) {
                payload.put("aud", new JsonArray(options.getAudience()));
            } else {
                payload.put("aud", options.getAudience().get(0));
            }
        }
        if (options.getIssuer() != null) {
            payload.put("iss", options.getIssuer());
        }
        if (options.getSubject() != null) {
            payload.put("sub", options.getSubject());
        }
        String headerSegment = JWT.base64urlEncode(header.encode());
        String payloadSegment = JWT.base64urlEncode(payload.encode());
        String signingInput = headerSegment + "." + payloadSegment;
        String signSegment = JWT.base64urlEncode(cryptos.get(RND.nextInt(cryptos.size())).sign(signingInput.getBytes(UTF8)));
        return headerSegment + "." + payloadSegment + "." + signSegment;
    }

    private static byte[] base64urlDecode(String str) {
        return decoder.decode(str.getBytes(UTF8));
    }

    private static String base64urlEncode(String str) {
        return JWT.base64urlEncode(str.getBytes(UTF8));
    }

    private static String base64urlEncode(byte[] bytes) {
        return encoder.encodeToString(bytes);
    }

    public boolean isUnsecure() {
        return this.cryptoMap.size() == 1;
    }

    public Collection<String> availableAlgorithms() {
        return this.cryptoMap.keySet();
    }
}

