/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.jwt.auth.principal;

import io.smallrye.jwt.auth.principal.DecryptionKeyLocationResolver;
import io.smallrye.jwt.auth.principal.JWTAuthContextInfo;
import io.smallrye.jwt.auth.principal.KeyLocationResolver;
import io.smallrye.jwt.auth.principal.ParseException;
import io.smallrye.jwt.auth.principal.PrincipalLogging;
import io.smallrye.jwt.auth.principal.PrincipalMessages;
import io.smallrye.jwt.auth.principal.PrincipalUtils;
import io.smallrye.jwt.auth.principal.X509KeyLocationResolver;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;
import org.eclipse.microprofile.jwt.Claims;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwe.JsonWebEncryption;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.NumericDate;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.JwtContext;
import org.jose4j.keys.resolvers.DecryptionKeyResolver;
import org.jose4j.keys.resolvers.VerificationKeyResolver;
import org.jose4j.lang.JoseException;
import org.jose4j.lang.UnresolvableKeyException;

public class DefaultJWTTokenParser {
    private volatile VerificationKeyResolver keyResolver;
    private volatile DecryptionKeyResolver decryptionKeyResolver;

    public JwtContext parse(String token, JWTAuthContextInfo authContextInfo) throws ParseException {
        String tokenSequence = token;
        ProtectionLevel level = this.getProtectionLevel(authContextInfo);
        if (level == ProtectionLevel.SIGN_ENCRYPT) {
            tokenSequence = this.decryptSignedToken(tokenSequence, authContextInfo);
            level = ProtectionLevel.SIGN;
        }
        return this.parseClaims(tokenSequence, authContextInfo, level);
    }

    private String decryptSignedToken(String token, JWTAuthContextInfo authContextInfo) throws ParseException {
        try {
            JsonWebEncryption jwe = new JsonWebEncryption();
            jwe.setAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, authContextInfo.getKeyEncryptionAlgorithm().getAlgorithm()));
            if (authContextInfo.getPrivateDecryptionKey() != null) {
                jwe.setKey(authContextInfo.getPrivateDecryptionKey());
            } else if (authContextInfo.getSecretDecryptionKey() != null) {
                jwe.setKey(authContextInfo.getSecretDecryptionKey());
            } else {
                jwe.setKey(this.getDecryptionKeyResolver(authContextInfo).resolveKey(jwe, null));
            }
            jwe.setCompactSerialization(token);
            if (!"JWT".equals(jwe.getContentTypeHeaderValue())) {
                PrincipalLogging.log.encryptedTokenMissingContentType();
                throw PrincipalMessages.msg.encryptedTokenMissingContentType();
            }
            return jwe.getPlaintextString();
        }
        catch (UnresolvableKeyException e) {
            PrincipalLogging.log.decryptionKeyUnresolvable();
            throw PrincipalMessages.msg.decryptionKeyUnresolvable(e);
        }
        catch (JoseException e) {
            PrincipalLogging.log.encryptedTokenSequenceInvalid();
            throw PrincipalMessages.msg.encryptedTokenSequenceInvalid(e);
        }
    }

    private JwtContext parseClaims(String token, JWTAuthContextInfo authContextInfo, ProtectionLevel level) throws ParseException {
        try {
            boolean issuedAtRequired;
            JwtConsumerBuilder builder = new JwtConsumerBuilder();
            if (level == ProtectionLevel.SIGN) {
                if (authContextInfo.getPublicVerificationKey() != null) {
                    builder.setVerificationKey(authContextInfo.getPublicVerificationKey());
                } else if (authContextInfo.getSecretVerificationKey() != null) {
                    builder.setVerificationKey(authContextInfo.getSecretVerificationKey());
                } else {
                    builder.setVerificationKeyResolver(this.getVerificationKeyResolver(authContextInfo));
                }
                builder.setJwsAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, authContextInfo.getSignatureAlgorithm().getAlgorithm()));
            } else {
                builder.setEnableRequireEncryption();
                builder.setDisableRequireSignature();
                if (authContextInfo.getPrivateDecryptionKey() != null) {
                    builder.setDecryptionKey(authContextInfo.getPrivateDecryptionKey());
                } else if (authContextInfo.getSecretDecryptionKey() != null) {
                    builder.setDecryptionKey(authContextInfo.getSecretDecryptionKey());
                } else {
                    builder.setDecryptionKeyResolver(this.getDecryptionKeyResolver(authContextInfo));
                }
                builder.setJweAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, authContextInfo.getKeyEncryptionAlgorithm().getAlgorithm()));
            }
            builder.setRequireExpirationTime();
            boolean bl = issuedAtRequired = authContextInfo.getMaxTimeToLiveSecs() == null || authContextInfo.getMaxTimeToLiveSecs() > 0L;
            if (issuedAtRequired) {
                builder.setRequireIssuedAt();
            }
            if (authContextInfo.getIssuedBy() != null) {
                builder.setExpectedIssuer(authContextInfo.getIssuedBy());
            }
            if (authContextInfo.getExpGracePeriodSecs() > 0) {
                builder.setAllowedClockSkewInSeconds(authContextInfo.getExpGracePeriodSecs());
            }
            this.setExpectedAudience(builder, authContextInfo);
            if (authContextInfo.isRelaxVerificationKeyValidation()) {
                builder.setRelaxVerificationKeyValidation();
            }
            JwtConsumer jwtConsumer = builder.build();
            JwtContext jwtContext = jwtConsumer.process(token);
            JwtClaims claimsSet = jwtContext.getJwtClaims();
            if (issuedAtRequired) {
                this.verifyIatAndExpAndTimeToLive(authContextInfo, claimsSet);
            }
            this.verifyRequiredClaims(authContextInfo, jwtContext);
            PrincipalUtils.setClaims(claimsSet, token, authContextInfo);
            if (authContextInfo.isRequireNamedPrincipal()) {
                this.checkNameClaims(jwtContext);
            }
            return jwtContext;
        }
        catch (InvalidJwtException e) {
            if (e.getCause() instanceof UnresolvableKeyException) {
                PrincipalLogging.log.verificationKeyUnresolvable();
                throw PrincipalMessages.msg.failedToVerifyToken(e.getCause());
            }
            PrincipalLogging.log.tokenInvalid();
            throw PrincipalMessages.msg.failedToVerifyToken(e);
        }
        catch (UnresolvableKeyException e) {
            PrincipalLogging.log.verificationKeyUnresolvable();
            throw PrincipalMessages.msg.failedToVerifyToken(e);
        }
    }

    void setExpectedAudience(JwtConsumerBuilder builder, JWTAuthContextInfo authContextInfo) {
        Set<String> expectedAudience = authContextInfo.getExpectedAudience();
        if (expectedAudience != null) {
            builder.setExpectedAudience(expectedAudience.toArray(new String[0]));
        } else {
            builder.setSkipDefaultAudienceValidation();
        }
    }

    private void checkNameClaims(JwtContext jwtContext) throws InvalidJwtException {
        boolean hasPrincipalClaim;
        JwtClaims claimsSet = jwtContext.getJwtClaims();
        boolean bl = hasPrincipalClaim = claimsSet.getClaimValue(Claims.sub.name()) != null || claimsSet.getClaimValue(Claims.upn.name()) != null || claimsSet.getClaimValue(Claims.preferred_username.name()) != null;
        if (!hasPrincipalClaim) {
            throw PrincipalMessages.msg.claimNotFound(s -> new InvalidJwtException((String)s, Collections.emptyList(), jwtContext));
        }
    }

    private void verifyIatAndExpAndTimeToLive(JWTAuthContextInfo authContextInfo, JwtClaims claimsSet) throws ParseException {
        NumericDate exp;
        NumericDate iat;
        try {
            iat = claimsSet.getIssuedAt();
            exp = claimsSet.getExpirationTime();
        }
        catch (Exception ex) {
            throw PrincipalMessages.msg.invalidIatExp();
        }
        if (iat.getValue() > exp.getValue()) {
            throw PrincipalMessages.msg.failedToVerifyIatExp(exp, iat);
        }
        Long maxTimeToLiveSecs = authContextInfo.getMaxTimeToLiveSecs();
        if (maxTimeToLiveSecs != null) {
            if (exp.getValue() - iat.getValue() > maxTimeToLiveSecs) {
                throw PrincipalMessages.msg.expExceeded(exp, maxTimeToLiveSecs, iat);
            }
        } else {
            PrincipalLogging.log.noMaxTTLSpecified();
        }
    }

    private void verifyRequiredClaims(JWTAuthContextInfo authContextInfo, JwtContext jwtContext) throws InvalidJwtException {
        Set<String> requiredClaims = authContextInfo.getRequiredClaims();
        if (requiredClaims != null && !jwtContext.getJwtClaims().getClaimsMap().keySet().containsAll(requiredClaims)) {
            if (PrincipalLogging.log.isDebugEnabled()) {
                String missingClaims = requiredClaims.stream().filter(claim -> !jwtContext.getJwtClaims().getClaimsMap().containsKey(claim)).collect(Collectors.joining(","));
                PrincipalLogging.log.missingClaims(missingClaims);
            }
            throw PrincipalMessages.msg.missingClaims(s -> new InvalidJwtException((String)s, Collections.emptyList(), jwtContext));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected VerificationKeyResolver getVerificationKeyResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        if (this.keyResolver == null) {
            DefaultJWTTokenParser defaultJWTTokenParser = this;
            synchronized (defaultJWTTokenParser) {
                if (this.keyResolver == null) {
                    this.keyResolver = authContextInfo.isVerifyCertificateThumbprint() ? new X509KeyLocationResolver(authContextInfo) : new KeyLocationResolver(authContextInfo);
                }
            }
        }
        return this.keyResolver;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected DecryptionKeyResolver getDecryptionKeyResolver(JWTAuthContextInfo authContextInfo) throws UnresolvableKeyException {
        if (this.decryptionKeyResolver == null) {
            DefaultJWTTokenParser defaultJWTTokenParser = this;
            synchronized (defaultJWTTokenParser) {
                if (this.decryptionKeyResolver == null) {
                    this.decryptionKeyResolver = new DecryptionKeyLocationResolver(authContextInfo);
                }
            }
        }
        return this.decryptionKeyResolver;
    }

    protected ProtectionLevel getProtectionLevel(JWTAuthContextInfo authContextInfo) {
        if (authContextInfo.getDecryptionKeyLocation() != null || authContextInfo.getDecryptionKeyContent() != null || authContextInfo.getPrivateDecryptionKey() != null || authContextInfo.getSecretDecryptionKey() != null) {
            boolean sign = authContextInfo.getPublicVerificationKey() != null || authContextInfo.getSecretVerificationKey() != null || authContextInfo.getPublicKeyContent() != null || authContextInfo.getPublicKeyLocation() != null;
            return sign ? ProtectionLevel.SIGN_ENCRYPT : ProtectionLevel.ENCRYPT;
        }
        return ProtectionLevel.SIGN;
    }

    protected static enum ProtectionLevel {
        SIGN,
        ENCRYPT,
        SIGN_ENCRYPT;

    }
}

