/*
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package elemental2.core;

import java.lang.Double;
import java.lang.Object;
import java.lang.String;
import jsinterop.annotations.JsOverlay;
import jsinterop.annotations.JsPackage;
import jsinterop.annotations.JsType;
import jsinterop.base.Js;

@JsType(isNative = true, name = "Number", namespace = JsPackage.GLOBAL)
public class JsNumber {
  @JsType(isNative = true, name = "?", namespace = JsPackage.GLOBAL)
  public interface ToLocaleStringLocalesUnionType {
    @JsOverlay
    static ToLocaleStringLocalesUnionType of(Object o) {
      return Js.cast(o);
    }

    @JsOverlay
    default String asString() {
      return Js.cast(this);
    }

    @JsOverlay
    default String[] asStringArray() {
      return Js.cast(this);
    }

    @JsOverlay
    default boolean isString() {
      return (Object) this instanceof String;
    }

    @JsOverlay
    default boolean isStringArray() {
      return (Object) this instanceof Object[];
    }
  }

  @JsType(isNative = true, name = "?", namespace = JsPackage.GLOBAL)
  public interface ToStringRadixUnionType {
    @JsOverlay
    static ToStringRadixUnionType of(Object o) {
      return Js.cast(o);
    }

    @JsOverlay
    default double asDouble() {
      return Js.castToDouble(this);
    }

    @JsOverlay
    default JsNumber asJsNumber() {
      return Js.cast(this);
    }

    @JsOverlay
    default boolean isDouble() {
      return (Object) this instanceof Double;
    }

    @JsOverlay
    default boolean isJsNumber() {
      return (Object) this instanceof JsNumber;
    }
  }

  @JsOverlay public static final double EPSILON = Number__Constants.EPSILON;
  @JsOverlay public static final double MAX_SAFE_INTEGER = Number__Constants.MAX_SAFE_INTEGER;
  @JsOverlay public static final double MAX_VALUE = Number__Constants.MAX_VALUE;
  @JsOverlay public static final double MIN_SAFE_INTEGER = Number__Constants.MIN_SAFE_INTEGER;
  @JsOverlay public static final double MIN_VALUE = Number__Constants.MIN_VALUE;
  @JsOverlay public static final double NEGATIVE_INFINITY = Number__Constants.NEGATIVE_INFINITY;
  @JsOverlay public static final double NaN = Number__Constants.NaN;
  @JsOverlay public static final double POSITIVE_INFINITY = Number__Constants.POSITIVE_INFINITY;

  public static native boolean isFinite(double value);

  public static native boolean isInteger(double value);

  public static native boolean isNaN(double value);

  public static native boolean isSafeInteger(double value);

  public static native double parseFloat(String string);

  public static native double parseInt(String string, double radix);

  public JsNumber() {}

  public JsNumber(Object value) {}

  public native String toExponential();

  public native String toExponential(double fractionDigits);

  public native String toFixed();

  public native String toFixed(Object digits);

  public native String toLocaleString();

  @JsOverlay
  public final String toLocaleString(String locales, JsObject options) {
    return toLocaleString(Js.<ToLocaleStringLocalesUnionType>uncheckedCast(locales), options);
  }

  @JsOverlay
  public final String toLocaleString(String[] locales, JsObject options) {
    return toLocaleString(Js.<ToLocaleStringLocalesUnionType>uncheckedCast(locales), options);
  }

  @JsOverlay
  public final String toLocaleString(String locales) {
    return toLocaleString(Js.<ToLocaleStringLocalesUnionType>uncheckedCast(locales));
  }

  @JsOverlay
  public final String toLocaleString(String[] locales) {
    return toLocaleString(Js.<ToLocaleStringLocalesUnionType>uncheckedCast(locales));
  }

  public native String toLocaleString(ToLocaleStringLocalesUnionType locales, JsObject options);

  public native String toLocaleString(ToLocaleStringLocalesUnionType locales);

  public native String toPrecision();

  public native String toPrecision(double precision);

  public native String toString();

  @JsOverlay
  public final String toString(JsNumber radix) {
    return toString(Js.<ToStringRadixUnionType>uncheckedCast(radix));
  }

  public native String toString(ToStringRadixUnionType radix);

  @JsOverlay
  public final String toString(double radix) {
    return toString(Js.<ToStringRadixUnionType>uncheckedCast(radix));
  }
}
