/*
 * ModeShape (http://www.modeshape.org)
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * See the AUTHORS.txt file in the distribution for a full listing of
 * individual contributors.
 *
 * ModeShape is free software. Unless otherwise indicated, all code in ModeShape
 * is licensed to you under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * ModeShape is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.modeshape.jca;

import javax.jcr.Repository;
import javax.jcr.Session;

import javax.resource.ResourceException;

import javax.resource.spi.ManagedConnectionMetaData;
import org.modeshape.common.annotation.Immutable;

/**
 * Implements Managed Connection Metadata.
 *
 * @author kulikov
 */
@Immutable
public class JcrManagedConnectionMetaData implements ManagedConnectionMetaData {

    private final Repository repository;
    private final Session session;

    /**
     * Constructs new object instance.
     *
     * @param repository JCR repository instance
     * @param session JSR session instance.
     */
    public JcrManagedConnectionMetaData(Repository repository, Session session) {
        this.repository = repository;
        this.session = session;
    }

    /**
     * Returns Product name of the underlying EIS instance connected through the
     * ManagedConnection.
     *
     * @return Product name of the EIS instance
     * @throws ResourceException Thrown if an error occurs
     */
    @Override
    public String getEISProductName() throws ResourceException {
        return repository.getDescriptor(Repository.REP_NAME_DESC);
    }

    /**
     * Returns Product version of the underlying EIS instance connected through
     * the ManagedConnection.
     *
     * @return Product version of the EIS instance
     * @throws ResourceException Thrown if an error occurs
     */
    @Override
    public String getEISProductVersion() throws ResourceException {
        return repository.getDescriptor(Repository.REP_VERSION_DESC);
    }

    /**
     * Returns maximum limit on number of active concurrent connections
     *
     * @return Maximum limit for number of active concurrent connections
     * @throws ResourceException Thrown if an error occurs
     */
    @Override
    public int getMaxConnections() throws ResourceException {
        return Integer.MAX_VALUE;
    }

    /**
     * Returns name of the user associated with the ManagedConnection instance
     *
     * @return Name of the user
     * @throws ResourceException Thrown if an error occurs
     */
    @Override
    public String getUserName() throws ResourceException {
        return session.getUserID();
    }
}
