/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.schematic;

import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.modeshape.schematic.SchematicDb;
import org.modeshape.schematic.SchematicEntry;
import org.modeshape.schematic.document.Document;
import org.modeshape.schematic.document.EditableDocument;
import org.modeshape.schematic.document.Editor;
import org.modeshape.schematic.document.Json;
import org.modeshape.schematic.document.ParsingException;
import org.modeshape.schematic.internal.document.BasicDocument;

public abstract class AbstractSchematicDBTest {
    protected static final Document DEFAULT_CONTENT;
    private static final String VALUE_FIELD = "value";
    protected SchematicDb db;
    protected boolean print = false;

    protected abstract SchematicDb getDb() throws Exception;

    @Before
    public void before() throws Exception {
        this.db = this.getDb();
        this.db.start();
    }

    @After
    public void after() throws Exception {
        this.db.stop();
    }

    @Test
    public void shouldGetAndPut() throws Exception {
        List<SchematicEntry> dbEntries = this.randomEntries(3);
        this.db.txStarted("0");
        dbEntries.forEach(dbEntry -> this.db.put(dbEntry.id(), dbEntry.content()));
        Set expectedIds = dbEntries.stream().map(SchematicEntry::id).collect(Collectors.toCollection(TreeSet::new));
        junit.framework.Assert.assertTrue((boolean)this.db.keys().containsAll(expectedIds));
        this.db.txCommitted("0");
        junit.framework.Assert.assertTrue((boolean)this.db.keys().containsAll(expectedIds));
        dbEntries.stream().forEach(entry -> junit.framework.Assert.assertEquals((Object)entry.content(), (Object)this.db.getEntry(entry.id()).content()));
        SchematicEntry firstEntry = dbEntries.get(0);
        String idToUpdate = firstEntry.id();
        Editor updatedDocument = firstEntry.content().edit(true);
        updatedDocument.setNumber(VALUE_FIELD, 2);
        this.db.txStarted("1");
        this.db.get(idToUpdate);
        this.db.put(idToUpdate, (Document)updatedDocument);
        junit.framework.Assert.assertEquals((Object)updatedDocument, (Object)this.db.getEntry(idToUpdate).content());
        this.db.txCommitted("1");
        junit.framework.Assert.assertEquals((Object)updatedDocument, (Object)this.db.getEntry(idToUpdate).content());
    }

    @Test
    public void shouldReadSchematicEntry() throws Exception {
        SchematicEntry entry = this.writeSingleEntry();
        SchematicEntry schematicEntry = this.db.getEntry(entry.id());
        junit.framework.Assert.assertNotNull((Object)schematicEntry);
        junit.framework.Assert.assertTrue((boolean)this.db.containsKey(entry.id()));
    }

    @Test
    public void shouldEditContentDirectly() throws Exception {
        SchematicEntry entry = this.writeSingleEntry();
        EditableDocument editableDocument = this.simulateTransaction(() -> this.db.editContent(entry.id(), false));
        junit.framework.Assert.assertNotNull((Object)editableDocument);
        junit.framework.Assert.assertEquals((Object)entry.content(), (Object)editableDocument);
        this.simulateTransaction(() -> {
            EditableDocument document = this.db.editContent(entry.id(), false);
            document.setNumber(VALUE_FIELD, 2);
            return null;
        });
        Document doc = this.db.getEntry(entry.id()).content();
        junit.framework.Assert.assertEquals((int)2, (int)doc.getInteger(VALUE_FIELD));
        String newId = UUID.randomUUID().toString();
        EditableDocument newDocument = this.simulateTransaction(() -> this.db.editContent(newId, true));
        junit.framework.Assert.assertNotNull((Object)newDocument);
        SchematicEntry schematicEntry = this.db.getEntry(newId);
        junit.framework.Assert.assertEquals((String)newId, (String)schematicEntry.id());
        junit.framework.Assert.assertEquals((Object)new BasicDocument(), (Object)schematicEntry.content());
        newDocument = this.simulateTransaction(() -> this.db.editContent(UUID.randomUUID().toString(), false));
        Assert.assertNull((Object)newDocument);
    }

    @Test
    public void shouldPutIfAbsent() throws Exception {
        SchematicEntry entry = this.writeSingleEntry();
        Editor editableDocument = entry.content().edit(true);
        editableDocument.setNumber(VALUE_FIELD, 100);
        SchematicEntry updatedEntry = this.simulateTransaction(() -> this.db.putIfAbsent(entry.id(), entry.content()));
        junit.framework.Assert.assertNotNull((Object)updatedEntry);
        junit.framework.Assert.assertEquals((int)1, (int)updatedEntry.content().getInteger(VALUE_FIELD));
        SchematicEntry newEntry = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
        Assert.assertNull((Object)this.simulateTransaction(() -> this.db.putIfAbsent(newEntry.id(), newEntry.content())));
        updatedEntry = this.db.getEntry(newEntry.id());
        junit.framework.Assert.assertNotNull((Object)updatedEntry);
    }

    @Test
    public void shouldPutSchematicEntry() throws Exception {
        SchematicEntry originalEntry = this.randomEntries(1).get(0);
        this.simulateTransaction(() -> {
            this.db.putEntry(originalEntry.source());
            return null;
        });
        SchematicEntry actualEntry = this.db.getEntry(originalEntry.id());
        junit.framework.Assert.assertNotNull((Object)actualEntry);
        junit.framework.Assert.assertEquals((Object)originalEntry.getMetadata(), (Object)actualEntry.getMetadata());
        junit.framework.Assert.assertEquals((Object)originalEntry.content(), (Object)actualEntry.content());
        junit.framework.Assert.assertEquals((Object)DEFAULT_CONTENT, (Object)actualEntry.content());
    }

    @Test
    public void shouldRemoveDocument() throws Exception {
        SchematicEntry entry = this.writeSingleEntry();
        this.simulateTransaction(() -> this.db.remove(entry.id()));
        junit.framework.Assert.assertFalse((boolean)this.db.containsKey(entry.id()));
    }

    @Test
    public void shouldRemoveAllDocuments() throws Exception {
        int count = 3;
        this.simulateTransaction(() -> {
            this.randomEntries(count).forEach(entry -> this.db.put(entry.id(), entry.content()));
            return null;
        });
        junit.framework.Assert.assertFalse((boolean)this.db.keys().isEmpty());
        this.simulateTransaction(() -> {
            this.db.removeAll();
            return null;
        });
        junit.framework.Assert.assertTrue((boolean)this.db.keys().isEmpty());
    }

    @Test
    public void shouldIsolateChangesWithinTransaction() throws Exception {
        SchematicEntry entry1 = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
        SchematicEntry entry2 = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
        CyclicBarrier syncBarrier = new CyclicBarrier(2);
        CompletableFuture<Void> thread1 = CompletableFuture.runAsync(() -> this.changeAndCommit(entry1, entry2, syncBarrier));
        CompletableFuture<Void> thread2 = CompletableFuture.runAsync(() -> this.changeAndCommit(entry2, entry1, syncBarrier));
        thread1.get(3L, TimeUnit.SECONDS);
        thread2.get(3L, TimeUnit.SECONDS);
        Assert.assertFalse((boolean)this.db.containsKey(entry1.id()));
        Assert.assertFalse((boolean)this.db.containsKey(entry2.id()));
    }

    @Test
    public void shouldRollbackChangesWithinTransaction() throws Exception {
        SchematicEntry entry1 = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
        SchematicEntry entry2 = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
        CyclicBarrier syncBarrier = new CyclicBarrier(2);
        CompletableFuture<Void> thread1 = CompletableFuture.runAsync(() -> this.changeAndRollback(entry1, entry2, syncBarrier));
        CompletableFuture<Void> thread2 = CompletableFuture.runAsync(() -> this.changeAndRollback(entry2, entry1, syncBarrier));
        thread1.get(2L, TimeUnit.SECONDS);
        thread2.get(2L, TimeUnit.SECONDS);
        Assert.assertEquals((Object)entry1.content(), (Object)this.db.getEntry(entry1.id()).content());
        Assert.assertEquals((Object)entry2.content(), (Object)this.db.getEntry(entry2.id()).content());
    }

    @Test
    public void shouldInsertAndUpdateEntriesConcurrentlyWithMultipleWriters() throws Exception {
        int threadsCount = 100;
        int entriesPerThread = 100;
        ExecutorService executors = Executors.newFixedThreadPool(threadsCount);
        this.print = false;
        this.print("Starting the run of " + threadsCount + " threads with " + entriesPerThread + " insertions per thread...");
        long startTime = System.nanoTime();
        List results = IntStream.range(0, threadsCount).mapToObj(value -> this.insertMultipleEntries(entriesPerThread, executors)).collect(Collectors.toList());
        results.stream().map(future -> {
            try {
                return (List)future.get(2L, TimeUnit.MINUTES);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }).flatMap(Collection::stream).forEach(id -> Assert.assertTrue((boolean)this.db.containsKey(id)));
        long durationMillis = TimeUnit.MILLISECONDS.convert(System.nanoTime() - startTime, TimeUnit.NANOSECONDS);
        if (this.print) {
            System.out.printf("Total duration to insert " + threadsCount * entriesPerThread + " entries : " + (double)durationMillis / 1000.0 + " seconds", new Object[0]);
        }
    }

    protected CompletableFuture<List<String>> insertMultipleEntries(int entriesPerThread, ExecutorService executors) {
        return CompletableFuture.supplyAsync(() -> {
            if (this.print) {
                System.out.println(Thread.currentThread().getName() + " inserting " + entriesPerThread + " entries...");
            }
            String txId = UUID.randomUUID().toString();
            this.db.txStarted(txId);
            List ids = null;
            try {
                ids = this.randomEntries(entriesPerThread).stream().map(dbEntry -> {
                    this.db.put(dbEntry.id(), dbEntry.content());
                    return dbEntry.id();
                }).collect(Collectors.toList());
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            this.db.txCommitted(txId);
            return ids;
        }, executors);
    }

    private void changeAndRollback(SchematicEntry ourEntry, SchematicEntry otherEntry, CyclicBarrier syncBarrier) {
        try {
            String txId = UUID.randomUUID().toString();
            this.db.txStarted(txId);
            this.db.put(ourEntry.id(), ourEntry.content());
            this.db.txCommitted(txId);
            syncBarrier.await();
            Document ourDocument = this.db.getEntry(ourEntry.id()).content();
            Document otherDocument = this.db.getEntry(otherEntry.id()).content();
            Assert.assertEquals((Object)ourDocument, (Object)otherDocument);
            txId = UUID.randomUUID().toString();
            this.db.txStarted(txId);
            this.db.put(ourEntry.id(), (Document)new BasicDocument());
            this.db.txRolledback(txId);
            syncBarrier.await();
            Assert.assertEquals((Object)ourDocument, (Object)this.db.getEntry(ourEntry.id()).content());
            Assert.assertEquals((Object)otherDocument, (Object)this.db.getEntry(otherEntry.id()).content());
        }
        catch (RuntimeException re) {
            syncBarrier.reset();
            throw re;
        }
        catch (Throwable t) {
            t.printStackTrace();
            syncBarrier.reset();
            throw new RuntimeException(t);
        }
    }

    protected void changeAndCommit(SchematicEntry ourEntry, SchematicEntry otherEntry, CyclicBarrier syncBarrier) {
        try {
            String txId = UUID.randomUUID().toString();
            this.db.txStarted(txId);
            this.db.put(ourEntry.id(), ourEntry.content());
            Assert.assertTrue((boolean)this.db.containsKey(ourEntry.id()));
            Assert.assertFalse((boolean)this.db.containsKey(otherEntry.id()));
            BasicDocument updatedDoc = new BasicDocument();
            this.db.put(ourEntry.id(), (Document)updatedDoc);
            Document actualDocument = this.db.getEntry(ourEntry.id()).content();
            Assert.assertTrue((boolean)this.db.containsKey(ourEntry.id()));
            Assert.assertFalse((boolean)this.db.containsKey(otherEntry.id()));
            Assert.assertEquals((Object)updatedDoc, (Object)actualDocument);
            syncBarrier.await();
            this.db.txCommitted(txId);
            syncBarrier.await();
            Assert.assertTrue((boolean)this.db.containsKey(otherEntry.id()));
            Document otherDocument = this.db.getEntry(otherEntry.id()).content();
            Assert.assertEquals((Object)updatedDoc, (Object)otherDocument);
            txId = UUID.randomUUID().toString();
            this.db.txStarted(txId);
            this.db.remove(ourEntry.id());
            syncBarrier.await();
            Assert.assertFalse((boolean)this.db.containsKey(ourEntry.id()));
            Assert.assertTrue((boolean)this.db.containsKey(otherEntry.id()));
            syncBarrier.await();
            this.db.txCommitted(txId);
            syncBarrier.await();
            Assert.assertFalse((boolean)this.db.containsKey(ourEntry.id()));
            Assert.assertFalse((boolean)this.db.containsKey(otherEntry.id()));
        }
        catch (RuntimeException re) {
            syncBarrier.reset();
            throw re;
        }
        catch (Throwable t) {
            t.printStackTrace();
            syncBarrier.reset();
            throw new RuntimeException(t);
        }
    }

    protected <T> T simulateTransaction(Callable<T> operation) throws Exception {
        this.db.txStarted("0");
        T result = operation.call();
        this.db.txCommitted("0");
        return result;
    }

    protected SchematicEntry writeSingleEntry() throws Exception {
        return this.simulateTransaction(() -> {
            SchematicEntry entry = SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT);
            this.db.putEntry(entry.source());
            return entry;
        });
    }

    protected List<SchematicEntry> randomEntries(int sampleSize) throws Exception {
        return IntStream.range(0, sampleSize).mapToObj(i -> SchematicEntry.create((String)UUID.randomUUID().toString(), (Document)DEFAULT_CONTENT)).collect(Collectors.toList());
    }

    protected void print(String s) {
        if (this.print) {
            System.out.println(Thread.currentThread().getName() + ": " + s);
        }
    }

    static {
        try {
            DEFAULT_CONTENT = Json.read((InputStream)AbstractSchematicDBTest.class.getClassLoader().getResourceAsStream("document.json"));
        }
        catch (ParsingException e) {
            throw new RuntimeException(e);
        }
    }
}

