/*
 * JBoss, Home of Professional Open Source. Copyright 2008, Red Hat Middleware LLC, and individual contributors as
 * indicated by the @author tags. See the copyright.txt file in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this software; if not, write to
 * the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF site:
 * http://www.fsf.org.
 */
package org.picketlink.identity.federation.core.wstrust;

import java.net.URI;
import java.security.Principal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPPart;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.Service.Mode;
import javax.xml.ws.soap.SOAPBinding;

import org.picketlink.identity.federation.core.saml.v2.util.DocumentUtil;
import org.picketlink.identity.federation.core.wstrust.wrappers.RequestSecurityToken;
import org.picketlink.identity.federation.core.wstrust.wrappers.RequestSecurityTokenCollection;
import org.picketlink.identity.federation.core.wstrust.wrappers.RequestSecurityTokenResponse;
import org.picketlink.identity.federation.core.wstrust.wrappers.RequestSecurityTokenResponseCollection;
import org.picketlink.identity.federation.ws.trust.CancelTargetType;
import org.picketlink.identity.federation.ws.trust.RenewTargetType;
import org.picketlink.identity.federation.ws.trust.StatusType;
import org.picketlink.identity.federation.ws.trust.ValidateTargetType;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * WS-Trust Client
 * 
 * @author Anil.Saldhana@redhat.com
 * @since Aug 29, 2009
 */
public class STSClient
{
   private ThreadLocal<Dispatch<Source>> dispatchLocal = new InheritableThreadLocal<Dispatch<Source>>();

   private String targetNS = "http://org.picketlink.trust/sts/";

   public STSClient(STSClientConfig config)
   {
      QName service = new QName(targetNS, config.getServiceName());
      QName portName = new QName(targetNS, config.getPortName());

      Service jaxwsService = Service.create(service);
      jaxwsService.addPort(portName, SOAPBinding.SOAP11HTTP_BINDING, config.getEndPointAddress());
      Dispatch<Source> dispatch = jaxwsService.createDispatch(portName, Source.class, Mode.PAYLOAD);

      Map<String, Object> reqContext = dispatch.getRequestContext();
      String username = config.getUsername();
      if (username != null)
      {
         // add the username and password to the request context.
         reqContext.put(BindingProvider.USERNAME_PROPERTY, config.getUsername());
         reqContext.put(BindingProvider.PASSWORD_PROPERTY, config.getPassword());
      }
      dispatchLocal.set(dispatch);
   }

   public Element issueToken(String tokenType) throws WSTrustException
   {
      // create a custom token request message.
      RequestSecurityToken request = new RequestSecurityToken();
      setTokenType(tokenType, request);
      // send the token request to JBoss STS and get the response.
      return issueToken(request);
   }

   /**
    * Issues a Security Token for the ultimate recipient of the token.
    * 
    * @param endpointURI - The ultimate recipient of the token. This will be set at the AppliesTo for the
    *           RequestSecurityToken which is an optional element so it may be null.
    * @return Element - The Security Token Element which will be of the TokenType configured for the endpointURI passed
    *         in.
    * @throws WSTrustException
    */
   public Element issueTokenForEndpoint(String endpointURI) throws WSTrustException
   {
      RequestSecurityToken request = new RequestSecurityToken();
      setAppliesTo(endpointURI, request);
      return issueToken(request);
   }

   /**
    * Issues a Security Token from the STS. This methods has the option of specifying one or both of
    * endpointURI/tokenType but at least one must specified.
    * 
    * @param endpointURI - The ultimate recipient of the token. This will be set at the AppliesTo for the
    *           RequestSecurityToken which is an optional element so it may be null.
    * @param tokenType - The type of security token to be issued.
    * @return Element - The Security Token Element issued.
    * @throws IllegalArgumentException If neither endpointURI nor tokenType was specified.
    * @throws WSTrustException
    */
   public Element issueToken(String endpointURI, String tokenType) throws WSTrustException
   {
      if (endpointURI == null && tokenType == null)
         throw new IllegalArgumentException("One of endpointURI or tokenType must be provided.");

      RequestSecurityToken request = new RequestSecurityToken();
      setAppliesTo(endpointURI, request);
      setTokenType(tokenType, request);
      return issueToken(request);
   }

   /**
    * <p>
    * Issues a security token on behalf of the specified principal.
    * </p>
    * 
    * @param endpointURI the ultimate recipient of the token. This will be set at the AppliesTo for the
    *           RequestSecurityToken which is an optional element so it may be null.
    * @param tokenType the type of the token to be issued.
    * @param principal the {@code Principal} to whom the token will be issued.
    * @return an {@code Element} representing the issued security token.
    * @throws IllegalArgumentException If neither endpointURI nor tokenType was specified.
    * @throws WSTrustException if an error occurs while issuing the security token.
    */
   public Element issueTokenOnBehalfOf(String endpointURI, String tokenType, Principal principal)
         throws WSTrustException
   {
      if (endpointURI == null && tokenType == null)
         throw new IllegalArgumentException("One of endpointURI or tokenType must be provided.");

      RequestSecurityToken request = new RequestSecurityToken();
      setAppliesTo(endpointURI, request);
      setTokenType(tokenType, request);
      setOnBehalfOf(principal, request);
      return issueToken(request);
   }

   private RequestSecurityToken setAppliesTo(String endpointURI, RequestSecurityToken rst)
   {
      if (endpointURI != null)
         rst.setAppliesTo(WSTrustUtil.createAppliesTo(endpointURI));
      return rst;
   }

   private RequestSecurityToken setTokenType(String tokenType, RequestSecurityToken rst)
   {
      if (tokenType != null)
         rst.setTokenType(URI.create(tokenType));
      return rst;
   }

   private RequestSecurityToken setOnBehalfOf(Principal principal, RequestSecurityToken request)
   {
      if (principal != null)
         request.setOnBehalfOf(WSTrustUtil.createOnBehalfOfWithUsername(principal.getName(), null));
      return request;
   }

   public Element issueToken(RequestSecurityToken request) throws WSTrustException
   {
      // convert the request type to BatchIssue before dispatching the batch request.
      request.setRequestType(URI.create(WSTrustConstants.BATCH_ISSUE_REQUEST));
      RequestSecurityTokenCollection requestCollection = new RequestSecurityTokenCollection();
      requestCollection.addRequestSecurityToken(request);
      return this.issueTokens(requestCollection).get(0);
   }

   public List<Element> issueTokens(RequestSecurityTokenCollection requestCollection) throws WSTrustException
   {
      // check if all requests are of type BatchIssue.
      for (RequestSecurityToken request : requestCollection.getRequestSecurityTokens())
      {
         // if null or type: assume BatchIssue.
         if (request.getRequestType() == null)
            request.setRequestType(URI.create(WSTrustConstants.BATCH_ISSUE_REQUEST));
         // non-null type: check if type equals BatchIssue.
         else if (!request.getRequestType().toString().equals(WSTrustConstants.BATCH_ISSUE_REQUEST))
            throw new IllegalArgumentException("The request type must be of type BatchIssue");
      }

      // use the JAXB factory to marshal the batch request.
      WSTrustJAXBFactory jaxbFactory = WSTrustJAXBFactory.getInstance();
      DOMSource requestSource = (DOMSource) jaxbFactory.marshallRequestSecurityTokenCollection(requestCollection);

      // invoke the STS.
      Source response = dispatchLocal.get().invoke(requestSource);

      // parse the response and extract all nodes that contain the security tokens.
      NodeList nodes;
      try
      {
         Node documentNode = DocumentUtil.getNodeFromSource(response);
         Document responseDoc = documentNode instanceof Document ? (Document) documentNode : documentNode
               .getOwnerDocument();

         nodes = null;
         if (responseDoc instanceof SOAPPart)
         {
            SOAPPart soapPart = (SOAPPart) responseDoc;
            SOAPEnvelope env = soapPart.getEnvelope();
            SOAPBody body = env.getBody();
            Node data = body.getFirstChild();
            nodes = ((Element) data).getElementsByTagNameNS(WSTrustConstants.BASE_NAMESPACE, "RequestedSecurityToken");
            if (nodes == null || nodes.getLength() == 0)
               nodes = ((Element) data).getElementsByTagName("RequestedSecurityToken");
         }
         else
         {
            nodes = responseDoc.getElementsByTagNameNS(WSTrustConstants.BASE_NAMESPACE, "RequestedSecurityToken");
            if (nodes == null || nodes.getLength() == 0)
               nodes = responseDoc.getElementsByTagName("RequestedSecurityToken");
         }
      }
      catch (Exception e)
      {
         throw new WSTrustException("Exception in issuing token:", e);
      }
      if (nodes == null)
         throw new WSTrustException("NodeList is null");
      else
      {
         List<Element> tokens = new ArrayList<Element>();
         for (int i = 0; i < nodes.getLength(); i++)
         {
            Node node = nodes.item(i);
            tokens.add((Element) node.getFirstChild());
         }
         return tokens;
      }
   }

   /**
    * @deprecated tokenType parameter is unnecessary as the type can be derived from the token. Use
    *             {@link #renewToken(Element)} instead.
    */
   public Element renewToken(String tokenType, Element token) throws WSTrustException
   {
      return this.renewToken(token);
   }

   public Element renewToken(Element token) throws WSTrustException
   {
      List<Element> tokens = new ArrayList<Element>();
      tokens.add(token);
      return this.renewTokens(tokens).get(0);
   }

   public List<Element> renewTokens(List<Element> tokens) throws WSTrustException
   {
      // create a request collection containing all tokens to be renewed.
      RequestSecurityTokenCollection requestCollection = new RequestSecurityTokenCollection();
      for (Element token : tokens)
      {
         RequestSecurityToken request = new RequestSecurityToken();
         request.setRequestType(URI.create(WSTrustConstants.BATCH_RENEW_REQUEST));
         RenewTargetType renewTarget = new RenewTargetType();
         renewTarget.setAny(token);
         request.setRenewTarget(renewTarget);
         requestCollection.addRequestSecurityToken(request);
      }

      // use the JAXB factory to marshal the batch request.
      WSTrustJAXBFactory jaxbFactory = WSTrustJAXBFactory.getInstance();
      DOMSource requestSource = (DOMSource) jaxbFactory.marshallRequestSecurityTokenCollection(requestCollection);

      // invoke the STS.
      Source response = dispatchLocal.get().invoke(requestSource);

      // parse the response and extract all nodes that contain the security tokens.
      NodeList nodes;
      try
      {
         Node documentNode = DocumentUtil.getNodeFromSource(response);
         Document responseDoc = documentNode instanceof Document ? (Document) documentNode : documentNode
               .getOwnerDocument();

         nodes = null;
         if (responseDoc instanceof SOAPPart)
         {
            SOAPPart soapPart = (SOAPPart) responseDoc;
            SOAPEnvelope env = soapPart.getEnvelope();
            SOAPBody body = env.getBody();
            Node data = body.getFirstChild();
            nodes = ((Element) data).getElementsByTagNameNS(WSTrustConstants.BASE_NAMESPACE, "RequestedSecurityToken");
            if (nodes == null || nodes.getLength() == 0)
               nodes = ((Element) data).getElementsByTagName("RequestedSecurityToken");
         }
         else
         {
            nodes = responseDoc.getElementsByTagNameNS(WSTrustConstants.BASE_NAMESPACE, "RequestedSecurityToken");
            if (nodes == null || nodes.getLength() == 0)
               nodes = responseDoc.getElementsByTagName("RequestedSecurityToken");
         }
      }
      catch (Exception e)
      {
         throw new WSTrustException("Exception in issuing token:", e);
      }
      if (nodes == null)
         throw new WSTrustException("NodeList is null");
      else
      {
         List<Element> renewedTokens = new ArrayList<Element>();
         for (int i = 0; i < nodes.getLength(); i++)
         {
            Node node = nodes.item(i);
            renewedTokens.add((Element) node.getFirstChild());
         }
         return renewedTokens;
      }
   }

   public boolean validateToken(Element token) throws WSTrustException
   {
      List<Element> tokens = new ArrayList<Element>();
      tokens.add(token);
      return this.validateTokens(tokens).get(0);
   }

   public List<Boolean> validateTokens(List<Element> tokens) throws WSTrustException
   {
      // create a request collection containing all tokens to be validated.
      RequestSecurityTokenCollection requestCollection = new RequestSecurityTokenCollection();
      for (Element token : tokens)
      {
         RequestSecurityToken request = new RequestSecurityToken();
         request.setTokenType(URI.create(WSTrustConstants.STATUS_TYPE));
         request.setRequestType(URI.create(WSTrustConstants.BATCH_VALIDATE_REQUEST));
         ValidateTargetType validateTarget = new ValidateTargetType();
         validateTarget.setAny(token);
         request.setValidateTarget(validateTarget);
         requestCollection.addRequestSecurityToken(request);
      }

      // use the JAXB factory to marshal the batch request.
      WSTrustJAXBFactory jaxbFactory = WSTrustJAXBFactory.getInstance();
      DOMSource requestSource = (DOMSource) jaxbFactory.marshallRequestSecurityTokenCollection(requestCollection);

      // invoke the STS.
      Source response = dispatchLocal.get().invoke(requestSource);

      // parse the response and check the validation status of each security token.
      RequestSecurityTokenResponseCollection responseCollection = (RequestSecurityTokenResponseCollection) jaxbFactory
            .parseRequestSecurityTokenResponse(response);

      List<Boolean> result = new ArrayList<Boolean>();
      for (RequestSecurityTokenResponse tokenResponse : responseCollection.getRequestSecurityTokenResponses())
      {
         StatusType status = tokenResponse.getStatus();
         if (status != null)
         {
            String code = status.getCode();
            result.add(WSTrustConstants.STATUS_CODE_VALID.equals(code));
         }
         result.add(Boolean.FALSE);
      }
      return result;
   }

   /**
    * <p>
    * Cancels the specified security token by sending a WS-Trust cancel message to the STS.
    * </p>
    * 
    * @param securityToken the security token to be canceled.
    * @return {@code true} if the token has been canceled by the STS; {@code false} otherwise.
    * @throws WSTrustException if an error occurs while processing the cancel request.
    */
   public boolean cancelToken(Element securityToken) throws WSTrustException
   {
      List<Element> tokens = new ArrayList<Element>();
      tokens.add(securityToken);
      return this.cancelTokens(tokens).get(0);
   }

   public List<Boolean> cancelTokens(List<Element> tokens) throws WSTrustException
   {
      // create a request collection containing all tokens to be canceled.
      RequestSecurityTokenCollection requestCollection = new RequestSecurityTokenCollection();
      for (Element token : tokens)
      {
         RequestSecurityToken request = new RequestSecurityToken();
         request.setRequestType(URI.create(WSTrustConstants.BATCH_CANCEL_REQUEST));
         CancelTargetType cancelTarget = new CancelTargetType();
         cancelTarget.setAny(token);
         request.setCancelTarget(cancelTarget);
         requestCollection.addRequestSecurityToken(request);
      }

      // marshal the request and send it to the STS.
      WSTrustJAXBFactory jaxbFactory = WSTrustJAXBFactory.getInstance();
      DOMSource requestSource = (DOMSource) jaxbFactory.marshallRequestSecurityTokenCollection(requestCollection);
      Source response = dispatchLocal.get().invoke(requestSource);

      // get the WS-Trust response and check for presence of the RequestTokenCanceled element.
      RequestSecurityTokenResponseCollection responseCollection = (RequestSecurityTokenResponseCollection) jaxbFactory
            .parseRequestSecurityTokenResponse(response);

      List<Boolean> result = new ArrayList<Boolean>();
      for (RequestSecurityTokenResponse tokenResponse : responseCollection.getRequestSecurityTokenResponses())
      {
         if (tokenResponse.getRequestedTokenCancelled() != null)
            result.add(Boolean.TRUE);
         result.add(Boolean.FALSE);
      }

      return result;
   }

   public Dispatch<Source> getDispatch()
   {
      return dispatchLocal.get();
   }
}