/*
 * Copyright 2017 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.uberfire.ext.widgets.core.client.tree;

import java.util.Iterator;
import java.util.function.Supplier;

import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Style;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.ComplexPanel;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.SimplePanel;
import org.uberfire.ext.widgets.core.client.resources.TreeNavigatorResources;

import static org.uberfire.commons.validation.PortablePreconditions.checkNotNull;

public class TreeItem<I extends TreeItem> extends Composite {

    private final Type type;
    protected FlowPanel content;
    private Tree<I> tree;
    private Object userObject;
    private TreeItem parent;
    private State state;
    private String label;
    private String uuid;
    private boolean isSelected = false;
    private FlowPanel header;
    private IsWidget icon;
    private FlowPanel item;

    public TreeItem(final Type type,
                    final String value,
                    final String label,
                    final IsWidget icon) {
        this(type,
             value,
             label,
             icon,
             FlowPanel::new);
    }

    TreeItem(final Type type,
             final String value,
             final String label,
             final IsWidget icon,
             final Supplier<FlowPanel> contentProvider
    ) {
        this.label = label;
        this.uuid = value;
        this.type = checkNotNull("type",
                                 type);

        if (type.equals(Type.CONTAINER) || type.equals(Type.ROOT)) {
            final FlowPanel folder = contentProvider.get();
            folder.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeFolder());
            folder.getElement().getStyle().setDisplay(Style.Display.BLOCK);
            {
                this.state = State.CLOSE;
                this.header = GWT.create(FlowPanel.class);
                this.icon = icon;
                this.content = contentProvider.get();
                final Anchor name = new Anchor();
                {
                    header.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeFolderHeader());
                    folder.add(header);
                    {
                        header.add(icon);
                    }
                    final FlowPanel folderName = new FlowPanel();
                    {
                        folderName.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeFolderName());
                        header.add(folderName);
                        {
                            name.setText(label);
                            name.setTitle(value);
                            folderName.add(name);
                        }
                    }
                    header.addDomHandler(new ClickHandler() {
                                             @Override
                                             public void onClick(ClickEvent event) {
                                                 if (!isSelected) {
                                                     updateSelected();
                                                 }
                                                 if (state.equals(State.CLOSE)) {
                                                     setState(State.OPEN,
                                                              true);
                                                 } else {
                                                     setState(State.CLOSE,
                                                              true);
                                                 }
                                             }
                                         },
                                         ClickEvent.getType());
                }
                {
                    content.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeFolderContent());
                    content.getElement().getStyle().setDisplay(Style.Display.NONE);
                    folder.add(content);
                }
                initWidget(folder);
            }
        } else if (type.equals(Type.ITEM)) {
            //this.item = new FlowPanel();
            this.state = State.NONE;
            this.item = contentProvider.get();
            item.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeItem());
            {
                this.icon = icon;
                final FlowPanel itemName = new FlowPanel();
                final Anchor name = new Anchor();
                {
                    item.add(icon);
                }
                {
                    itemName.setStylePrimaryName(TreeNavigatorResources.INSTANCE.css().treeItemName());
                    item.add(itemName);
                    {
                        name.setText(label);
                        name.setTitle(value);

                        itemName.add(name);
                    }
                }
                item.addDomHandler(new ClickHandler() {
                                       @Override
                                       public void onClick(ClickEvent event) {
                                           tree.onSelection((I) TreeItem.this,
                                                            true);
                                       }
                                   },
                                   ClickEvent.getType());
            }
            initWidget(item);
        } else {
            final FlowPanel loader = new FlowPanel();
            {
                final SimplePanel loading = new SimplePanel();
                loading.getElement().setInnerText(value);
                loader.add(loading);
            }
            initWidget(loader);
        }
    }

    private void updateSelected() {
        tree.onSelection((I) this,
                         true);
    }

    public State getState() {
        return state;
    }

    public void setState(final State state) {
        setState(state,
                 false,
                 true);
    }

    public void setState(final State state,
                         boolean fireEvents) {
        setState(state,
                 false,
                 fireEvents);
    }

    public void setState(final State state,
                         boolean propagateParent,
                         boolean fireEvents) {
        if (notFolder()) {
            return;
        }

        if (!this.state.equals(state)) {
            this.state = state;
            updateState(state);

            if (fireEvents && tree != null) {
                tree.fireStateChanged((I) this,
                                      state);
            }
        }
        if (propagateParent && parent != null) {
            parent.setState(state,
                            true,
                            false);
        }
    }

    private boolean notFolder() {
        return !type.equals(Type.CONTAINER);
    }

    public Object getUserObject() {
        return userObject;
    }

    public void setUserObject(final Object userObject) {
        this.userObject = userObject;
    }

    public Type getType() {
        return this.type;
    }

    @SuppressWarnings("unchecked")
    public I addItem(final I item) {

        content.add(item);
        item.setTree(tree);
        item.setParent(this);

        return item;
    }

    @SuppressWarnings("unchecked")
    public TreeItem addItem(final Type type,
                            final String value,
                            final String label,
                            final IsWidget icon) {
        final I child = makeChild(type,
                                         value,
                                         label,
                                         icon);
        addItem(child);
        return child;
    }

    private I makeChild(final Type type,
                               final String value,
                               final String label,
                               final IsWidget icon) {
        return (I) new TreeItem(type,
                                value,
                                label,
                                icon);
    }

    public void removeItems() {
        content.clear();
    }

    public int getChildCount() {
        return content.getWidgetCount();
    }

    @SuppressWarnings("unchecked")
    public I getChild(final int i) {
        if (i + 1 > content.getWidgetCount()) {
            return null;
        }
        return (I)content.getWidget(i);
    }

    public Iterable<I> getChildren() {
        return () -> new TreeItemIterator(content);
    }

    void setTree(final Tree<I> tree) {
        this.tree = tree;
    }

    void setParent(final TreeItem<I> parent) {
        this.parent = parent;
    }

    void updateState(final State state) {
        // If the tree hasn't been set, there is no visual state to update.
        // If the tree is not attached, then update will be called on attach.
        if (tree == null) {
            return;
        }

        switch (state) {
            case OPEN:
                onOpenState();
                break;
            case CLOSE:
                onCloseState();
        }
    }

    protected void onOpenState() {
        content.getElement().getStyle().setDisplay(Style.Display.BLOCK);
    }

    protected void onCloseState() {
        content.getElement().getStyle().setDisplay(Style.Display.NONE);
    }

    @SuppressWarnings("unchecked")
    public void remove() {
        if (parent != null) {
            // If this item has a parent, remove self from it.
            parent.removeItem(this);
        } else if (tree != null) {
            // If the item has no parent, but is in the Tree, it must be a top-level
            // element.
            tree.removeItem((I) this);
        }
    }

    public void removeItem(final I treeItem) {
        content.remove(treeItem);
    }

    public String getText() {
        return getElement().getInnerText();
    }

    public String getUuid() {
        return uuid;
    }

    public String getLabel() {
        return label;
    }

    public boolean isSelected() {
        return isSelected;
    }

    void setSelected(boolean selected) {
        isSelected = selected;
        if (selected) {
            if (header != null) {
                header.addStyleName(TreeNavigatorResources.INSTANCE.css().treeSelected());
            } else {
                item.addStyleName(TreeNavigatorResources.INSTANCE.css().treeSelected());
            }
        } else {
            if (header != null) {
                header.removeStyleName(TreeNavigatorResources.INSTANCE.css().treeSelected());
            } else {
                item.removeStyleName(TreeNavigatorResources.INSTANCE.css().treeSelected());
            }
        }
    }

    public boolean isEmpty() {
        return content.getWidgetCount() == 0;
    }

    IsWidget getIconWidget() {
        return icon;
    }

    FlowPanel getContent() {
        return content;
    }

    public enum Type {
        ROOT,
        CONTAINER,
        ITEM,
    }

    public enum State {
        NONE,
        OPEN,
        CLOSE
    }

    protected static class TreeItemIterator<T> implements Iterator<T> {

        private final ComplexPanel container;
        private int index = 0;

        TreeItemIterator(ComplexPanel container) {
            this.container = container;
        }

        @Override
        public boolean hasNext() {
            if (container == null) {
                return false;
            }
            return index < container.getWidgetCount();
        }

        @Override
        @SuppressWarnings("unchecked")
        public T next() {
            return (T) container.getWidget(index++);
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}