/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.retry;

import java.util.Objects;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import org.neo4j.driver.internal.retry.ExponentialBackoffDecision;
import org.neo4j.driver.internal.retry.RetryDecision;
import org.neo4j.driver.internal.retry.RetryLogic;
import org.neo4j.driver.internal.retry.RetrySettings;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.v1.exceptions.ServiceUnavailableException;
import org.neo4j.driver.v1.exceptions.SessionExpiredException;
import org.neo4j.driver.v1.exceptions.TransientException;

public class ExponentialBackoff
implements RetryLogic<ExponentialBackoffDecision> {
    public static final long DEFAULT_MAX_RETRY_TIME_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final long INITIAL_RETRY_DELAY_MS = TimeUnit.SECONDS.toMillis(1L);
    private static final double RETRY_DELAY_MULTIPLIER = 2.0;
    private static final double RETRY_DELAY_JITTER_FACTOR = 0.2;
    private final long maxRetryTimeMs;
    private final long initialRetryDelayMs;
    private final double multiplier;
    private final double jitterFactor;
    private final Clock clock;

    ExponentialBackoff(long maxRetryTimeMs, long initialRetryDelayMs, double multiplier, double jitterFactor, Clock clock) {
        this.maxRetryTimeMs = maxRetryTimeMs;
        this.initialRetryDelayMs = initialRetryDelayMs;
        this.multiplier = multiplier;
        this.jitterFactor = jitterFactor;
        this.clock = clock;
        this.verifyAfterConstruction();
    }

    public static RetryLogic<RetryDecision> defaultRetryLogic() {
        return ExponentialBackoff.create(RetrySettings.DEFAULT, Clock.SYSTEM);
    }

    public static RetryLogic<RetryDecision> noRetries() {
        return ExponentialBackoff.create(new RetrySettings(0L), Clock.SYSTEM);
    }

    public static RetryLogic<RetryDecision> create(RetrySettings settings, Clock clock) {
        return new ExponentialBackoff(settings.maxRetryTimeMs(), INITIAL_RETRY_DELAY_MS, 2.0, 0.2, clock);
    }

    @Override
    public ExponentialBackoffDecision apply(Throwable error, ExponentialBackoffDecision previousDecision) {
        Objects.requireNonNull(error);
        ExponentialBackoffDecision decision = this.decision(previousDecision);
        long elapsedTimeMs = this.clock.millis() - decision.startTimestamp();
        if (elapsedTimeMs > this.maxRetryTimeMs || !ExponentialBackoff.canRetryOn(error)) {
            return decision.stopRetrying();
        }
        long delayWithJitterMs = this.computeDelayWithJitter(decision.delay());
        this.sleep(delayWithJitterMs);
        long nextDelayWithoutJitterMs = (long)((double)decision.delay() * this.multiplier);
        return decision.withDelay(nextDelayWithoutJitterMs);
    }

    private ExponentialBackoffDecision decision(ExponentialBackoffDecision previous) {
        return previous == null ? new ExponentialBackoffDecision(this.clock.millis(), this.initialRetryDelayMs) : previous;
    }

    private long computeDelayWithJitter(long delayMs) {
        long jitter = (long)((double)delayMs * this.jitterFactor);
        long min = delayMs - jitter;
        long max = delayMs + jitter;
        if (max < 0L) {
            max = --min;
        }
        return ThreadLocalRandom.current().nextLong(min, max + 1L);
    }

    private void sleep(long delayMs) {
        try {
            this.clock.sleep(delayMs);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Retries interrupted", e);
        }
    }

    private void verifyAfterConstruction() {
        if (this.maxRetryTimeMs < 0L) {
            throw new IllegalArgumentException("Max retry time should be >= 0: " + this.maxRetryTimeMs);
        }
        if (this.initialRetryDelayMs < 0L) {
            throw new IllegalArgumentException("Initial retry delay should >= 0: " + this.initialRetryDelayMs);
        }
        if (this.multiplier < 1.0) {
            throw new IllegalArgumentException("Multiplier should be >= 1.0: " + this.multiplier);
        }
        if (this.jitterFactor < 0.0 || this.jitterFactor > 1.0) {
            throw new IllegalArgumentException("Jitter factor should be in [0.0, 1.0]: " + this.jitterFactor);
        }
        if (this.clock == null) {
            throw new IllegalArgumentException("Clock should not be null");
        }
    }

    private static boolean canRetryOn(Throwable error) {
        return error instanceof SessionExpiredException || error instanceof ServiceUnavailableException || ExponentialBackoff.isTransientError(error);
    }

    private static boolean isTransientError(Throwable error) {
        if (error instanceof TransientException) {
            String code = ((TransientException)error).code();
            return !"Neo.TransientError.Transaction.Terminated".equals(code) && !"Neo.TransientError.Transaction.LockClientStopped".equals(code);
        }
        return false;
    }
}

