/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.sasl.plain;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Provider;
import java.security.Security;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.KeySpec;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.Statement;
import java.util.Collections;
import java.util.HashMap;
import java.util.concurrent.ThreadLocalRandom;
import javax.security.auth.callback.CallbackHandler;
import javax.security.sasl.Sasl;
import javax.security.sasl.SaslClient;
import javax.security.sasl.SaslServer;
import javax.security.sasl.SaslServerFactory;
import org.hsqldb.jdbc.JDBCDataSource;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.security.auth.client.AuthenticationConfiguration;
import org.wildfly.security.auth.client.AuthenticationContext;
import org.wildfly.security.auth.client.ClientUtils;
import org.wildfly.security.auth.client.MatchRule;
import org.wildfly.security.password.Password;
import org.wildfly.security.password.PasswordFactory;
import org.wildfly.security.password.WildFlyElytronPasswordProvider;
import org.wildfly.security.password.interfaces.BCryptPassword;
import org.wildfly.security.password.interfaces.SaltedSimpleDigestPassword;
import org.wildfly.security.password.interfaces.ScramDigestPassword;
import org.wildfly.security.password.interfaces.SimpleDigestPassword;
import org.wildfly.security.password.spec.Encoding;
import org.wildfly.security.password.spec.EncryptablePasswordSpec;
import org.wildfly.security.password.spec.IteratedSaltedHashPasswordSpec;
import org.wildfly.security.password.spec.IteratedSaltedPasswordAlgorithmSpec;
import org.wildfly.security.password.spec.SaltedPasswordAlgorithmSpec;
import org.wildfly.security.sasl.SaslMechanismSelector;
import org.wildfly.security.sasl.plain.PlainSaslServer;
import org.wildfly.security.sasl.plain.PlainSaslServerFactory;
import org.wildfly.security.sasl.plain.WildFlyElytronSaslPlainProvider;
import org.wildfly.security.sasl.test.SaslServerBuilder;
import org.wildfly.security.sasl.test.SaslTestUtil;

public class PlainTest {
    private static final String PLAIN = "PLAIN";
    private static final Provider[] providers = new Provider[]{WildFlyElytronSaslPlainProvider.getInstance(), WildFlyElytronPasswordProvider.getInstance()};
    @ClassRule
    public static final DataSourceRule dataSourceRule = new DataSourceRule();

    @BeforeClass
    public static void registerProvider() {
        for (Provider provider : providers) {
            Security.insertProviderAt(provider, 1);
        }
    }

    @AfterClass
    public static void removeProvider() {
        for (Provider provider : providers) {
            Security.removeProvider(provider.getName());
        }
    }

    @Test
    public void testPolicyIndirect() throws Exception {
        HashMap<String, String> props = new HashMap<String, String>();
        SaslServer server = Sasl.createSaslServer(PLAIN, "TestProtocol", "TestServer", props, null);
        Assert.assertEquals(PlainSaslServer.class, server.getClass());
        props.put("javax.security.sasl.policy.noplaintext", Boolean.toString(true));
        server = Sasl.createSaslServer(PLAIN, "TestProtocol", "TestServer", props, null);
        Assert.assertNull((Object)server);
    }

    @Test
    public void testPolicyDirect() {
        SaslServerFactory factory = SaslTestUtil.obtainSaslServerFactory(PlainSaslServerFactory.class);
        Assert.assertNotNull((String)"SaslServerFactory not registered", (Object)factory);
        HashMap<String, String> props = new HashMap<String, String>();
        String[] mechanisms = factory.getMechanismNames(props);
        SaslTestUtil.assertSingleMechanism((String)PLAIN, (String[])mechanisms);
        props.put("javax.security.sasl.policy.noplaintext", Boolean.toString(true));
        mechanisms = factory.getMechanismNames(props);
        SaslTestUtil.assertNoMechanisms((String[])mechanisms);
    }

    @Test
    public void testSuccessfulExchange() throws Exception {
        SaslServer server = this.createSaslServer("George", "gpwd".toCharArray());
        this.testExchange(server, "George", "gpwd");
    }

    @Test
    public void testSuccessfulExchange_FileSystemRealm_HexEncoding() throws Exception {
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("George").setPassword("gpwd".toCharArray()).setModifiableRealm().setHashEncoding(Encoding.HEX).build();
        this.testExchange(server, "George", "gpwd");
    }

    @Test
    public void testSuccessfulExchange_FileSystemRealm_BcryptCharset() throws Exception {
        char[] actualPassword = "password\u5bc6\u7801".toCharArray();
        EncryptablePasswordSpec spec = new EncryptablePasswordSpec(actualPassword, (AlgorithmParameterSpec)new IteratedSaltedPasswordAlgorithmSpec(10, PlainTest.generateRandomSalt(16)), Charset.forName("gb2312"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("George").setPassword("bcrypt", (KeySpec)spec).setModifiableRealm().setHashCharset(Charset.forName("gb2312")).build();
        this.testExchange(server, "George", "password\u5bc6\u7801");
    }

    @Test
    public void testSuccessfulExchange_FileSystemRealm_ScramCharset() throws Exception {
        char[] actualPassword = "passwordHyv\u00e4\u00e4\u00e4".toCharArray();
        EncryptablePasswordSpec spec = new EncryptablePasswordSpec(actualPassword, (AlgorithmParameterSpec)new IteratedSaltedPasswordAlgorithmSpec(4096, PlainTest.generateRandomSalt(16)), Charset.forName("ISO-8859-1"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("George").setPassword("scram-sha-256", (KeySpec)spec).setModifiableRealm().setHashCharset(Charset.forName("ISO-8859-1")).build();
        this.testExchange(server, "George", "passwordHyv\u00e4\u00e4\u00e4");
    }

    @Test
    public void testSuccessfulExchange_FileSystemRealm_SimpleDigestCharset() throws Exception {
        char[] actualPassword = "\u043f\u0430\u0440\u043e\u043b\u044c".toCharArray();
        EncryptablePasswordSpec spec = new EncryptablePasswordSpec(actualPassword, null, Charset.forName("KOI8-R"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("George").setPassword("simple-digest-sha-512", (KeySpec)spec).setModifiableRealm().setHashCharset(Charset.forName("KOI8-R")).build();
        this.testExchange(server, "George", "\u043f\u0430\u0440\u043e\u043b\u044c");
    }

    @Test
    public void testSuccessfulExchange_FileSystemRealm_SimpleSaltedDigestCharset() throws Exception {
        char[] actualPassword = "\u043f\u0430\u0440\u043e\u043b\u044c".toCharArray();
        EncryptablePasswordSpec spec = new EncryptablePasswordSpec(actualPassword, (AlgorithmParameterSpec)new SaltedPasswordAlgorithmSpec(PlainTest.generateRandomSalt(16)), Charset.forName("KOI8-R"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("George").setPassword("password-salt-digest-sha-512", (KeySpec)spec).setModifiableRealm().setHashCharset(Charset.forName("KOI8-R")).build();
        this.testExchange(server, "George", "\u043f\u0430\u0440\u043e\u043b\u044c");
    }

    @Test
    public void testSuccessfulExchange_LegacySecurityRealm_HexEncoding() throws Exception {
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("elytron2").setRealmName("ManagementRealm").setDefaultRealmName("ManagementRealm").setPlainText(true).setLegacyInputStream(this.getClass().getResourceAsStream("charset.properties")).setHashEncoding(Encoding.HEX).build();
        this.testExchange(server, "elytron2", "passwd12#$");
    }

    @Test
    public void testSuccessfulExchange_LegacySecurityRealm_DigestCharset() throws Exception {
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName("elytron").setRealmName("ManagementRealm").setDefaultRealmName("ManagementRealm").setLegacyInputStream(this.getClass().getResourceAsStream("charset.properties")).setHashCharset(Charset.forName("gb2312")).setHashEncoding(Encoding.HEX).build();
        this.testExchange(server, "elytron", "password\u5bc6\u7801");
    }

    @Test
    public void testSuccessfulExchange_JdbcRealm_BcryptCharset() throws Exception {
        this.createEncryptTableWithSaltAndIteration("George", "passwordHyv\u00e4\u00e4\u00e4", 10, Charset.forName("ISO-8859-1"), "user_bcrypt_encoded_password", "bcrypt");
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setMapperAlgorithm("bcrypt").setPrincipalQuery("SELECT password, salt, iterationCount FROM user_bcrypt_encoded_password where name = ?").setDataSource(dataSourceRule.getDataSource()).setHashCharset(Charset.forName("ISO-8859-1")).build();
        this.testExchange(server, "George", "passwordHyv\u00e4\u00e4\u00e4");
    }

    @Test
    public void testSuccessfulExchange_JdbcRealm_ScramCharset() throws Exception {
        this.createEncryptTableWithSaltAndIteration("George", "passwordHyv\u00e4\u00e4\u00e4", 4096, Charset.forName("ISO-8859-1"), "user_scram_digest_password", "scram-sha-256");
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setMapperAlgorithm("scram-sha-256").setPrincipalQuery("SELECT password, salt, iterationCount FROM user_scram_digest_password where name = ?").setDataSource(dataSourceRule.getDataSource()).setHashCharset(Charset.forName("ISO-8859-1")).build();
        this.testExchange(server, "George", "passwordHyv\u00e4\u00e4\u00e4");
    }

    @Test
    public void testSuccessfulExchange_JdbcRealm_SimpleDigestCharset() throws Exception {
        this.createSimpleDigestPasswordTable("simple-digest-sha-512", "George", "password\u5bc6\u7801", Charset.forName("gb2312"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setMapperAlgorithm("simple-digest-sha-512").setPrincipalQuery("SELECT digest FROM user_simple_digest_password where name = ?").setDataSource(dataSourceRule.getDataSource()).setHashCharset(Charset.forName("gb2312")).build();
        this.testExchange(server, "George", "password\u5bc6\u7801");
    }

    @Test
    public void testSuccessfulExchange_JdbcRealm_SimpleSaltedDigestCharset() throws Exception {
        this.createSimpleSaltedDigestTable("George", "password\u5bc6\u7801", Charset.forName("gb2312"));
        SaslServer server = new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setMapperAlgorithm("password-salt-digest-sha-512").setPrincipalQuery("SELECT digest, salt FROM user_salted_simple_digest_password where name = ?").setDataSource(dataSourceRule.getDataSource()).setHashCharset(Charset.forName("gb2312")).build();
        this.testExchange(server, "George", "password\u5bc6\u7801");
    }

    @Test
    public void testBadPassword() throws Exception {
        SaslServer server = this.createSaslServer("George", "gpwd".toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler("George", "bad".toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, "George", "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)"George\u0000George\u0000bad", (Object)new String(message, StandardCharsets.UTF_8));
        try {
            server.evaluateResponse(message);
            Assert.fail((String)"Expection exception not thrown.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
    }

    @Test
    public void testBadUsername() throws Exception {
        SaslServer server = this.createSaslServer("Borris", "gpwd".toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler("George", "gpwd".toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, "George", "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)"George\u0000George\u0000gpwd", (Object)new String(message, StandardCharsets.UTF_8));
        try {
            server.evaluateResponse(message);
            Assert.fail((String)"Expection exception not thrown.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
    }

    @Test
    public void testSuccessfulExchange_NoAuthorization() throws Exception {
        SaslServer server = this.createSaslServer("George", "gpwd".toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler("George", "gpwd".toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, null, "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)"\u0000George\u0000gpwd", (Object)new String(message, StandardCharsets.UTF_8));
        server.evaluateResponse(message);
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
        Assert.assertEquals((Object)"George", (Object)server.getAuthorizationID());
    }

    @Test
    public void testSuccessfulExchange_DifferentAuthorizationID() throws Exception {
        SaslServer server = this.createSaslServer("George", "gpwd".toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler("George", "gpwd".toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, "Borris", "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)"Borris\u0000George\u0000gpwd", (Object)new String(message, StandardCharsets.UTF_8));
        try {
            server.evaluateResponse(message);
            Assert.fail((String)"Exception not thrown.");
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
    }

    @Test
    public void testSuccessfulExchange_NoNormalization() throws Exception {
        String username = "George\u00a8";
        String password = "password\u00a8";
        SaslServer server = this.createSaslServer(username, password.toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler(username, password.toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, username, "TestProtocol", "TestServer", Collections.singletonMap("org.wildfly.sasl.skip-normalization", "true"), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)(username + "\u0000" + username + "\u0000" + password), (Object)new String(message, StandardCharsets.UTF_8));
        server.evaluateResponse(message);
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
        Assert.assertEquals((Object)username, (Object)server.getAuthorizationID());
    }

    @Test
    public void testMaximumLength() throws Exception {
        SaslServer server = this.createSaslServer("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb".toCharArray());
        CallbackHandler clientCallback = this.createClientCallbackHandler("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb".toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        Assert.assertEquals((Object)"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\u0000aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\u0000bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb", (Object)new String(message, StandardCharsets.UTF_8));
        server.evaluateResponse(message);
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
        Assert.assertEquals((Object)"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", (Object)server.getAuthorizationID());
    }

    private void createSimpleSaltedDigestTable(String username, String password, Charset hashCharset) throws Exception {
        try (Connection connection = dataSourceRule.getDataSource().getConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("DROP TABLE IF EXISTS user_salted_simple_digest_password");
            statement.executeUpdate("CREATE TABLE user_salted_simple_digest_password ( id INTEGER IDENTITY, name VARCHAR(100), digest VARCHAR(100), salt OTHER)");
        }
        connection = dataSourceRule.getDataSource().getConnection();
        try (PreparedStatement preparedStatement = connection.prepareStatement("INSERT INTO user_salted_simple_digest_password (name, digest, salt) VALUES (?, ?, ?)");){
            byte[] salt = PlainTest.generateRandomSalt(16);
            EncryptablePasswordSpec eps = new EncryptablePasswordSpec(password.toCharArray(), (AlgorithmParameterSpec)new SaltedPasswordAlgorithmSpec(salt), hashCharset);
            PasswordFactory passwordFactory = PasswordFactory.getInstance((String)"password-salt-digest-sha-512");
            SaltedSimpleDigestPassword saltedPassword = (SaltedSimpleDigestPassword)passwordFactory.generatePassword((KeySpec)eps);
            preparedStatement.setString(1, username);
            preparedStatement.setString(2, ByteIterator.ofBytes((byte[])saltedPassword.getDigest()).base64Encode().drainToString());
            preparedStatement.setBytes(3, saltedPassword.getSalt());
            preparedStatement.execute();
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    private SimpleDigestPassword createSimpleDigestPasswordTable(String algorithm, String username, String password, Charset hashCharset) throws Exception {
        try (Connection connection = dataSourceRule.getDataSource().getConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("DROP TABLE IF EXISTS user_simple_digest_password");
            statement.executeUpdate("CREATE TABLE user_simple_digest_password ( id INTEGER IDENTITY, name VARCHAR(100), digest VARCHAR(100))");
        }
        connection = dataSourceRule.getDataSource().getConnection();
        try {
            SimpleDigestPassword simpleDigestPassword;
            block25: {
                PreparedStatement preparedStatement = connection.prepareStatement("INSERT INTO user_simple_digest_password (name, digest) VALUES (?, ?)");
                try {
                    EncryptablePasswordSpec eps = new EncryptablePasswordSpec(password.toCharArray(), null, hashCharset);
                    PasswordFactory passwordFactory = PasswordFactory.getInstance((String)algorithm);
                    SimpleDigestPassword tsdp = (SimpleDigestPassword)passwordFactory.generatePassword((KeySpec)eps);
                    preparedStatement.setString(1, username);
                    preparedStatement.setString(2, ByteIterator.ofBytes((byte[])tsdp.getDigest()).base64Encode().drainToString());
                    preparedStatement.execute();
                    simpleDigestPassword = tsdp;
                    if (preparedStatement == null) break block25;
                }
                catch (Throwable throwable) {
                    if (preparedStatement != null) {
                        try {
                            preparedStatement.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                preparedStatement.close();
            }
            return simpleDigestPassword;
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    private void createEncryptTableWithSaltAndIteration(String username, String password, int iterationCount, Charset hashCharset, String tableName, String algorithm) throws Exception {
        try (Connection connection = dataSourceRule.getDataSource().getConnection();
             Statement statement = connection.createStatement();){
            statement.executeUpdate("DROP TABLE IF EXISTS " + tableName);
            statement.executeUpdate("CREATE TABLE " + tableName + " ( id INTEGER IDENTITY, name VARCHAR(100), password VARCHAR(100), salt OTHER, iterationCount INTEGER)");
        }
        connection = dataSourceRule.getDataSource().getConnection();
        try (PreparedStatement preparedStatement = connection.prepareStatement("INSERT INTO " + tableName + " (name, password, salt, iterationCount) VALUES (?, ?, ?, ?)");){
            PasswordFactory passwordFactory = PasswordFactory.getInstance((String)algorithm);
            EncryptablePasswordSpec spec = new EncryptablePasswordSpec(password.toCharArray(), (AlgorithmParameterSpec)new IteratedSaltedPasswordAlgorithmSpec(iterationCount, PlainTest.generateRandomSalt(16)), hashCharset);
            preparedStatement.setString(1, username);
            if (algorithm.equals("bcrypt")) {
                BCryptPassword bcryptPassword = (BCryptPassword)passwordFactory.generatePassword((KeySpec)spec);
                preparedStatement.setString(2, ByteIterator.ofBytes((byte[])bcryptPassword.getHash()).base64Encode().drainToString());
                preparedStatement.setBytes(3, bcryptPassword.getSalt());
                preparedStatement.setInt(4, bcryptPassword.getIterationCount());
            } else {
                ScramDigestPassword scramPassword = (ScramDigestPassword)passwordFactory.generatePassword((KeySpec)spec);
                IteratedSaltedHashPasswordSpec keySpec = (IteratedSaltedHashPasswordSpec)passwordFactory.getKeySpec((Password)scramPassword, IteratedSaltedHashPasswordSpec.class);
                preparedStatement.setString(2, ByteIterator.ofBytes((byte[])keySpec.getHash()).base64Encode().drainToString());
                preparedStatement.setBytes(3, keySpec.getSalt());
                preparedStatement.setInt(4, keySpec.getIterationCount());
            }
            preparedStatement.execute();
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    private void testExchange(SaslServer server, String username, String password) throws Exception {
        CallbackHandler clientCallback = this.createClientCallbackHandler(username, password.toCharArray());
        SaslClient client = Sasl.createSaslClient(new String[]{PLAIN}, username, "TestProtocol", "TestServer", Collections.emptyMap(), clientCallback);
        Assert.assertFalse((boolean)server.isComplete());
        Assert.assertFalse((boolean)client.isComplete());
        Assert.assertTrue((boolean)client.hasInitialResponse());
        byte[] message = client.evaluateChallenge(new byte[0]);
        server.evaluateResponse(message);
        Assert.assertTrue((boolean)server.isComplete());
        Assert.assertTrue((boolean)client.isComplete());
        Assert.assertEquals((Object)username, (Object)server.getAuthorizationID());
    }

    private static byte[] generateRandomSalt(int saltSize) {
        byte[] randomSalt = new byte[saltSize];
        ThreadLocalRandom.current().nextBytes(randomSalt);
        return randomSalt;
    }

    private SaslServer createSaslServer(String expectedUsername, char[] expectedPassword) throws Exception {
        return new SaslServerBuilder(PlainSaslServerFactory.class, PLAIN).setUserName(expectedUsername).setPassword(expectedPassword).build();
    }

    private CallbackHandler createClientCallbackHandler(String username, char[] password) throws Exception {
        AuthenticationContext context = AuthenticationContext.empty().with(MatchRule.ALL, AuthenticationConfiguration.empty().useName(username).usePassword(password).setSaslMechanismSelector(SaslMechanismSelector.NONE.addMechanism(PLAIN)));
        return ClientUtils.getCallbackHandler(new URI("doesnot://matter?"), context);
    }

    static class DataSourceRule
    implements TestRule {
        private final Provider provider = WildFlyElytronPasswordProvider.getInstance();
        private JDBCDataSource dataSource;

        DataSourceRule() {
        }

        public org.junit.runners.model.Statement apply(final org.junit.runners.model.Statement current, Description description) {
            return new org.junit.runners.model.Statement(){

                public void evaluate() throws Throwable {
                    Security.addProvider(provider);
                    dataSource = new JDBCDataSource();
                    dataSource.setDatabase("mem:elytron-jdbc-realm-test");
                    dataSource.setUser("sa");
                    try {
                        current.evaluate();
                    }
                    catch (Exception e) {
                        throw e;
                    }
                    finally {
                        Security.removeProvider(provider.getName());
                    }
                }
            };
        }

        public JDBCDataSource getDataSource() {
            return this.dataSource;
        }
    }
}

