/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a data quality result.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataQualityResult implements SdkPojo, Serializable,
        ToCopyableBuilder<DataQualityResult.Builder, DataQualityResult> {
    private static final SdkField<String> RESULT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResultId").getter(getter(DataQualityResult::resultId)).setter(setter(Builder::resultId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResultId").build()).build();

    private static final SdkField<Double> SCORE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Score")
            .getter(getter(DataQualityResult::score)).setter(setter(Builder::score))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Score").build()).build();

    private static final SdkField<DataSource> DATA_SOURCE_FIELD = SdkField.<DataSource> builder(MarshallingType.SDK_POJO)
            .memberName("DataSource").getter(getter(DataQualityResult::dataSource)).setter(setter(Builder::dataSource))
            .constructor(DataSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataSource").build()).build();

    private static final SdkField<String> RULESET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RulesetName").getter(getter(DataQualityResult::rulesetName)).setter(setter(Builder::rulesetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RulesetName").build()).build();

    private static final SdkField<String> EVALUATION_CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvaluationContext").getter(getter(DataQualityResult::evaluationContext))
            .setter(setter(Builder::evaluationContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationContext").build()).build();

    private static final SdkField<Instant> STARTED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedOn").getter(getter(DataQualityResult::startedOn)).setter(setter(Builder::startedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedOn").build()).build();

    private static final SdkField<Instant> COMPLETED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletedOn").getter(getter(DataQualityResult::completedOn)).setter(setter(Builder::completedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletedOn").build()).build();

    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(DataQualityResult::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<String> JOB_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobRunId").getter(getter(DataQualityResult::jobRunId)).setter(setter(Builder::jobRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobRunId").build()).build();

    private static final SdkField<String> RULESET_EVALUATION_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RulesetEvaluationRunId").getter(getter(DataQualityResult::rulesetEvaluationRunId))
            .setter(setter(Builder::rulesetEvaluationRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RulesetEvaluationRunId").build())
            .build();

    private static final SdkField<List<DataQualityRuleResult>> RULE_RESULTS_FIELD = SdkField
            .<List<DataQualityRuleResult>> builder(MarshallingType.LIST)
            .memberName("RuleResults")
            .getter(getter(DataQualityResult::ruleResults))
            .setter(setter(Builder::ruleResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleResults").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataQualityRuleResult> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataQualityRuleResult::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESULT_ID_FIELD, SCORE_FIELD,
            DATA_SOURCE_FIELD, RULESET_NAME_FIELD, EVALUATION_CONTEXT_FIELD, STARTED_ON_FIELD, COMPLETED_ON_FIELD,
            JOB_NAME_FIELD, JOB_RUN_ID_FIELD, RULESET_EVALUATION_RUN_ID_FIELD, RULE_RESULTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resultId;

    private final Double score;

    private final DataSource dataSource;

    private final String rulesetName;

    private final String evaluationContext;

    private final Instant startedOn;

    private final Instant completedOn;

    private final String jobName;

    private final String jobRunId;

    private final String rulesetEvaluationRunId;

    private final List<DataQualityRuleResult> ruleResults;

    private DataQualityResult(BuilderImpl builder) {
        this.resultId = builder.resultId;
        this.score = builder.score;
        this.dataSource = builder.dataSource;
        this.rulesetName = builder.rulesetName;
        this.evaluationContext = builder.evaluationContext;
        this.startedOn = builder.startedOn;
        this.completedOn = builder.completedOn;
        this.jobName = builder.jobName;
        this.jobRunId = builder.jobRunId;
        this.rulesetEvaluationRunId = builder.rulesetEvaluationRunId;
        this.ruleResults = builder.ruleResults;
    }

    /**
     * <p>
     * A unique result ID for the data quality result.
     * </p>
     * 
     * @return A unique result ID for the data quality result.
     */
    public final String resultId() {
        return resultId;
    }

    /**
     * <p>
     * An aggregate data quality score. Represents the ratio of rules that passed to the total number of rules.
     * </p>
     * 
     * @return An aggregate data quality score. Represents the ratio of rules that passed to the total number of rules.
     */
    public final Double score() {
        return score;
    }

    /**
     * <p>
     * The table associated with the data quality result, if any.
     * </p>
     * 
     * @return The table associated with the data quality result, if any.
     */
    public final DataSource dataSource() {
        return dataSource;
    }

    /**
     * <p>
     * The name of the ruleset associated with the data quality result.
     * </p>
     * 
     * @return The name of the ruleset associated with the data quality result.
     */
    public final String rulesetName() {
        return rulesetName;
    }

    /**
     * <p>
     * In the context of a job in Glue Studio, each node in the canvas is typically assigned some sort of name and data
     * quality nodes will have names. In the case of multiple nodes, the <code>evaluationContext</code> can
     * differentiate the nodes.
     * </p>
     * 
     * @return In the context of a job in Glue Studio, each node in the canvas is typically assigned some sort of name
     *         and data quality nodes will have names. In the case of multiple nodes, the <code>evaluationContext</code>
     *         can differentiate the nodes.
     */
    public final String evaluationContext() {
        return evaluationContext;
    }

    /**
     * <p>
     * The date and time when this data quality run started.
     * </p>
     * 
     * @return The date and time when this data quality run started.
     */
    public final Instant startedOn() {
        return startedOn;
    }

    /**
     * <p>
     * The date and time when this data quality run completed.
     * </p>
     * 
     * @return The date and time when this data quality run completed.
     */
    public final Instant completedOn() {
        return completedOn;
    }

    /**
     * <p>
     * The job name associated with the data quality result, if any.
     * </p>
     * 
     * @return The job name associated with the data quality result, if any.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The job run ID associated with the data quality result, if any.
     * </p>
     * 
     * @return The job run ID associated with the data quality result, if any.
     */
    public final String jobRunId() {
        return jobRunId;
    }

    /**
     * <p>
     * The unique run ID for the ruleset evaluation for this data quality result.
     * </p>
     * 
     * @return The unique run ID for the ruleset evaluation for this data quality result.
     */
    public final String rulesetEvaluationRunId() {
        return rulesetEvaluationRunId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RuleResults property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRuleResults() {
        return ruleResults != null && !(ruleResults instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRuleResults} method.
     * </p>
     * 
     * @return A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
     */
    public final List<DataQualityRuleResult> ruleResults() {
        return ruleResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resultId());
        hashCode = 31 * hashCode + Objects.hashCode(score());
        hashCode = 31 * hashCode + Objects.hashCode(dataSource());
        hashCode = 31 * hashCode + Objects.hashCode(rulesetName());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationContext());
        hashCode = 31 * hashCode + Objects.hashCode(startedOn());
        hashCode = 31 * hashCode + Objects.hashCode(completedOn());
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(jobRunId());
        hashCode = 31 * hashCode + Objects.hashCode(rulesetEvaluationRunId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRuleResults() ? ruleResults() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataQualityResult)) {
            return false;
        }
        DataQualityResult other = (DataQualityResult) obj;
        return Objects.equals(resultId(), other.resultId()) && Objects.equals(score(), other.score())
                && Objects.equals(dataSource(), other.dataSource()) && Objects.equals(rulesetName(), other.rulesetName())
                && Objects.equals(evaluationContext(), other.evaluationContext())
                && Objects.equals(startedOn(), other.startedOn()) && Objects.equals(completedOn(), other.completedOn())
                && Objects.equals(jobName(), other.jobName()) && Objects.equals(jobRunId(), other.jobRunId())
                && Objects.equals(rulesetEvaluationRunId(), other.rulesetEvaluationRunId())
                && hasRuleResults() == other.hasRuleResults() && Objects.equals(ruleResults(), other.ruleResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataQualityResult").add("ResultId", resultId()).add("Score", score())
                .add("DataSource", dataSource()).add("RulesetName", rulesetName()).add("EvaluationContext", evaluationContext())
                .add("StartedOn", startedOn()).add("CompletedOn", completedOn()).add("JobName", jobName())
                .add("JobRunId", jobRunId()).add("RulesetEvaluationRunId", rulesetEvaluationRunId())
                .add("RuleResults", hasRuleResults() ? ruleResults() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResultId":
            return Optional.ofNullable(clazz.cast(resultId()));
        case "Score":
            return Optional.ofNullable(clazz.cast(score()));
        case "DataSource":
            return Optional.ofNullable(clazz.cast(dataSource()));
        case "RulesetName":
            return Optional.ofNullable(clazz.cast(rulesetName()));
        case "EvaluationContext":
            return Optional.ofNullable(clazz.cast(evaluationContext()));
        case "StartedOn":
            return Optional.ofNullable(clazz.cast(startedOn()));
        case "CompletedOn":
            return Optional.ofNullable(clazz.cast(completedOn()));
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "JobRunId":
            return Optional.ofNullable(clazz.cast(jobRunId()));
        case "RulesetEvaluationRunId":
            return Optional.ofNullable(clazz.cast(rulesetEvaluationRunId()));
        case "RuleResults":
            return Optional.ofNullable(clazz.cast(ruleResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataQualityResult, T> g) {
        return obj -> g.apply((DataQualityResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataQualityResult> {
        /**
         * <p>
         * A unique result ID for the data quality result.
         * </p>
         * 
         * @param resultId
         *        A unique result ID for the data quality result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultId(String resultId);

        /**
         * <p>
         * An aggregate data quality score. Represents the ratio of rules that passed to the total number of rules.
         * </p>
         * 
         * @param score
         *        An aggregate data quality score. Represents the ratio of rules that passed to the total number of
         *        rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder score(Double score);

        /**
         * <p>
         * The table associated with the data quality result, if any.
         * </p>
         * 
         * @param dataSource
         *        The table associated with the data quality result, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSource(DataSource dataSource);

        /**
         * <p>
         * The table associated with the data quality result, if any.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataSource.Builder} avoiding the need to
         * create one manually via {@link DataSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataSource.Builder#build()} is called immediately and its result
         * is passed to {@link #dataSource(DataSource)}.
         * 
         * @param dataSource
         *        a consumer that will call methods on {@link DataSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataSource(DataSource)
         */
        default Builder dataSource(Consumer<DataSource.Builder> dataSource) {
            return dataSource(DataSource.builder().applyMutation(dataSource).build());
        }

        /**
         * <p>
         * The name of the ruleset associated with the data quality result.
         * </p>
         * 
         * @param rulesetName
         *        The name of the ruleset associated with the data quality result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rulesetName(String rulesetName);

        /**
         * <p>
         * In the context of a job in Glue Studio, each node in the canvas is typically assigned some sort of name and
         * data quality nodes will have names. In the case of multiple nodes, the <code>evaluationContext</code> can
         * differentiate the nodes.
         * </p>
         * 
         * @param evaluationContext
         *        In the context of a job in Glue Studio, each node in the canvas is typically assigned some sort of
         *        name and data quality nodes will have names. In the case of multiple nodes, the
         *        <code>evaluationContext</code> can differentiate the nodes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationContext(String evaluationContext);

        /**
         * <p>
         * The date and time when this data quality run started.
         * </p>
         * 
         * @param startedOn
         *        The date and time when this data quality run started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedOn(Instant startedOn);

        /**
         * <p>
         * The date and time when this data quality run completed.
         * </p>
         * 
         * @param completedOn
         *        The date and time when this data quality run completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completedOn(Instant completedOn);

        /**
         * <p>
         * The job name associated with the data quality result, if any.
         * </p>
         * 
         * @param jobName
         *        The job name associated with the data quality result, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The job run ID associated with the data quality result, if any.
         * </p>
         * 
         * @param jobRunId
         *        The job run ID associated with the data quality result, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRunId(String jobRunId);

        /**
         * <p>
         * The unique run ID for the ruleset evaluation for this data quality result.
         * </p>
         * 
         * @param rulesetEvaluationRunId
         *        The unique run ID for the ruleset evaluation for this data quality result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rulesetEvaluationRunId(String rulesetEvaluationRunId);

        /**
         * <p>
         * A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
         * </p>
         * 
         * @param ruleResults
         *        A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleResults(Collection<DataQualityRuleResult> ruleResults);

        /**
         * <p>
         * A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
         * </p>
         * 
         * @param ruleResults
         *        A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleResults(DataQualityRuleResult... ruleResults);

        /**
         * <p>
         * A list of <code>DataQualityRuleResult</code> objects representing the results for each rule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.DataQualityRuleResult.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.glue.model.DataQualityRuleResult#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.DataQualityRuleResult.Builder#build()} is called
         * immediately and its result is passed to {@link #ruleResults(List<DataQualityRuleResult>)}.
         * 
         * @param ruleResults
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.DataQualityRuleResult.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleResults(java.util.Collection<DataQualityRuleResult>)
         */
        Builder ruleResults(Consumer<DataQualityRuleResult.Builder>... ruleResults);
    }

    static final class BuilderImpl implements Builder {
        private String resultId;

        private Double score;

        private DataSource dataSource;

        private String rulesetName;

        private String evaluationContext;

        private Instant startedOn;

        private Instant completedOn;

        private String jobName;

        private String jobRunId;

        private String rulesetEvaluationRunId;

        private List<DataQualityRuleResult> ruleResults = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataQualityResult model) {
            resultId(model.resultId);
            score(model.score);
            dataSource(model.dataSource);
            rulesetName(model.rulesetName);
            evaluationContext(model.evaluationContext);
            startedOn(model.startedOn);
            completedOn(model.completedOn);
            jobName(model.jobName);
            jobRunId(model.jobRunId);
            rulesetEvaluationRunId(model.rulesetEvaluationRunId);
            ruleResults(model.ruleResults);
        }

        public final String getResultId() {
            return resultId;
        }

        public final void setResultId(String resultId) {
            this.resultId = resultId;
        }

        @Override
        public final Builder resultId(String resultId) {
            this.resultId = resultId;
            return this;
        }

        public final Double getScore() {
            return score;
        }

        public final void setScore(Double score) {
            this.score = score;
        }

        @Override
        public final Builder score(Double score) {
            this.score = score;
            return this;
        }

        public final DataSource.Builder getDataSource() {
            return dataSource != null ? dataSource.toBuilder() : null;
        }

        public final void setDataSource(DataSource.BuilderImpl dataSource) {
            this.dataSource = dataSource != null ? dataSource.build() : null;
        }

        @Override
        public final Builder dataSource(DataSource dataSource) {
            this.dataSource = dataSource;
            return this;
        }

        public final String getRulesetName() {
            return rulesetName;
        }

        public final void setRulesetName(String rulesetName) {
            this.rulesetName = rulesetName;
        }

        @Override
        public final Builder rulesetName(String rulesetName) {
            this.rulesetName = rulesetName;
            return this;
        }

        public final String getEvaluationContext() {
            return evaluationContext;
        }

        public final void setEvaluationContext(String evaluationContext) {
            this.evaluationContext = evaluationContext;
        }

        @Override
        public final Builder evaluationContext(String evaluationContext) {
            this.evaluationContext = evaluationContext;
            return this;
        }

        public final Instant getStartedOn() {
            return startedOn;
        }

        public final void setStartedOn(Instant startedOn) {
            this.startedOn = startedOn;
        }

        @Override
        public final Builder startedOn(Instant startedOn) {
            this.startedOn = startedOn;
            return this;
        }

        public final Instant getCompletedOn() {
            return completedOn;
        }

        public final void setCompletedOn(Instant completedOn) {
            this.completedOn = completedOn;
        }

        @Override
        public final Builder completedOn(Instant completedOn) {
            this.completedOn = completedOn;
            return this;
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final String getJobRunId() {
            return jobRunId;
        }

        public final void setJobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
        }

        @Override
        public final Builder jobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
            return this;
        }

        public final String getRulesetEvaluationRunId() {
            return rulesetEvaluationRunId;
        }

        public final void setRulesetEvaluationRunId(String rulesetEvaluationRunId) {
            this.rulesetEvaluationRunId = rulesetEvaluationRunId;
        }

        @Override
        public final Builder rulesetEvaluationRunId(String rulesetEvaluationRunId) {
            this.rulesetEvaluationRunId = rulesetEvaluationRunId;
            return this;
        }

        public final List<DataQualityRuleResult.Builder> getRuleResults() {
            List<DataQualityRuleResult.Builder> result = DataQualityRuleResultsCopier.copyToBuilder(this.ruleResults);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRuleResults(Collection<DataQualityRuleResult.BuilderImpl> ruleResults) {
            this.ruleResults = DataQualityRuleResultsCopier.copyFromBuilder(ruleResults);
        }

        @Override
        public final Builder ruleResults(Collection<DataQualityRuleResult> ruleResults) {
            this.ruleResults = DataQualityRuleResultsCopier.copy(ruleResults);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleResults(DataQualityRuleResult... ruleResults) {
            ruleResults(Arrays.asList(ruleResults));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ruleResults(Consumer<DataQualityRuleResult.Builder>... ruleResults) {
            ruleResults(Stream.of(ruleResults).map(c -> DataQualityRuleResult.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public DataQualityResult build() {
            return new DataQualityResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
