/*
 *
 * suco: Mini IoC framework a-la-guice style for GWT
 *
 * (c) 2009 The suco development team (see CREDITS for details)
 * This file is part of emite.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package com.calclab.suco.examples.events;

import static com.calclab.suco.testing.events.Eventito.anyListener2;
import static com.calclab.suco.testing.events.Eventito.fire;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Test;
import org.mockito.Mockito;

import com.calclab.suco.client.events.Event2;
import com.calclab.suco.client.events.Listener2;
import com.calclab.suco.testing.events.MockedListener2;

/**
 * This example show how to test a publisher and a subscriber using the
 * excellent Mockito library.
 * 
 * Here we use a two params event (Event2 classes and Listener2 classes). You
 * can use Event and Listener for one param event, and Event0 and Listener0 for
 * no param event
 * 
 */
public class TestingSucoEventsExample {
}

/**
 * A convenience publisher class to ilustrate the tests
 */
class MessagePublisher {
    private final Event2<String, String> onMessage = new Event2<String, String>("onMessage");

    public void onMessage(final Listener2<String, String> listener) {
	onMessage.add(listener);
    }

    public void sendMessage(final String user, final String message) {
	onMessage.fire(user, message);
    }
}

/**
 * We should verify that when we call messagePublisher.sendMessage(...) the
 * listeners are informed. This is performed with the MockedListener class test
 * helper.
 */
class MessagePublisherTest {
    @Test
    public void messagePublishersShouldFireMessages() {
	final MessagePublisher messagePublisher = new MessagePublisher();
	final MockedListener2<String, String> listener = new MockedListener2<String, String>();
	messagePublisher.onMessage(listener);
	messagePublisher.sendMessage("aretha", "this is the message");
	assertTrue(listener.isCalledWithEquals("aretha", "this is the message"));
    }
}

/**
 * The convenience subscriber class to ilustrate the tests. The goal is to test
 * the code inside the annonymous class (that implements Listener2). So the
 * question is, how can I execute that code? or, how can I fire the onMessage
 * event if messagePublisher is a mock?
 */
class MessageSubscriber {
    private String lastMessage;
    private String lastUser;

    public MessageSubscriber(final MessagePublisher messagePublisher) {
	messagePublisher.onMessage(new Listener2<String, String>() {
	    public void onEvent(final String user, final String message) {
		lastMessage = message;
		lastUser = user;
	    }
	});
    }

    public String getLastMessage() {
	return lastMessage;
    }

    public String getLastUser() {
	return lastUser;
    }
}

/**
 * We should verify the reaction of the subscriptor when the publisher fires the
 * event. In this case we could create a real messagePublisher and call
 * "sendMessage" but this approach has two disadvantages:
 * 
 * First: usually publishers don't have a method to fire an event
 * 
 * Second: we only want to test the listener part. If we use a real
 * messagePublisher and this messagePublisher have a problem in its
 * "sendMessage" method, we could have an error in this test (the test of the
 * listener) even when the error is in the messagePublisher side.
 * 
 * Thats why we mock everything except the class to be tested
 */
class MessageSubscriberTest {
    @Test
    public void listenersShouldListenToProducers() {
	final MessagePublisher messagePublisher = Mockito.mock(MessagePublisher.class);
	final MessageSubscriber listener = new MessageSubscriber(messagePublisher);

	fire("simone", "send this!").when(messagePublisher).onMessage(anyListener2(String.class, String.class));
	assertEquals("send this message", listener.getLastMessage());
    }
}
