/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.graph;

import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.Graphs;
import com.google.common.graph.MutableGraph;
import com.google.common.graph.MutableNetwork;
import com.google.common.graph.Network;
import com.google.common.graph.NetworkBuilder;
import com.google.common.truth.Truth;
import java.util.Collection;
import java.util.Set;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class GraphsTest {
    private static final Integer N1 = 1;
    private static final Integer N2 = 2;
    private static final Integer N3 = 3;
    private static final Integer N4 = 4;
    private static final String E11 = "1-1";
    private static final String E11_A = "1-1a";
    private static final String E12 = "1-2";
    private static final String E12_A = "1-2a";
    private static final String E12_B = "1-2b";
    private static final String E21 = "2-1";
    private static final String E22 = "2-2";
    private static final String E13 = "1-3";
    private static final String E44 = "4-4";
    private static final int NODE_COUNT = 20;
    private static final int EDGE_COUNT = 20;
    private static final String ERROR_PARALLEL_EDGE = "connected by a different edge";
    private static final String ERROR_NEGATIVE_NODE_COUNT = "expected number of nodes can't be negative";
    private static final String ERROR_NEGATIVE_EDGE_COUNT = "expected number of edges can't be negative";
    private static final String ERROR_ADDED_PARALLEL_EDGE = "Should not be allowed to add a parallel edge.";
    private static final String ERROR_ADDED_SELF_LOOP = "Should not be allowed to add a self-loop edge.";
    static final String ERROR_SELF_LOOP = "self-loops are not allowed";

    @Test
    public void oppositeNode_basic() {
        ImmutableList testNetworks = ImmutableList.of((Object)NetworkBuilder.directed().build(), (Object)NetworkBuilder.undirected().build());
        for (MutableNetwork graph : testNetworks) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12, (Object)N2))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_parallelEdge() {
        ImmutableList testNetworks = ImmutableList.of((Object)NetworkBuilder.directed().allowsParallelEdges(true).build(), (Object)NetworkBuilder.undirected().allowsParallelEdges(true).build());
        for (MutableNetwork graph : testNetworks) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            graph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12, (Object)N2))).isEqualTo((Object)N1);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12_A, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E12_A, (Object)N2))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_selfLoop() {
        ImmutableList testNetworks = ImmutableList.of((Object)NetworkBuilder.directed().build(), (Object)NetworkBuilder.undirected().build());
        for (MutableNetwork graph : testNetworks) {
            graph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Network)graph, (Object)E11, (Object)N1))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_nodeNotIncident() {
        ImmutableList testNetworks = ImmutableList.of((Object)NetworkBuilder.directed().build(), (Object)NetworkBuilder.undirected().build());
        for (MutableNetwork graph : testNetworks) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            graph.addEdge((Object)E13, (Object)N1, (Object)N3);
            try {
                Graphs.oppositeNode((Network)graph, (Object)E12, (Object)N3);
                Assert.fail((String)"Should have rejected oppositeNode() called without a node incident to edge");
            }
            catch (IllegalArgumentException illegalArgumentException) {}
        }
    }

    @Test
    public void parallelEdges_directed() {
        MutableNetwork directedGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)directedGraph, (Object)E12)).containsExactly(new Object[]{E12_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)directedGraph, (Object)E12_A)).containsExactly(new Object[]{E12});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)directedGraph, (Object)E21)).isEmpty();
    }

    @Test
    public void parallelEdges_selfLoop_directed() {
        MutableNetwork directedGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)directedGraph, (Object)E11)).containsExactly(new Object[]{E11_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)directedGraph, (Object)E11_A)).containsExactly(new Object[]{E11});
    }

    @Test
    public void parallelEdges_undirected() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)undirectedGraph, (Object)E12)).containsExactly(new Object[]{E12_A, E21});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)undirectedGraph, (Object)E12_A)).containsExactly(new Object[]{E12, E21});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)undirectedGraph, (Object)E21)).containsExactly(new Object[]{E12, E12_A});
    }

    @Test
    public void parallelEdges_selfLoop_undirected() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)undirectedGraph, (Object)E11)).containsExactly(new Object[]{E11_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Network)undirectedGraph, (Object)E11_A)).containsExactly(new Object[]{E11});
    }

    @Test
    public void parallelEdges_unmodifiableView() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        Set parallelEdges = Graphs.parallelEdges((Network)undirectedGraph, (Object)E12);
        Truth.assertThat((Iterable)parallelEdges).containsExactly(new Object[]{E12_A});
        undirectedGraph.addEdge((Object)E12_B, (Object)N1, (Object)N2);
        Truth.assertThat((Iterable)parallelEdges).containsExactly(new Object[]{E12_A, E12_B});
        try {
            parallelEdges.add(E21);
            Assert.fail((String)"Set returned by parallelEdges() should be unmodifiable");
        }
        catch (UnsupportedOperationException unsupportedOperationException) {
            // empty catch block
        }
    }

    @Test
    public void createDirected() {
        MutableNetwork directedGraph = NetworkBuilder.directed().build();
        Truth.assertThat((Iterable)directedGraph.nodes()).isEmpty();
        Truth.assertThat((Iterable)directedGraph.edges()).isEmpty();
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
        try {
            directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1)).isTrue();
    }

    @Test
    public void createUndirected() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().build();
        Truth.assertThat((Iterable)undirectedGraph.nodes()).isEmpty();
        Truth.assertThat((Iterable)undirectedGraph.edges()).isEmpty();
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        try {
            undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        try {
            undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1)).isTrue();
    }

    @Test
    public void createDirected_multigraph() {
        MutableNetwork directedMultigraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        Truth.assertThat((Boolean)directedMultigraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)directedMultigraph.addEdge((Object)E12_A, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedMultigraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12, (Object)E12_A));
        Truth.assertThat((Iterable)directedMultigraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_multigraph() {
        MutableNetwork undirectedMultigraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E12_A, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E21, (Object)N2, (Object)N1)).isTrue();
        Truth.assertThat((Iterable)undirectedMultigraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12, (Object)E12_A, (Object)E21));
    }

    @Test
    public void createDirected_expectedNodeCount() {
        MutableNetwork directedGraph = NetworkBuilder.directed().expectedNodeCount(20).build();
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_expectedNodeCount() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().expectedNodeCount(20).build();
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
    }

    @Test
    public void builder_expectedNodeCount_negative() {
        try {
            NetworkBuilder.directed().expectedNodeCount(-1);
            Assert.fail((String)ERROR_NEGATIVE_NODE_COUNT);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_NEGATIVE_NODE_COUNT);
        }
    }

    @Test
    public void createDirected_expectedEdgeCount() {
        MutableNetwork directedGraph = NetworkBuilder.directed().expectedEdgeCount(20).build();
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_expectedEdgeCount() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().expectedEdgeCount(20).build();
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
    }

    @Test
    public void builder_expectedEdgeCount_negative() {
        try {
            NetworkBuilder.directed().expectedEdgeCount(-1);
            Assert.fail((String)ERROR_NEGATIVE_EDGE_COUNT);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_NEGATIVE_EDGE_COUNT);
        }
    }

    @Test
    public void createDirected_noSelfLoops() {
        MutableNetwork directedGraph = NetworkBuilder.directed().allowsSelfLoops(false).build();
        try {
            directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Assert.fail((String)ERROR_ADDED_SELF_LOOP);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_SELF_LOOP);
        }
    }

    @Test
    public void createUndirected_noSelfLoops() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().allowsSelfLoops(false).build();
        try {
            undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Assert.fail((String)ERROR_ADDED_SELF_LOOP);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_SELF_LOOP);
        }
    }

    @Test
    public void addEdge_nullGraph() {
        try {
            Graphs.addEdge(null, (Object)E11, (Iterable)ImmutableSet.of((Object)N1));
            Assert.fail((String)"Should have rejected null graph");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_nullNodes() {
        try {
            Graphs.addEdge((MutableNetwork)NetworkBuilder.directed().build(), (Object)E11, null);
            Assert.fail((String)"Should have rejected null nodes");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_tooManyNodes() {
        try {
            Graphs.addEdge((MutableNetwork)NetworkBuilder.directed().build(), (Object)E11, (Iterable)ImmutableSet.of((Object)N1, (Object)N2, (Object)N3));
            Assert.fail((String)"Should have rejected adding an edge to a Graph with > 2 nodes");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_notEnoughNodes() {
        try {
            Graphs.addEdge((MutableNetwork)NetworkBuilder.directed().build(), (Object)E11, (Iterable)ImmutableSet.of());
            Assert.fail((String)"Should have rejected adding an edge to a Graph with < 1 nodes");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_selfLoop() {
        MutableNetwork directedGraph = NetworkBuilder.directed().build();
        Truth.assertThat((Boolean)Graphs.addEdge((MutableNetwork)directedGraph, (Object)E11, (Iterable)ImmutableSet.of((Object)N1))).isTrue();
        Truth.assertThat((Iterable)directedGraph.edges()).containsExactly(new Object[]{E11});
        Truth.assertThat((Iterable)directedGraph.nodes()).containsExactly(new Object[]{N1});
        Truth.assertThat((Iterable)directedGraph.incidentNodes((Object)E11)).containsExactly(new Object[]{N1});
    }

    @Test
    public void addEdge_basic() {
        MutableNetwork directedGraph = NetworkBuilder.directed().build();
        Truth.assertThat((Boolean)Graphs.addEdge((MutableNetwork)directedGraph, (Object)E12, (Iterable)ImmutableSet.of((Object)N1, (Object)N2))).isTrue();
        Truth.assertThat((Iterable)directedGraph.edges()).containsExactly(new Object[]{E12});
        Truth.assertThat((Iterable)directedGraph.nodes()).containsExactly(new Object[]{N1, N2}).inOrder();
        Truth.assertThat((Iterable)directedGraph.incidentNodes((Object)E12)).containsExactly(new Object[]{N1, N2}).inOrder();
    }

    @Test
    public void copyOf_nullArgument() {
        try {
            Graphs.copyOf(null);
            Assert.fail((String)"Should have rejected a null graph");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void copyOf_directedGraph() {
        MutableGraph<Integer> directedGraph = GraphsTest.buildDirectedTestGraph();
        MutableGraph copy = Graphs.copyOf(directedGraph);
        Truth.assertThat((Object)copy).isEqualTo(directedGraph);
    }

    @Test
    public void copyOf_undirectedGraph() {
        MutableGraph<Integer> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        MutableGraph copy = Graphs.copyOf(undirectedGraph);
        Truth.assertThat((Object)copy).isEqualTo(undirectedGraph);
    }

    @Test
    public void copyOf_filtered_undirectedGraph() {
        MutableGraph<Integer> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addEdge((Object)N3, (Object)N1);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableGraph filteredCopy = Graphs.copyOf(undirectedGraph, (Predicate)nodePredicate);
        MutableGraph expectedGraph = GraphBuilder.undirected().build();
        expectedGraph.addEdge((Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void copyOf_filtered_directedGraph() {
        MutableGraph<Integer> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addEdge((Object)N3, (Object)N1);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableGraph filteredCopy = Graphs.copyOf(directedGraph, (Predicate)nodePredicate);
        MutableGraph expectedGraph = GraphBuilder.directed().build();
        expectedGraph.addEdge((Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_directedGraph() {
        MutableGraph<Integer> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addNode((Object)N3);
        MutableGraph actualGraph = GraphBuilder.directed().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (MutableGraph)actualGraph);
        MutableGraph expectedGraph = GraphBuilder.directed().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_directedGraph() {
        MutableGraph<Integer> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addEdge((Object)N3, (Object)N1);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableGraph actualGraph = GraphBuilder.directed().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (MutableGraph)actualGraph, (Predicate)nodePredicate);
        MutableGraph expectedGraph = GraphBuilder.directed().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_undirectedGraph() {
        MutableGraph<Integer> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addNode((Object)N3);
        MutableGraph actualGraph = GraphBuilder.undirected().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (MutableGraph)actualGraph);
        MutableGraph expectedGraph = GraphBuilder.undirected().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_undirectedGraph() {
        MutableGraph<Integer> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addEdge((Object)N3, (Object)N1);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableGraph actualGraph = GraphBuilder.undirected().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (MutableGraph)actualGraph, (Predicate)nodePredicate);
        MutableGraph expectedGraph = GraphBuilder.undirected().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void copyOf_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        MutableNetwork copy = Graphs.copyOf(directedGraph);
        Truth.assertThat((Object)copy).isEqualTo(directedGraph);
    }

    @Test
    public void copyOf_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        MutableNetwork copy = Graphs.copyOf(undirectedGraph);
        Truth.assertThat((Object)copy).isEqualTo(undirectedGraph);
    }

    @Test
    public void copyOf_filtered_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        undirectedGraph.addEdge((Object)E13, (Object)N1, (Object)N3);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(undirectedGraph));
        MutableNetwork filteredCopy = Graphs.copyOf(undirectedGraph, (Predicate)nodePredicate, (Predicate)edgePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void copyOf_filtered_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        directedGraph.addEdge((Object)E13, (Object)N1, (Object)N3);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(directedGraph));
        MutableNetwork filteredCopy = Graphs.copyOf(directedGraph, (Predicate)nodePredicate, (Predicate)edgePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        directedGraph.addNode((Object)N3);
        MutableNetwork actualGraph = NetworkBuilder.directed().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (MutableNetwork)actualGraph);
        MutableNetwork expectedGraph = NetworkBuilder.directed().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        directedGraph.addEdge((Object)E13, (Object)N1, (Object)N3);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableNetwork actualGraph = NetworkBuilder.directed().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (MutableNetwork)actualGraph, (Predicate)nodePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.directed().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        MutableNetwork actualGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Graphs.mergeEdgesFrom(directedGraph, (MutableNetwork)actualGraph);
        MutableNetwork<Integer, String> expectedGraph = GraphsTest.buildDirectedTestNetwork();
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Truth.assertThat((Object)actualGraph).isEqualTo(expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_filtered_directedNetwork() {
        MutableNetwork<Integer, String> directedGraph = GraphsTest.buildDirectedTestNetwork();
        directedGraph.addNode((Object)N3);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(directedGraph));
        MutableNetwork actualGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        actualGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        actualGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        Graphs.mergeEdgesFrom(directedGraph, (MutableNetwork)actualGraph, (Predicate)edgePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        expectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        expectedGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        undirectedGraph.addNode((Object)N3);
        MutableNetwork actualGraph = NetworkBuilder.undirected().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (MutableNetwork)actualGraph);
        MutableNetwork expectedGraph = NetworkBuilder.undirected().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        undirectedGraph.addEdge((Object)E13, (Object)N1, (Object)N3);
        Predicate nodePredicate = Predicates.in((Collection)ImmutableSet.of((Object)N1, (Object)N2));
        MutableNetwork actualGraph = NetworkBuilder.undirected().build();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (MutableNetwork)actualGraph, (Predicate)nodePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.undirected().build();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        MutableNetwork actualGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Graphs.mergeEdgesFrom(undirectedGraph, (MutableNetwork)actualGraph);
        MutableNetwork<Integer, String> expectedGraph = GraphsTest.buildUndirectedTestNetwork();
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Truth.assertThat((Object)actualGraph).isEqualTo(expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_filtered_undirectedNetwork() {
        MutableNetwork<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestNetwork();
        undirectedGraph.addNode((Object)N3);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(undirectedGraph));
        MutableNetwork actualGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        actualGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        actualGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        Graphs.mergeEdgesFrom(undirectedGraph, (MutableNetwork)actualGraph, (Predicate)edgePredicate);
        MutableNetwork expectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        expectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        expectedGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    private static MutableGraph<Integer> buildDirectedTestGraph() {
        MutableGraph directedGraph = GraphBuilder.directed().build();
        directedGraph.addEdge((Object)N1, (Object)N1);
        directedGraph.addEdge((Object)N1, (Object)N2);
        directedGraph.addEdge((Object)N2, (Object)N1);
        return directedGraph;
    }

    private static MutableGraph<Integer> buildUndirectedTestGraph() {
        MutableGraph undirectedGraph = GraphBuilder.undirected().build();
        undirectedGraph.addEdge((Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)N2, (Object)N1);
        return undirectedGraph;
    }

    private static MutableNetwork<Integer, String> buildDirectedTestNetwork() {
        MutableNetwork directedGraph = NetworkBuilder.directed().allowsParallelEdges(true).build();
        directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        return directedGraph;
    }

    private static MutableNetwork<Integer, String> buildUndirectedTestNetwork() {
        MutableNetwork undirectedGraph = NetworkBuilder.undirected().allowsParallelEdges(true).build();
        undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        return undirectedGraph;
    }
}

