/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.cache;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.LocalCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.FakeTicker;
import com.google.common.truth.Truth;
import java.lang.ref.Reference;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReferenceArray;
import javax.annotation.Nullable;
import junit.framework.Assert;

class CacheTesting {
    CacheTesting() {
    }

    static <K, V> void simulateValueReclamation(Cache<K, V> cache, K key) {
        LocalCache.ReferenceEntry<K, V> entry = CacheTesting.getReferenceEntry(cache, key);
        if (entry != null) {
            LocalCache.ValueReference valueRef = entry.getValueReference();
            Preconditions.checkState((boolean)(valueRef instanceof Reference));
            Reference ref = (Reference)valueRef;
            if (ref != null) {
                ref.clear();
            }
        }
    }

    static <K, V> void simulateKeyReclamation(Cache<K, V> cache, K key) {
        LocalCache.ReferenceEntry<K, V> entry = CacheTesting.getReferenceEntry(cache, key);
        Preconditions.checkState((boolean)(entry instanceof Reference));
        Reference ref = (Reference)entry;
        if (ref != null) {
            ref.clear();
        }
    }

    static <K, V> LocalCache.ReferenceEntry<K, V> getReferenceEntry(Cache<K, V> cache, K key) {
        Preconditions.checkNotNull(cache);
        Preconditions.checkNotNull(key);
        LocalCache<K, V> map = CacheTesting.toLocalCache(cache);
        return map.getEntry(key);
    }

    static <K, V> void forceExpandSegment(Cache<K, V> cache, K key) {
        Preconditions.checkNotNull(cache);
        Preconditions.checkNotNull(key);
        LocalCache<K, V> map = CacheTesting.toLocalCache(cache);
        int hash = map.hash(key);
        LocalCache.Segment segment = map.segmentFor(hash);
        segment.expand();
    }

    static <K, V> LocalCache<K, V> toLocalCache(Cache<K, V> cache) {
        if (cache instanceof LocalCache.LocalLoadingCache) {
            return ((LocalCache.LocalLoadingCache)cache).localCache;
        }
        throw new IllegalArgumentException("Cache of type " + cache.getClass() + " doesn't have a LocalCache.");
    }

    static boolean hasLocalCache(Cache<?, ?> cache) {
        return Preconditions.checkNotNull(cache) instanceof LocalCache.LocalLoadingCache;
    }

    static void drainRecencyQueues(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            LocalCache<?, ?> map = CacheTesting.toLocalCache(cache);
            for (LocalCache.Segment segment : map.segments) {
                CacheTesting.drainRecencyQueue(segment);
            }
        }
    }

    static void drainRecencyQueue(LocalCache.Segment<?, ?> segment) {
        segment.lock();
        try {
            segment.cleanUp();
        }
        finally {
            segment.unlock();
        }
    }

    static void drainReferenceQueues(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            CacheTesting.drainReferenceQueues(CacheTesting.toLocalCache(cache));
        }
    }

    static void drainReferenceQueues(LocalCache<?, ?> cchm) {
        for (LocalCache.Segment segment : cchm.segments) {
            CacheTesting.drainReferenceQueue(segment);
        }
    }

    static void drainReferenceQueue(LocalCache.Segment<?, ?> segment) {
        segment.lock();
        try {
            segment.drainReferenceQueues();
        }
        finally {
            segment.unlock();
        }
    }

    static int getTotalSegmentSize(Cache<?, ?> cache) {
        LocalCache<?, ?> map = CacheTesting.toLocalCache(cache);
        int totalSize = 0;
        for (LocalCache.Segment segment : map.segments) {
            totalSize = (int)((long)totalSize + segment.maxSegmentWeight);
        }
        return totalSize;
    }

    static void checkValidState(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            CacheTesting.checkValidState(CacheTesting.toLocalCache(cache));
        }
    }

    static void checkValidState(LocalCache<?, ?> cchm) {
        for (LocalCache.Segment segment : cchm.segments) {
            segment.cleanUp();
            Assert.assertFalse((boolean)segment.isLocked());
            Map table = CacheTesting.segmentTable(segment);
            segment.cleanUp();
            Truth.assertThat((Integer)table.size()).isAtMost((Comparable)Integer.valueOf(segment.count));
            for (Map.Entry entry : table.entrySet()) {
                Assert.assertNotNull(entry.getKey());
                Assert.assertNotNull(entry.getValue());
                Assert.assertSame(entry.getValue(), (Object)cchm.get(entry.getKey()));
            }
        }
        CacheTesting.checkEviction(cchm);
        CacheTesting.checkExpiration(cchm);
    }

    static void checkExpiration(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            CacheTesting.checkExpiration(CacheTesting.toLocalCache(cache));
        }
    }

    static void checkExpiration(LocalCache<?, ?> cchm) {
        for (LocalCache.Segment segment : cchm.segments) {
            Object key;
            LocalCache.ReferenceEntry prev;
            Set entries;
            if (cchm.usesWriteQueue()) {
                entries = Sets.newIdentityHashSet();
                prev = null;
                for (LocalCache.ReferenceEntry current : segment.writeQueue) {
                    Assert.assertTrue((boolean)entries.add(current));
                    if (prev != null) {
                        Assert.assertSame(prev, (Object)current.getPreviousInWriteQueue());
                        Assert.assertSame((Object)prev.getNextInWriteQueue(), (Object)current);
                        Truth.assertThat((Long)prev.getWriteTime()).isAtMost((Comparable)Long.valueOf(current.getWriteTime()));
                    }
                    if ((key = current.getKey()) != null) {
                        Assert.assertSame((Object)current, (Object)segment.getEntry(key, current.getHash()));
                    }
                    prev = current;
                }
                Assert.assertEquals((int)segment.count, (int)entries.size());
            } else {
                Assert.assertTrue((boolean)segment.writeQueue.isEmpty());
            }
            if (cchm.usesAccessQueue()) {
                entries = Sets.newIdentityHashSet();
                prev = null;
                for (LocalCache.ReferenceEntry current : segment.accessQueue) {
                    Assert.assertTrue((boolean)entries.add(current));
                    if (prev != null) {
                        Assert.assertSame((Object)prev, (Object)current.getPreviousInAccessQueue());
                        Assert.assertSame((Object)prev.getNextInAccessQueue(), (Object)current);
                        Assert.assertTrue((prev.getAccessTime() <= current.getAccessTime() || prev.getAccessTime() - current.getAccessTime() < 1000L ? 1 : 0) != 0);
                    }
                    if ((key = current.getKey()) != null) {
                        Assert.assertSame((Object)current, (Object)segment.getEntry(key, current.getHash()));
                    }
                    prev = current;
                }
                Assert.assertEquals((int)segment.count, (int)entries.size());
                continue;
            }
            Assert.assertTrue((boolean)segment.accessQueue.isEmpty());
        }
    }

    static void checkEviction(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            CacheTesting.checkEviction(CacheTesting.toLocalCache(cache));
        }
    }

    static void checkEviction(LocalCache<?, ?> map) {
        if (map.evictsBySize()) {
            for (LocalCache.Segment segment : map.segments) {
                CacheTesting.drainRecencyQueue(segment);
                Assert.assertEquals((int)0, (int)segment.recencyQueue.size());
                Assert.assertEquals((int)0, (int)segment.readCount.get());
                LocalCache.ReferenceEntry prev = null;
                for (LocalCache.ReferenceEntry current : segment.accessQueue) {
                    Object key;
                    if (prev != null) {
                        Assert.assertSame(prev, (Object)current.getPreviousInAccessQueue());
                        Assert.assertSame((Object)prev.getNextInAccessQueue(), (Object)current);
                    }
                    if ((key = current.getKey()) != null) {
                        Assert.assertSame((Object)current, (Object)segment.getEntry(key, current.getHash()));
                    }
                    prev = current;
                }
            }
        } else {
            for (LocalCache.Segment segment : map.segments) {
                Assert.assertEquals((int)0, (int)segment.recencyQueue.size());
            }
        }
    }

    static int segmentSize(LocalCache.Segment<?, ?> segment) {
        Map<?, ?> map = CacheTesting.segmentTable(segment);
        return map.size();
    }

    static <K, V> Map<K, V> segmentTable(LocalCache.Segment<K, V> segment) {
        AtomicReferenceArray table = segment.table;
        LinkedHashMap map = Maps.newLinkedHashMap();
        for (int i = 0; i < table.length(); ++i) {
            for (LocalCache.ReferenceEntry entry = (LocalCache.ReferenceEntry)table.get(i); entry != null; entry = entry.getNext()) {
                Object key = entry.getKey();
                Object value = entry.getValueReference().get();
                if (key == null || value == null) continue;
                Assert.assertNull((Object)map.put(key, value));
            }
        }
        return map;
    }

    static int writeQueueSize(Cache<?, ?> cache) {
        LocalCache<?, ?> cchm = CacheTesting.toLocalCache(cache);
        int size = 0;
        for (LocalCache.Segment segment : cchm.segments) {
            size += CacheTesting.writeQueueSize(segment);
        }
        return size;
    }

    static int writeQueueSize(LocalCache.Segment<?, ?> segment) {
        return segment.writeQueue.size();
    }

    static int accessQueueSize(Cache<?, ?> cache) {
        LocalCache<?, ?> cchm = CacheTesting.toLocalCache(cache);
        int size = 0;
        for (LocalCache.Segment segment : cchm.segments) {
            size += CacheTesting.accessQueueSize(segment);
        }
        return size;
    }

    static int accessQueueSize(LocalCache.Segment<?, ?> segment) {
        return segment.accessQueue.size();
    }

    static int expirationQueueSize(Cache<?, ?> cache) {
        return Math.max(CacheTesting.accessQueueSize(cache), CacheTesting.writeQueueSize(cache));
    }

    static void processPendingNotifications(Cache<?, ?> cache) {
        if (CacheTesting.hasLocalCache(cache)) {
            LocalCache<?, ?> cchm = CacheTesting.toLocalCache(cache);
            cchm.processPendingNotifications();
        }
    }

    static void checkRecency(LoadingCache<Integer, Integer> cache, int maxSize, Receiver<LocalCache.ReferenceEntry<Integer, Integer>> operation) {
        Preconditions.checkNotNull(operation);
        if (CacheTesting.hasLocalCache(cache)) {
            LocalCache.ReferenceEntry originalHead;
            CacheTesting.warmUp(cache, 0, 2 * maxSize);
            LocalCache<Integer, Integer> cchm = CacheTesting.toLocalCache(cache);
            LocalCache.Segment segment = cchm.segments[0];
            CacheTesting.drainRecencyQueue(segment);
            Assert.assertEquals((int)maxSize, (int)CacheTesting.accessQueueSize(cache));
            Assert.assertEquals((long)maxSize, (long)cache.size());
            LocalCache.ReferenceEntry entry = originalHead = (LocalCache.ReferenceEntry)segment.accessQueue.peek();
            operation.accept((LocalCache.ReferenceEntry<Integer, Integer>)entry);
            CacheTesting.drainRecencyQueue(segment);
            Assert.assertNotSame((Object)originalHead, segment.accessQueue.peek());
            Assert.assertEquals((long)cache.size(), (long)CacheTesting.accessQueueSize(cache));
        }
    }

    static void warmUp(LoadingCache<Integer, Integer> map, int start, int end) {
        Preconditions.checkNotNull(map);
        for (int i = start; i < end; ++i) {
            map.getUnchecked((Object)i);
        }
    }

    static void expireEntries(Cache<?, ?> cache, long expiringTime, FakeTicker ticker) {
        Preconditions.checkNotNull((Object)ticker);
        CacheTesting.expireEntries(CacheTesting.toLocalCache(cache), expiringTime, ticker);
    }

    static void expireEntries(LocalCache<?, ?> cchm, long expiringTime, FakeTicker ticker) {
        for (LocalCache.Segment segment : cchm.segments) {
            CacheTesting.drainRecencyQueue(segment);
        }
        ticker.advance(2L * expiringTime, TimeUnit.MILLISECONDS);
        long now = ticker.read();
        for (LocalCache.Segment segment : cchm.segments) {
            CacheTesting.expireEntries(segment, now);
            Assert.assertEquals((String)"Expiration queue must be empty by now", (int)0, (int)CacheTesting.writeQueueSize(segment));
            Assert.assertEquals((String)"Expiration queue must be empty by now", (int)0, (int)CacheTesting.accessQueueSize(segment));
            Assert.assertEquals((String)"Segments must be empty by now", (int)0, (int)CacheTesting.segmentSize(segment));
        }
        cchm.processPendingNotifications();
    }

    static void expireEntries(LocalCache.Segment<?, ?> segment, long now) {
        segment.lock();
        try {
            segment.expireEntries(now);
            segment.cleanUp();
        }
        finally {
            segment.unlock();
        }
    }

    static void checkEmpty(Cache<?, ?> cache) {
        Assert.assertEquals((long)0L, (long)cache.size());
        Assert.assertFalse((boolean)cache.asMap().containsKey(null));
        Assert.assertFalse((boolean)cache.asMap().containsKey(6));
        Assert.assertFalse((boolean)cache.asMap().containsValue(null));
        Assert.assertFalse((boolean)cache.asMap().containsValue(6));
        CacheTesting.checkEmpty(cache.asMap());
    }

    static void checkEmpty(ConcurrentMap<?, ?> map) {
        CacheTesting.checkEmpty(map.keySet());
        CacheTesting.checkEmpty(map.values());
        CacheTesting.checkEmpty(map.entrySet());
        Assert.assertEquals((Object)ImmutableMap.of(), map);
        Assert.assertEquals((int)ImmutableMap.of().hashCode(), (int)map.hashCode());
        Assert.assertEquals((String)ImmutableMap.of().toString(), (String)map.toString());
        if (map instanceof LocalCache) {
            LocalCache cchm = (LocalCache)map;
            CacheTesting.checkValidState(cchm);
            Assert.assertTrue((boolean)cchm.isEmpty());
            Assert.assertEquals((int)0, (int)cchm.size());
            for (LocalCache.Segment segment : cchm.segments) {
                Assert.assertEquals((int)0, (int)segment.count);
                Assert.assertEquals((int)0, (int)CacheTesting.segmentSize(segment));
                Assert.assertTrue((boolean)segment.writeQueue.isEmpty());
                Assert.assertTrue((boolean)segment.accessQueue.isEmpty());
            }
        }
    }

    static void checkEmpty(Collection<?> collection) {
        Assert.assertTrue((boolean)collection.isEmpty());
        Assert.assertEquals((int)0, (int)collection.size());
        Assert.assertFalse((boolean)collection.iterator().hasNext());
        Truth.assertThat((Object[])collection.toArray()).isEmpty();
        Truth.assertThat((Object[])collection.toArray(new Object[0])).isEmpty();
        if (collection instanceof Set) {
            new EqualsTester().addEqualityGroup(new Object[]{ImmutableSet.of(), collection}).addEqualityGroup(new Object[]{ImmutableSet.of((Object)"")}).testEquals();
        } else if (collection instanceof List) {
            new EqualsTester().addEqualityGroup(new Object[]{ImmutableList.of(), collection}).addEqualityGroup(new Object[]{ImmutableList.of((Object)"")}).testEquals();
        }
    }

    static interface Receiver<T> {
        public void accept(@Nullable T var1);
    }
}

