/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.graph;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.graph.DirectedGraph;
import com.google.common.graph.Graph;
import com.google.common.graph.GraphConfig;
import com.google.common.graph.Graphs;
import com.google.common.graph.UndirectedGraph;
import com.google.common.truth.Truth;
import java.util.Set;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class GraphsTest {
    private static final Integer N1 = 1;
    private static final Integer N2 = 2;
    private static final Integer N3 = 3;
    private static final Integer N4 = 4;
    private static final String E11 = "1-1";
    private static final String E11_A = "1-1a";
    private static final String E12 = "1-2";
    private static final String E12_A = "1-2a";
    private static final String E12_B = "1-2b";
    private static final String E21 = "2-1";
    private static final String E22 = "2-2";
    private static final String E13 = "1-3";
    private static final String E44 = "4-4";
    private static final int NODE_COUNT = 20;
    private static final int EDGE_COUNT = 20;
    private static final String ERROR_PARALLEL_EDGE = "connected by a different edge";
    private static final String ERROR_NEGATIVE_NODE_COUNT = "expected number of nodes can't be negative";
    private static final String ERROR_NEGATIVE_EDGE_COUNT = "expected number of edges can't be negative";
    private static final String ERROR_ADDED_PARALLEL_EDGE = "Should not be allowed to add a parallel edge.";
    private static final String ERROR_ADDED_SELF_LOOP = "Should not be allowed to add a self-loop edge.";
    static final String ERROR_SELF_LOOP = "self-loops are not allowed";

    @Test
    public void oppositeNode_basic() {
        ImmutableList testGraphs = ImmutableList.of((Object)Graphs.createDirected(), (Object)Graphs.createUndirected());
        for (Graph graph : testGraphs) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12, (Object)N2))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_parallelEdge() {
        ImmutableList testGraphs = ImmutableList.of((Object)Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH), (Object)Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH));
        for (Graph graph : testGraphs) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            graph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12, (Object)N2))).isEqualTo((Object)N1);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12_A, (Object)N1))).isEqualTo((Object)N2);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E12_A, (Object)N2))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_selfLoop() {
        ImmutableList testGraphs = ImmutableList.of((Object)Graphs.createDirected(), (Object)Graphs.createUndirected());
        for (Graph graph : testGraphs) {
            graph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Truth.assertThat((Integer)((Integer)Graphs.oppositeNode((Graph)graph, (Object)E11, (Object)N1))).isEqualTo((Object)N1);
        }
    }

    @Test
    public void oppositeNode_nodeNotIncident() {
        ImmutableList testGraphs = ImmutableList.of((Object)Graphs.createDirected(), (Object)Graphs.createUndirected());
        for (Graph graph : testGraphs) {
            graph.addEdge((Object)E12, (Object)N1, (Object)N2);
            graph.addEdge((Object)E13, (Object)N1, (Object)N3);
            try {
                Graphs.oppositeNode((Graph)graph, (Object)E12, (Object)N3);
                Assert.fail((String)"Should have rejected oppositeNode() called without a node incident to edge");
            }
            catch (IllegalArgumentException illegalArgumentException) {}
        }
    }

    @Test
    public void parallelEdges_directed() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)directedGraph, (Object)E12)).containsExactly(new Object[]{E12_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)directedGraph, (Object)E12_A)).containsExactly(new Object[]{E12});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)directedGraph, (Object)E21)).isEmpty();
    }

    @Test
    public void parallelEdges_selfLoop_directed() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)directedGraph, (Object)E11)).containsExactly(new Object[]{E11_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)directedGraph, (Object)E11_A)).containsExactly(new Object[]{E11});
    }

    @Test
    public void parallelEdges_undirected() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)undirectedGraph, (Object)E12)).containsExactly(new Object[]{E12_A, E21});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)undirectedGraph, (Object)E12_A)).containsExactly(new Object[]{E12, E21});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)undirectedGraph, (Object)E21)).containsExactly(new Object[]{E12, E12_A});
    }

    @Test
    public void parallelEdges_selfLoop_undirected() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)undirectedGraph, (Object)E11)).containsExactly(new Object[]{E11_A});
        Truth.assertThat((Iterable)Graphs.parallelEdges((Graph)undirectedGraph, (Object)E11_A)).containsExactly(new Object[]{E11});
    }

    @Test
    public void parallelEdges_unmodifiableView() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        Set parallelEdges = Graphs.parallelEdges((Graph)undirectedGraph, (Object)E12);
        Truth.assertThat((Iterable)parallelEdges).containsExactly(new Object[]{E12_A});
        undirectedGraph.addEdge((Object)E12_B, (Object)N1, (Object)N2);
        Truth.assertThat((Iterable)parallelEdges).containsExactly(new Object[]{E12_A, E12_B});
        try {
            parallelEdges.add(E21);
            Assert.fail((String)"Set returned by parallelEdges() should be unmodifiable");
        }
        catch (UnsupportedOperationException unsupportedOperationException) {
            // empty catch block
        }
    }

    @Test
    public void createDirected() {
        DirectedGraph directedGraph = Graphs.createDirected();
        Truth.assertThat((Iterable)directedGraph.nodes()).isEmpty();
        Truth.assertThat((Iterable)directedGraph.edges()).isEmpty();
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
        try {
            directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1)).isTrue();
    }

    @Test
    public void createUndirected() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected();
        Truth.assertThat((Iterable)undirectedGraph.nodes()).isEmpty();
        Truth.assertThat((Iterable)undirectedGraph.edges()).isEmpty();
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        try {
            undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        try {
            undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
            Assert.fail((String)ERROR_ADDED_PARALLEL_EDGE);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_PARALLEL_EDGE);
        }
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1)).isTrue();
    }

    @Test
    public void createDirected_multigraph() {
        DirectedGraph directedMultigraph = Graphs.createDirected((GraphConfig)Graphs.config().multigraph());
        Truth.assertThat((Boolean)directedMultigraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)directedMultigraph.addEdge((Object)E12_A, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedMultigraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12, (Object)E12_A));
        Truth.assertThat((Iterable)directedMultigraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_multigraph() {
        UndirectedGraph undirectedMultigraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E12_A, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)undirectedMultigraph.addEdge((Object)E21, (Object)N2, (Object)N1)).isTrue();
        Truth.assertThat((Iterable)undirectedMultigraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12, (Object)E12_A, (Object)E21));
    }

    @Test
    public void createDirected_expectedNodeCount() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.config().expectedNodeCount(20));
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_expectedNodeCount() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.config().expectedNodeCount(20));
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
    }

    @Test
    public void config_expectedNodeCount_negative() {
        try {
            Graphs.config().expectedNodeCount(-1);
            Assert.fail((String)ERROR_NEGATIVE_NODE_COUNT);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_NEGATIVE_NODE_COUNT);
        }
    }

    @Test
    public void config_expectedNodeCount_overridden() {
        GraphConfig oldConfig = Graphs.config().expectedNodeCount(20);
        Truth.assertThat((Integer)((Integer)oldConfig.getExpectedNodeCount().get())).isEqualTo((Object)20);
        GraphConfig newConfig = oldConfig.expectedNodeCount(21);
        Truth.assertThat((Integer)((Integer)oldConfig.getExpectedNodeCount().get())).isEqualTo((Object)20);
        Truth.assertThat((Integer)((Integer)newConfig.getExpectedNodeCount().get())).isEqualTo((Object)21);
    }

    @Test
    public void createDirected_expectedEdgeCount() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.config().expectedEdgeCount(20));
        Truth.assertThat((Boolean)directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)directedGraph.edgesConnecting((Object)N2, (Object)N1)).isEmpty();
    }

    @Test
    public void createUndirected_expectedEdgeCount() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.config().expectedEdgeCount(20));
        Truth.assertThat((Boolean)undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N1, (Object)N2)).isEqualTo((Object)ImmutableSet.of((Object)E12));
        Truth.assertThat((Iterable)undirectedGraph.edgesConnecting((Object)N2, (Object)N1)).isEqualTo((Object)ImmutableSet.of((Object)E12));
    }

    @Test
    public void config_expectedEdgeCount_negative() {
        try {
            Graphs.config().expectedEdgeCount(-1);
            Assert.fail((String)ERROR_NEGATIVE_EDGE_COUNT);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_NEGATIVE_EDGE_COUNT);
        }
    }

    @Test
    public void config_expectedEdgeCount_overridden() {
        GraphConfig oldConfig = Graphs.config().expectedEdgeCount(20);
        Truth.assertThat((Integer)((Integer)oldConfig.getExpectedEdgeCount().get())).isEqualTo((Object)20);
        GraphConfig newConfig = oldConfig.expectedEdgeCount(21);
        Truth.assertThat((Integer)((Integer)oldConfig.getExpectedEdgeCount().get())).isEqualTo((Object)20);
        Truth.assertThat((Integer)((Integer)newConfig.getExpectedEdgeCount().get())).isEqualTo((Object)21);
    }

    @Test
    public void createDirected_noSelfLoops() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.config().noSelfLoops());
        try {
            directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Assert.fail((String)ERROR_ADDED_SELF_LOOP);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_SELF_LOOP);
        }
    }

    @Test
    public void createUndirected_noSelfLoops() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.config().noSelfLoops());
        try {
            undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
            Assert.fail((String)ERROR_ADDED_SELF_LOOP);
        }
        catch (IllegalArgumentException e) {
            Truth.assertThat((String)e.getMessage()).contains((CharSequence)ERROR_SELF_LOOP);
        }
    }

    @Test
    public void config_immutability() {
        GraphConfig unused = Graphs.config().multigraph();
        Truth.assertThat((Boolean)Graphs.config().isMultigraph()).isFalse();
        unused = Graphs.config().expectedNodeCount(20);
        Truth.assertThat((Optional)Graphs.config().getExpectedNodeCount()).isAbsent();
        unused = Graphs.config().expectedEdgeCount(20);
        Truth.assertThat((Optional)Graphs.config().getExpectedEdgeCount()).isAbsent();
        unused = Graphs.config().noSelfLoops();
        Truth.assertThat((Boolean)Graphs.config().isSelfLoopsAllowed()).isTrue();
    }

    @Test
    public void addEdge_nullGraph() {
        try {
            Graphs.addEdge(null, (Object)E11, (Iterable)ImmutableSet.of((Object)N1));
            Assert.fail((String)"Should have rejected null graph");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_nullNodes() {
        try {
            Graphs.addEdge((Graph)Graphs.createDirected(), (Object)E11, null);
            Assert.fail((String)"Should have rejected null nodes");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_tooManyNodes() {
        try {
            Graphs.addEdge((Graph)Graphs.createDirected(), (Object)E11, (Iterable)ImmutableSet.of((Object)N1, (Object)N2, (Object)N3));
            Assert.fail((String)"Should have rejected adding an edge to a Graph with > 2 nodes");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_notEnoughNodes() {
        try {
            Graphs.addEdge((Graph)Graphs.createDirected(), (Object)E11, (Iterable)ImmutableSet.of());
            Assert.fail((String)"Should have rejected adding an edge to a Graph with < 1 nodes");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    public void addEdge_selfLoop() {
        DirectedGraph directedGraph = Graphs.createDirected();
        Truth.assertThat((Boolean)Graphs.addEdge((Graph)directedGraph, (Object)E11, (Iterable)ImmutableSet.of((Object)N1))).isTrue();
        Truth.assertThat((Iterable)directedGraph.edges()).containsExactly(new Object[]{E11});
        Truth.assertThat((Iterable)directedGraph.nodes()).containsExactly(new Object[]{N1});
        Truth.assertThat((Iterable)directedGraph.incidentNodes((Object)E11)).containsExactly(new Object[]{N1});
    }

    @Test
    public void addEdge_basic() {
        DirectedGraph directedGraph = Graphs.createDirected();
        Truth.assertThat((Boolean)Graphs.addEdge((Graph)directedGraph, (Object)E12, (Iterable)ImmutableSet.of((Object)N1, (Object)N2))).isTrue();
        Truth.assertThat((Iterable)directedGraph.edges()).containsExactly(new Object[]{E12});
        Truth.assertThat((Iterable)directedGraph.nodes()).containsExactly(new Object[]{N1, N2}).inOrder();
        Truth.assertThat((Iterable)directedGraph.incidentNodes((Object)E12)).containsExactly(new Object[]{N1, N2}).inOrder();
    }

    @Test
    public void copyOf_nullArgument() {
        try {
            Graphs.copyOf((DirectedGraph)null);
            Assert.fail((String)"Should have rejected a null graph");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
        try {
            Graphs.copyOf((UndirectedGraph)null);
            Assert.fail((String)"Should have rejected a null graph");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void copyOf_directedGraph() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        DirectedGraph copy = Graphs.copyOf(directedGraph);
        Truth.assertThat((Object)copy).isEqualTo(directedGraph);
    }

    @Test
    public void copyOf_undirectedGraph() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        UndirectedGraph copy = Graphs.copyOf(undirectedGraph);
        Truth.assertThat((Object)copy).isEqualTo(undirectedGraph);
    }

    @Test
    public void copyOf_filtered_undirected() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addNode((Object)N3);
        Predicate<Integer> nodePredicate = GraphsTest.connectedNodePredicate(undirectedGraph);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(undirectedGraph));
        UndirectedGraph filteredCopy = Graphs.copyOf(undirectedGraph, nodePredicate, (Predicate)edgePredicate);
        UndirectedGraph expectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void copyOf_filtered_directed() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addNode((Object)N3);
        Predicate<Integer> nodePredicate = GraphsTest.connectedNodePredicate(directedGraph);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(directedGraph));
        DirectedGraph filteredCopy = Graphs.copyOf(directedGraph, nodePredicate, (Predicate)edgePredicate);
        DirectedGraph expectedGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)filteredCopy).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_directed() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addNode((Object)N3);
        DirectedGraph actualGraph = Graphs.createDirected();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (Graph)actualGraph);
        DirectedGraph expectedGraph = Graphs.createDirected();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_directed() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addNode((Object)N3);
        Predicate<Integer> nodePredicate = GraphsTest.connectedNodePredicate(directedGraph);
        DirectedGraph actualGraph = Graphs.createDirected();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(directedGraph, (Graph)actualGraph, nodePredicate);
        DirectedGraph expectedGraph = Graphs.createDirected();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_directed() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        DirectedGraph actualGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Graphs.mergeEdgesFrom(directedGraph, (DirectedGraph)actualGraph);
        DirectedGraph<Integer, String> expectedGraph = GraphsTest.buildDirectedTestGraph();
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Truth.assertThat((Object)actualGraph).isEqualTo(expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_filtered_directed() {
        DirectedGraph<Integer, String> directedGraph = GraphsTest.buildDirectedTestGraph();
        directedGraph.addNode((Object)N3);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(directedGraph));
        DirectedGraph actualGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        actualGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        actualGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        Graphs.mergeEdgesFrom(directedGraph, (DirectedGraph)actualGraph, (Predicate)edgePredicate);
        DirectedGraph expectedGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        expectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        expectedGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_undirected() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addNode((Object)N3);
        UndirectedGraph actualGraph = Graphs.createUndirected();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (Graph)actualGraph);
        UndirectedGraph expectedGraph = Graphs.createUndirected();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N3);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeNodesFrom_filtered_undirected() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addNode((Object)N3);
        Predicate<Integer> nodePredicate = GraphsTest.connectedNodePredicate(undirectedGraph);
        UndirectedGraph actualGraph = Graphs.createUndirected();
        actualGraph.addNode((Object)N4);
        actualGraph.addNode((Object)N2);
        Graphs.mergeNodesFrom(undirectedGraph, (Graph)actualGraph, nodePredicate);
        UndirectedGraph expectedGraph = Graphs.createUndirected();
        expectedGraph.addNode((Object)N1);
        expectedGraph.addNode((Object)N2);
        expectedGraph.addNode((Object)N4);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_undirected() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        UndirectedGraph actualGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Graphs.mergeEdgesFrom(undirectedGraph, (Graph)actualGraph);
        UndirectedGraph<Integer, String> expectedGraph = GraphsTest.buildUndirectedTestGraph();
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        Truth.assertThat((Object)actualGraph).isEqualTo(expectedGraph);
    }

    @Test
    public void mergeEdgesFrom_filtered_undirected() {
        UndirectedGraph<Integer, String> undirectedGraph = GraphsTest.buildUndirectedTestGraph();
        undirectedGraph.addNode((Object)N3);
        Predicate edgePredicate = Predicates.not((Predicate)Graphs.selfLoopPredicate(undirectedGraph));
        UndirectedGraph actualGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        actualGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        actualGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        actualGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        actualGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        Graphs.mergeEdgesFrom(undirectedGraph, (Graph)actualGraph, (Predicate)edgePredicate);
        UndirectedGraph expectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        expectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        expectedGraph.addEdge((Object)E22, (Object)N2, (Object)N2);
        expectedGraph.addEdge((Object)E44, (Object)N4, (Object)N4);
        expectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        expectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        Truth.assertThat((Object)actualGraph).isEqualTo((Object)expectedGraph);
    }

    private static DirectedGraph<Integer, String> buildDirectedTestGraph() {
        DirectedGraph directedGraph = Graphs.createDirected((GraphConfig)Graphs.MULTIGRAPH);
        directedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        directedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        directedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        return directedGraph;
    }

    private static UndirectedGraph<Integer, String> buildUndirectedTestGraph() {
        UndirectedGraph undirectedGraph = Graphs.createUndirected((GraphConfig)Graphs.MULTIGRAPH);
        undirectedGraph.addEdge((Object)E11, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E12, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E11_A, (Object)N1, (Object)N1);
        undirectedGraph.addEdge((Object)E12_A, (Object)N1, (Object)N2);
        undirectedGraph.addEdge((Object)E21, (Object)N2, (Object)N1);
        return undirectedGraph;
    }

    private static <N> Predicate<N> connectedNodePredicate(final Graph<N, ?> graph) {
        Preconditions.checkNotNull(graph, (Object)"graph");
        return new Predicate<N>(){

            public boolean apply(N node) {
                return graph.degree(node) > 0;
            }
        };
    }
}

