/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableList;
import com.google.common.hash.HashCode;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import com.google.common.testing.ClassSanityTester;
import java.util.Arrays;
import junit.framework.TestCase;

public class HashCodeTest
extends TestCase {
    private static final ImmutableList<ExpectedHashCode> expectedHashCodes = ImmutableList.of((Object)new ExpectedHashCode(new byte[]{-17, -51, -85, -119, 103, 69, 35, 1}, -1985229329, 81985529216486895L, "efcdab8967452301"), (Object)new ExpectedHashCode(new byte[]{-17, -51, -85, -119, 103, 69, 35, 1, 1, 2, 3, 4, 5, 6, 7, 8}, -1985229329, 81985529216486895L, "efcdab89674523010102030405060708"), (Object)new ExpectedHashCode(new byte[]{-33, -101, 87, 19}, 324508639, null, "df9b5713"), (Object)new ExpectedHashCode(new byte[]{-51, -85, 0, 0}, 43981, null, "cdab0000"), (Object)new ExpectedHashCode(new byte[]{-17, -51, -85, 0, 0, 0, 0, 0}, 11259375, 11259375L, "efcdab0000000000"));
    private static final HashCode HASH_ABCD = HashCode.fromBytes((byte[])new byte[]{-86, -69, -52, -35});

    public void testFromInt() {
        for (ExpectedHashCode expected : expectedHashCodes) {
            if (expected.bytes.length != 4) continue;
            HashCode fromInt = HashCode.fromInt((int)expected.asInt);
            HashCodeTest.assertExpectedHashCode(expected, fromInt);
        }
    }

    public void testFromLong() {
        for (ExpectedHashCode expected : expectedHashCodes) {
            if (expected.bytes.length != 8) continue;
            HashCode fromLong = HashCode.fromLong((long)expected.asLong);
            HashCodeTest.assertExpectedHashCode(expected, fromLong);
        }
    }

    public void testFromBytes() {
        for (ExpectedHashCode expected : expectedHashCodes) {
            HashCode fromBytes = HashCode.fromBytes((byte[])expected.bytes);
            HashCodeTest.assertExpectedHashCode(expected, fromBytes);
        }
    }

    public void testFromBytes_copyOccurs() {
        byte[] bytes = new byte[]{-51, -85, 0, 0};
        HashCode hashCode = HashCode.fromBytes((byte[])bytes);
        int expectedInt = 43981;
        String expectedToString = "cdab0000";
        HashCodeTest.assertEquals((int)expectedInt, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)expectedToString, (String)hashCode.toString());
        bytes[0] = 0;
        HashCodeTest.assertEquals((int)expectedInt, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)expectedToString, (String)hashCode.toString());
    }

    public void testFromBytesNoCopy_noCopyOccurs() {
        byte[] bytes = new byte[]{-51, -85, 0, 0};
        HashCode hashCode = HashCode.fromBytesNoCopy((byte[])bytes);
        HashCodeTest.assertEquals((int)43981, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)"cdab0000", (String)hashCode.toString());
        bytes[0] = 0;
        HashCodeTest.assertEquals((int)43776, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)"00ab0000", (String)hashCode.toString());
    }

    public void testGetBytesInternal_noCloneOccurs() {
        byte[] bytes = new byte[]{-51, -85, 0, 0};
        HashCode hashCode = HashCode.fromBytes((byte[])bytes);
        HashCodeTest.assertEquals((int)43981, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)"cdab0000", (String)hashCode.toString());
        hashCode.getBytesInternal()[0] = 0;
        HashCodeTest.assertEquals((int)43776, (int)hashCode.asInt());
        HashCodeTest.assertEquals((String)"00ab0000", (String)hashCode.toString());
    }

    public void testPadToLong() {
        HashCodeTest.assertEquals((long)0x1111111111111111L, (long)HashCode.fromLong((long)0x1111111111111111L).padToLong());
        HashCodeTest.assertEquals((long)-7378697629483820647L, (long)HashCode.fromLong((long)-7378697629483820647L).padToLong());
        HashCodeTest.assertEquals((long)0x11111111L, (long)HashCode.fromInt((int)0x11111111).padToLong());
        HashCodeTest.assertEquals((long)0x99999999L, (long)HashCode.fromInt((int)-1717986919).padToLong());
    }

    public void testPadToLongWith4Bytes() {
        HashCodeTest.assertEquals((long)0x99999999L, (long)HashCode.fromBytesNoCopy((byte[])HashCodeTest.byteArrayWith9s(4)).padToLong());
    }

    public void testPadToLongWith6Bytes() {
        HashCodeTest.assertEquals((long)0x999999999999L, (long)HashCode.fromBytesNoCopy((byte[])HashCodeTest.byteArrayWith9s(6)).padToLong());
    }

    public void testPadToLongWith8Bytes() {
        HashCodeTest.assertEquals((long)-7378697629483820647L, (long)HashCode.fromBytesNoCopy((byte[])HashCodeTest.byteArrayWith9s(8)).padToLong());
    }

    private static byte[] byteArrayWith9s(int size) {
        byte[] bytez = new byte[size];
        Arrays.fill(bytez, (byte)-103);
        return bytez;
    }

    public void testToString() {
        byte[] data = new byte[]{127, -128, 5, -1, 14};
        HashCodeTest.assertEquals((String)"7f8005ff0e", (String)HashCode.fromBytes((byte[])data).toString());
        HashCodeTest.assertEquals((String)"7f8005ff0e", (String)BaseEncoding.base16().lowerCase().encode(data));
    }

    public void testHashCode_nulls() throws Exception {
        HashCodeTest.sanityTester().testNulls();
    }

    public void testHashCode_equalsAndSerializable() throws Exception {
        HashCodeTest.sanityTester().testEqualsAndSerializable();
    }

    public void testRoundTripHashCodeUsingBaseEncoding() {
        HashCode hash1 = Hashing.sha1().hashString((CharSequence)"foo", Charsets.US_ASCII);
        HashCode hash2 = HashCode.fromBytes((byte[])BaseEncoding.base16().lowerCase().decode((CharSequence)hash1.toString()));
        HashCodeTest.assertEquals((Object)hash1, (Object)hash2);
    }

    public void testObjectHashCode() {
        HashCode hashCode42 = HashCode.fromInt((int)42);
        HashCodeTest.assertEquals((int)42, (int)hashCode42.hashCode());
    }

    public void testObjectHashCodeWithSameLowOrderBytes() {
        byte[] bytesA = new byte[5];
        byte[] bytesB = new byte[5];
        bytesA[4] = -66;
        bytesB[4] = -17;
        HashCode hashCodeA = HashCode.fromBytes((byte[])bytesA);
        HashCode hashCodeB = HashCode.fromBytes((byte[])bytesB);
        HashCodeTest.assertFalse((boolean)hashCodeA.equals((Object)hashCodeB));
        HashCodeTest.assertEquals((int)hashCodeA.hashCode(), (int)hashCodeB.hashCode());
    }

    public void testRoundTripHashCodeUsingFromString() {
        HashCode hash1 = Hashing.sha1().hashString((CharSequence)"foo", Charsets.US_ASCII);
        HashCode hash2 = HashCode.fromString((String)hash1.toString());
        HashCodeTest.assertEquals((Object)hash1, (Object)hash2);
    }

    public void testRoundTrip() {
        for (ExpectedHashCode expected : expectedHashCodes) {
            String string = HashCode.fromBytes((byte[])expected.bytes).toString();
            HashCodeTest.assertEquals((String)expected.toString, (String)string);
            HashCodeTest.assertEquals((String)expected.toString, (String)HashCode.fromBytes((byte[])BaseEncoding.base16().lowerCase().decode((CharSequence)string)).toString());
        }
    }

    public void testFromStringFailsWithInvalidHexChar() {
        try {
            HashCode.fromString((String)"7f8005ff0z");
            HashCodeTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testFromStringFailsWithUpperCaseString() {
        String string = Hashing.sha1().hashString((CharSequence)"foo", Charsets.US_ASCII).toString().toUpperCase();
        try {
            HashCode.fromString((String)string);
            HashCodeTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testFromStringFailsWithShortInputs() {
        try {
            HashCode.fromString((String)"");
            HashCodeTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            HashCode.fromString((String)"7");
            HashCodeTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        HashCode unused = HashCode.fromString((String)"7f");
    }

    public void testFromStringFailsWithOddLengthInput() {
        try {
            HashCode.fromString((String)"7f8");
            HashCodeTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testIntWriteBytesTo() {
        byte[] dest = new byte[4];
        HashCode.fromInt((int)42).writeBytesTo(dest, 0, 4);
        HashCodeTest.assertTrue((boolean)Arrays.equals(HashCode.fromInt((int)42).asBytes(), dest));
    }

    public void testLongWriteBytesTo() {
        byte[] dest = new byte[8];
        HashCode.fromLong((long)42L).writeBytesTo(dest, 0, 8);
        HashCodeTest.assertTrue((boolean)Arrays.equals(HashCode.fromLong((long)42L).asBytes(), dest));
    }

    public void testWriteBytesTo() {
        byte[] dest = new byte[4];
        HASH_ABCD.writeBytesTo(dest, 0, 4);
        HashCodeTest.assertTrue((boolean)Arrays.equals(new byte[]{-86, -69, -52, -35}, dest));
    }

    public void testWriteBytesToOversizedArray() {
        byte[] dest = new byte[5];
        HASH_ABCD.writeBytesTo(dest, 0, 4);
        HashCodeTest.assertTrue((boolean)Arrays.equals(new byte[]{-86, -69, -52, -35, 0}, dest));
    }

    public void testWriteBytesToOversizedArrayLongMaxLength() {
        byte[] dest = new byte[5];
        HASH_ABCD.writeBytesTo(dest, 0, 5);
        HashCodeTest.assertTrue((boolean)Arrays.equals(new byte[]{-86, -69, -52, -35, 0}, dest));
    }

    public void testWriteBytesToOversizedArrayShortMaxLength() {
        byte[] dest = new byte[5];
        HASH_ABCD.writeBytesTo(dest, 0, 3);
        HashCodeTest.assertTrue((boolean)Arrays.equals(new byte[]{-86, -69, -52, 0, 0}, dest));
    }

    public void testWriteBytesToUndersizedArray() {
        byte[] dest = new byte[3];
        try {
            HASH_ABCD.writeBytesTo(dest, 0, 4);
            HashCodeTest.fail();
        }
        catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            // empty catch block
        }
    }

    public void testWriteBytesToUndersizedArrayLongMaxLength() {
        byte[] dest = new byte[3];
        try {
            HASH_ABCD.writeBytesTo(dest, 0, 5);
            HashCodeTest.fail();
        }
        catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            // empty catch block
        }
    }

    public void testWriteBytesToUndersizedArrayShortMaxLength() {
        byte[] dest = new byte[3];
        HASH_ABCD.writeBytesTo(dest, 0, 2);
        HashCodeTest.assertTrue((boolean)Arrays.equals(new byte[]{-86, -69, 0}, dest));
    }

    private static ClassSanityTester.FactoryMethodReturnValueTester sanityTester() {
        return new ClassSanityTester().setDefault(byte[].class, (Object)new byte[]{1, 2, 3, 4}).setDistinctValues(byte[].class, (Object)new byte[]{1, 2, 3, 4}, (Object)new byte[]{5, 6, 7, 8}).setDistinctValues(String.class, (Object)"7f8005ff0e", (Object)"7f8005ff0f").forAllPublicStaticMethods(HashCode.class);
    }

    private static void assertExpectedHashCode(ExpectedHashCode expectedHashCode, HashCode hash) {
        HashCodeTest.assertTrue((boolean)Arrays.equals(expectedHashCode.bytes, hash.asBytes()));
        byte[] bb = new byte[hash.bits() / 8];
        hash.writeBytesTo(bb, 0, bb.length);
        HashCodeTest.assertTrue((boolean)Arrays.equals(expectedHashCode.bytes, bb));
        HashCodeTest.assertEquals((int)expectedHashCode.asInt, (int)hash.asInt());
        if (expectedHashCode.asLong == null) {
            try {
                hash.asLong();
                HashCodeTest.fail();
            }
            catch (IllegalStateException illegalStateException) {}
        } else {
            HashCodeTest.assertEquals((long)expectedHashCode.asLong, (long)hash.asLong());
        }
        HashCodeTest.assertEquals((String)expectedHashCode.toString, (String)hash.toString());
        HashCodeTest.assertSideEffectFree(hash);
        HashCodeTest.assertReadableBytes(hash);
    }

    private static void assertSideEffectFree(HashCode hash) {
        byte[] original = hash.asBytes();
        byte[] mutated = hash.asBytes();
        mutated[0] = (byte)(mutated[0] + 1);
        HashCodeTest.assertTrue((boolean)Arrays.equals(original, hash.asBytes()));
    }

    private static void assertReadableBytes(HashCode hashCode) {
        HashCodeTest.assertTrue((hashCode.bits() >= 32 ? 1 : 0) != 0);
        byte[] hashBytes = hashCode.asBytes();
        int totalBytes = hashCode.bits() / 8;
        for (int bytes = 0; bytes < totalBytes; ++bytes) {
            byte[] bb = new byte[bytes];
            hashCode.writeBytesTo(bb, 0, bb.length);
            HashCodeTest.assertTrue((boolean)Arrays.equals(Arrays.copyOf(hashBytes, bytes), bb));
        }
    }

    private static class ExpectedHashCode {
        final byte[] bytes;
        final int asInt;
        final Long asLong;
        final String toString;

        ExpectedHashCode(byte[] bytes, int asInt, Long asLong, String toString) {
            this.bytes = bytes;
            this.asInt = asInt;
            this.asLong = asLong;
            this.toString = toString;
        }
    }
}

