/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2012 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.undertow.util;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.Executor;

import io.undertow.UndertowLogger;
import io.undertow.server.HttpServerExchange;
import org.xnio.channels.StreamSourceChannel;

/**
 * Class that deals with a worker thread pools
 *
 * @author Stuart Douglas
 */
public class WorkerDispatcher {

    private static final ThreadLocal<DispatchData> executingInWorker = new ThreadLocal<DispatchData>();

    public static final AttachmentKey<Executor> EXECUTOR_ATTACHMENT_KEY = AttachmentKey.create(Executor.class);

    /**
     * Dispatches the request. By default this will be dispatched to the Xnio Worker.This can be changed by
     * attaching an executor to the exchange using {@link #EXECUTOR_ATTACHMENT_KEY}.
     *
     * If the request is already running in the selected executor then no dispatch takes place and the
     * Runnable is simply run in the current thread
     *
     * @param exchange The exchange
     * @param runnable The task to run
     */
    public static void dispatch(final HttpServerExchange exchange, final Runnable runnable) {
        Executor executor = exchange.getAttachment(EXECUTOR_ATTACHMENT_KEY);
        if (executor == null) {
            executor = exchange.getConnection().getWorker();
        }
        final DispatchData dd = executingInWorker.get();
        if (dd != null && dd.executor == executor) {
            runnable.run();
        } else {
            executor.execute(new DispatchedRunnable(executor, runnable));
        }
    }
    /**
     * Dispatches the request. By default this will be dispatched to the Xnio Worker.This can be changed by
     * attaching an executor to the exchange using {@link #EXECUTOR_ATTACHMENT_KEY}.
     *
     * This method will always dispatch, even if the request is already running in the executor.
     *
     * @param exchange The exchange
     * @param runnable The task to run
     */
    public static void forceDispatch(final HttpServerExchange exchange, final Runnable runnable) {
        Executor executor = exchange.getAttachment(EXECUTOR_ATTACHMENT_KEY);
        if (executor == null) {
            executor = exchange.getConnection().getWorker();
        }
        final DispatchData dd = executingInWorker.get();
        executor.execute(new DispatchedRunnable(executor, runnable));
    }

    /**
     * Forces a task dispatch with the specified executor
     *
     * @param executor The executor to use
     * @param runnable The runnable
     */
    public static void dispatch(final Executor executor, final Runnable runnable) {
        executor.execute(new DispatchedRunnable(executor, runnable));
    }

    /**
     * Dispatches the next request in the current exectutor. If there is no current executor then the
     * channels read thread is used.
     *
     * @param channel  The channel that will be used for the next request
     * @param runnable The task to run
     */
    public static void dispatchNextRequest(final StreamSourceChannel channel, final Runnable runnable) {
        final DispatchData dd = executingInWorker.get();
        if (dd == null) {
            channel.getReadThread().execute(runnable);
        } else {
            dd.tasks.add(runnable);
        }
    }

    private WorkerDispatcher() {

    }

    private static final class DispatchData {
        final Executor executor;
        final Deque<Runnable> tasks = new ArrayDeque<>();

        private DispatchData(Executor executor) {
            this.executor = executor;
        }
    }

    private static class DispatchedRunnable implements Runnable {
        private final Executor executor;
        private final Runnable runnable;

        public DispatchedRunnable(Executor executor, Runnable runnable) {
            this.executor = executor;
            this.runnable = runnable;
        }

        @Override
        public void run() {
            final DispatchData data = new DispatchData(executor);
            try {
                executingInWorker.set(data);
                runnable.run();
            } catch (Exception e) {
                UndertowLogger.REQUEST_LOGGER.exceptionProcessingRequest(e);
            } finally {
                Runnable next = data.tasks.poll();
                try {
                    while (next != null) {
                        try {
                            next.run();
                        } catch (Exception e) {
                            UndertowLogger.REQUEST_LOGGER.exceptionProcessingRequest(e);
                        }
                        next = data.tasks.poll();
                    }
                } finally {
                    executingInWorker.remove();
                }
            }
        }
    }
}
