/*
 * Copyright 2002-2004 Greg Hinkle
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.mc4j.ems.impl.jmx.connection.support.providers;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mc4j.ems.connection.ConnectionException;
import org.mc4j.ems.impl.jmx.connection.support.providers.proxy.GenericMBeanServerProxy;

import javax.management.MBeanServer;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.NoInitialContextException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.Principal;
import java.security.PrivilegedAction;
import java.util.Properties;

/**
 * Represents a Connection to a JBoss JMX Server. This connection
 * works against the JBoss RMI connector.
 *
 * @author Greg Hinkle (ghinkle@users.sourceforge.net), January 2002
 * @version $Revision: 594 $($Author: ianpspringer $ / $Date: 2008-11-05 11:38:56 -0500 (Wed, 05 Nov 2008) $)
 */
public class JBossConnectionProvider extends AbstractConnectionProvider {

    private MBeanServer mbeanServer;
    private GenericMBeanServerProxy proxy;

//    private Management mejb;
    private static final String MEJB_JNDI = "ejb/mgmt/MEJB";


    private static Log log = LogFactory.getLog(JBossConnectionProvider.class);


    protected void doConnect() throws Exception {
        ClassLoader currentLoader = Thread.currentThread().getContextClassLoader();

        try {
            System.setProperty("jmx.serial.form", "1.1");

            // TODO: Used to need this, but it appears to work without now (verify)
            // Change the context classloader as the JBoss version of the
            // MBeanServerFactory uses it to find their class

            ClassLoader childLoader = this.getClass().getClassLoader();
            Thread.currentThread().setContextClassLoader(childLoader);

            InitialContext context = getInitialContext();

            Object rmiAdaptor = context.lookup(connectionSettings.getJndiName());

            // GH: Works around a real strange "LinkageError: Duplicate class found"
            // by loading these classes in the main connection classloader
            //Class foo = RMINotificationListener.class;
            //foo = RMINotificationListenerMBean.class;

            // TODO GH!: I think this fixes notifications, but breaks compatibility with at least 3.0.8
            //RMIConnectorImpl connector = new RMIConnectorImpl(rmiAdaptor);

            if (this.proxy != null) {
                // This is a reconnect
                proxy.setRemoteServer(rmiAdaptor);
            } else {
                this.proxy = new GenericMBeanServerProxy(rmiAdaptor);
                this.proxy.setProvider(this);
                setStatsProxy(proxy);
                this.mbeanServer = proxy.buildServerProxy();
            }
            //this.mgmt = retrieveMEJB();

            // Set the context classloader back to what it was
        } finally {
            Thread.currentThread().setContextClassLoader(currentLoader);

        }
    }

    public static final String JNDI_LOGIN_CONTEXT_FACTORY_CLASS = "org.jboss.security.jndi.JndiLoginInitialContextFactory";

    private InitialContext getInitialContext() throws NamingException {
        Properties props = connectionSettings.getAdvancedProperties();

        if (connectionSettings.getPrincipal() != null && connectionSettings.getPrincipal().length() != 0) {
            try {
                Class.forName(JNDI_LOGIN_CONTEXT_FACTORY_CLASS);
                log.debug("Utilizing JNDI Login Context Factory for secured access [" + JNDI_LOGIN_CONTEXT_FACTORY_CLASS + "]");

                props.put(Context.INITIAL_CONTEXT_FACTORY, JNDI_LOGIN_CONTEXT_FACTORY_CLASS);
                props.put(Context.SECURITY_PRINCIPAL, connectionSettings.getPrincipal());
                props.put(Context.SECURITY_CREDENTIALS, connectionSettings.getCredentials());

            } catch (ClassNotFoundException e) {
                log.debug("JNDI Login Context Factory not available, directly utilizing SecurityAssociation");
                try {
                    props.put(Context.INITIAL_CONTEXT_FACTORY, connectionSettings.getInitialContextName());
                    resetPrincipalInfo();
                } catch (ClassNotFoundException e1) {
                    throw new ConnectionException("Secured connection not available with this version of JBoss " + e1.toString(),e1);
                } catch (Exception e1) {
                    throw new ConnectionException("Unable to make secured connection to JBoss due to missing or unexpected security classes",e1);
                }
            }

        } else {
            props.put(Context.INITIAL_CONTEXT_FACTORY, connectionSettings.getInitialContextName());
        }

        props.put(Context.PROVIDER_URL, connectionSettings.getServerUrl());

        try {
            InitialContext context = new InitialContext(props);
            return context;
        } catch(NoInitialContextException e) {
            // Try to be more helpful, indicating the reason we couldn't make the connection in this
            // common case of missing libraries
            if (e.getCause() instanceof ClassNotFoundException) {
                throw new ConnectionException("Necessary classes not found for remote connection, check installation path configuration.",e.getCause());
            }
            throw e;
        }
    }


    private static class SetPrincipalInfoAction implements PrivilegedAction {
        Principal principal;
        Object credential;

        public SetPrincipalInfoAction(Principal principal, Object credential) {
            this.principal = principal;
            this.credential = credential;
        }

        public Object run() {
            try {
                Class saClass = Class.forName("org.jboss.security.SecurityAssociation");
                Method setCredentialMethod = saClass.getMethod("setCredential", Object.class);

                setCredentialMethod.invoke(null, credential);
                credential = null;

                Method setPrincipleMethod = saClass.getMethod("setPrincipal", Principal.class);

                setPrincipleMethod.invoke(null, principal);
                principal = null;

                return null;
            } catch (NoSuchMethodException e) {
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            } catch (InvocationTargetException e) {
                e.printStackTrace();
            }
            return null;
        }
        static void setPrincipalInfo(Principal principal, Object credential) {
            SetPrincipalInfoAction action = new SetPrincipalInfoAction(principal, credential);
            //noinspection unchecked
            AccessController.doPrivileged(action);
        }
    }



    /* GH: an aborted attempt at manually changing the polling type
    public class RMIAdaptorExtension extends RMIConnectorImpl {
        public RMIAdaptorExtension(RMIAdaptor rmiAdaptor) {
            super(rmiAdaptor);

            try {
                Field field = RMIConnectorImpl.class.getField("mEventType");
                if (!Modifier.isPrivate(field.getModifiers())) {
                    field.set(this, new Integer(RMIConnectorImpl.NOTIFICATION_TYPE_POLLING));
                }
            } catch (NoSuchFieldException nsfe) {
            } catch (IllegalAccessException iae) {
            }
        }
    }
    */


    public void doDisconnect() {

    }

    /*   public Object getMEJB() {
            if (mejb == null) {
                mejb = retrieveMEJB();
            }
            return mejb;
        }


        private Management retrieveMEJB() {
            try {
                Context ic = getInitialContext();
                java.lang.Object objref = ic.lookup(MEJB_JNDI);
                ManagementHome home =
                    (ManagementHome)PortableRemoteObject.narrow(objref,ManagementHome.class);
                Management mejb = home.create();
                return mejb;
            } catch(NamingException ne) {
                ErrorManager.getDefault().notify(ne);
            } catch(RemoteException re) {
                 ErrorManager.getDefault().notify(re);
            } catch(Exception ce) {
                 ErrorManager.getDefault().notify(ce);
            }
            return null;
        }
    */
    public MBeanServer getMBeanServer() {
        return this.mbeanServer;
    }

    public void resetPrincipalInfo() throws Exception {
        Class simplePrincipalClass = Class.forName("org.jboss.security.SimplePrincipal");
        Principal principal = (Principal) simplePrincipalClass.getConstructor(String.class).newInstance(
            getConnectionSettings().getPrincipal());
        SetPrincipalInfoAction.setPrincipalInfo(principal, getConnectionSettings().getCredentials());
    }
}
