/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http2;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.http2.CloseState;
import org.eclipse.jetty.http2.ErrorCode;
import org.eclipse.jetty.http2.FlowControlStrategy;
import org.eclipse.jetty.http2.HTTP2Flusher;
import org.eclipse.jetty.http2.HTTP2Stream;
import org.eclipse.jetty.http2.ISession;
import org.eclipse.jetty.http2.IStream;
import org.eclipse.jetty.http2.api.Session;
import org.eclipse.jetty.http2.api.Stream;
import org.eclipse.jetty.http2.frames.DataFrame;
import org.eclipse.jetty.http2.frames.DisconnectFrame;
import org.eclipse.jetty.http2.frames.Frame;
import org.eclipse.jetty.http2.frames.FrameType;
import org.eclipse.jetty.http2.frames.GoAwayFrame;
import org.eclipse.jetty.http2.frames.HeadersFrame;
import org.eclipse.jetty.http2.frames.PingFrame;
import org.eclipse.jetty.http2.frames.PriorityFrame;
import org.eclipse.jetty.http2.frames.PushPromiseFrame;
import org.eclipse.jetty.http2.frames.ResetFrame;
import org.eclipse.jetty.http2.frames.SettingsFrame;
import org.eclipse.jetty.http2.frames.WindowUpdateFrame;
import org.eclipse.jetty.http2.generator.Generator;
import org.eclipse.jetty.http2.parser.Parser;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.util.Atomics;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.CountingCallback;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.Invocable;
import org.eclipse.jetty.util.thread.Scheduler;

@ManagedObject
public abstract class HTTP2Session
extends ContainerLifeCycle
implements ISession,
Parser.Listener {
    private static final Logger LOG = Log.getLogger(HTTP2Session.class);
    private final ConcurrentMap<Integer, IStream> streams = new ConcurrentHashMap<Integer, IStream>();
    private final AtomicInteger streamIds = new AtomicInteger();
    private final AtomicInteger lastStreamId = new AtomicInteger();
    private final AtomicInteger localStreamCount = new AtomicInteger();
    private final AtomicInteger remoteStreamCount = new AtomicInteger();
    private final AtomicInteger sendWindow = new AtomicInteger();
    private final AtomicInteger recvWindow = new AtomicInteger();
    private final AtomicReference<CloseState> closed = new AtomicReference<CloseState>(CloseState.NOT_CLOSED);
    private final Scheduler scheduler;
    private final EndPoint endPoint;
    private final Generator generator;
    private final Session.Listener listener;
    private final FlowControlStrategy flowControl;
    private final HTTP2Flusher flusher;
    private int maxLocalStreams;
    private int maxRemoteStreams;
    private long streamIdleTimeout;
    private int initialSessionRecvWindow;
    private boolean pushEnabled;
    private long idleTime;

    public HTTP2Session(Scheduler scheduler, EndPoint endPoint, Generator generator, Session.Listener listener, FlowControlStrategy flowControl, int initialStreamId) {
        this.scheduler = scheduler;
        this.endPoint = endPoint;
        this.generator = generator;
        this.listener = listener;
        this.flowControl = flowControl;
        this.flusher = new HTTP2Flusher(this);
        this.maxLocalStreams = -1;
        this.maxRemoteStreams = -1;
        this.streamIds.set(initialStreamId);
        this.streamIdleTimeout = endPoint.getIdleTimeout();
        this.sendWindow.set(65535);
        this.recvWindow.set(65535);
        this.pushEnabled = true;
        this.idleTime = System.nanoTime();
    }

    @Override
    protected void doStart() throws Exception {
        this.addBean(this.flowControl);
        super.doStart();
    }

    @Override
    protected void doStop() throws Exception {
        super.doStop();
        this.close(ErrorCode.NO_ERROR.code, "stop", new Callback(){

            @Override
            public void succeeded() {
                HTTP2Session.this.disconnect();
            }

            @Override
            public void failed(Throwable x) {
                HTTP2Session.this.disconnect();
            }

            @Override
            public Invocable.InvocationType getInvocationType() {
                return Invocable.InvocationType.NON_BLOCKING;
            }
        });
    }

    @ManagedAttribute(value="The flow control strategy", readonly=true)
    public FlowControlStrategy getFlowControlStrategy() {
        return this.flowControl;
    }

    public int getMaxLocalStreams() {
        return this.maxLocalStreams;
    }

    public void setMaxLocalStreams(int maxLocalStreams) {
        this.maxLocalStreams = maxLocalStreams;
    }

    public int getMaxRemoteStreams() {
        return this.maxRemoteStreams;
    }

    public void setMaxRemoteStreams(int maxRemoteStreams) {
        this.maxRemoteStreams = maxRemoteStreams;
    }

    @ManagedAttribute(value="The stream's idle timeout")
    public long getStreamIdleTimeout() {
        return this.streamIdleTimeout;
    }

    public void setStreamIdleTimeout(long streamIdleTimeout) {
        this.streamIdleTimeout = streamIdleTimeout;
    }

    @ManagedAttribute(value="The initial size of session's flow control receive window")
    public int getInitialSessionRecvWindow() {
        return this.initialSessionRecvWindow;
    }

    public void setInitialSessionRecvWindow(int initialSessionRecvWindow) {
        this.initialSessionRecvWindow = initialSessionRecvWindow;
    }

    public EndPoint getEndPoint() {
        return this.endPoint;
    }

    public Generator getGenerator() {
        return this.generator;
    }

    @Override
    public void onData(DataFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        int streamId = frame.getStreamId();
        final IStream stream = this.getStream(streamId);
        final int flowControlLength = frame.remaining() + frame.padding();
        this.flowControl.onDataReceived(this, stream, flowControlLength);
        if (stream != null) {
            if (this.getRecvWindow() < 0) {
                this.close(ErrorCode.FLOW_CONTROL_ERROR.code, "session_window_exceeded", Callback.NOOP);
            } else {
                stream.process(frame, new Callback(){

                    @Override
                    public void succeeded() {
                        this.complete();
                    }

                    @Override
                    public void failed(Throwable x) {
                        this.complete();
                    }

                    private void complete() {
                        HTTP2Session.this.notIdle();
                        stream.notIdle();
                        HTTP2Session.this.flowControl.onDataConsumed(HTTP2Session.this, stream, flowControlLength);
                    }
                });
            }
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Ignoring {}, stream #{} not found", frame, streamId);
            }
            this.flowControl.onDataConsumed(this, null, flowControlLength);
        }
    }

    @Override
    public abstract void onHeaders(HeadersFrame var1);

    @Override
    public void onPriority(PriorityFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
    }

    @Override
    public void onReset(ResetFrame frame) {
        IStream stream;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        if ((stream = this.getStream(frame.getStreamId())) != null) {
            stream.process(frame, Callback.NOOP);
        } else {
            this.notifyReset(this, frame);
        }
    }

    @Override
    public void onSettings(SettingsFrame frame) {
        this.onSettings(frame, true);
    }

    public void onSettings(SettingsFrame frame, boolean reply) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        if (frame.isReply()) {
            return;
        }
        block8: for (Map.Entry<Integer, Integer> entry : frame.getSettings().entrySet()) {
            int key = entry.getKey();
            int value = entry.getValue();
            switch (key) {
                case 1: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update HPACK header table size to {}", value);
                    }
                    this.generator.setHeaderTableSize(value);
                    continue block8;
                }
                case 2: {
                    if (value != 0 && value != 1) {
                        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "invalid_settings_enable_push");
                        return;
                    }
                    this.pushEnabled = value == 1;
                    continue block8;
                }
                case 3: {
                    this.maxLocalStreams = value;
                    if (!LOG.isDebugEnabled()) continue block8;
                    LOG.debug("Update max local concurrent streams to {}", this.maxLocalStreams);
                    continue block8;
                }
                case 4: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update initial window size to {}", value);
                    }
                    this.flowControl.updateInitialStreamWindow(this, value, false);
                    continue block8;
                }
                case 5: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update max frame size to {}", value);
                    }
                    if (value < 16384 || value > 0xFFFFFF) {
                        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "invalid_settings_max_frame_size");
                        return;
                    }
                    this.generator.setMaxFrameSize(value);
                    continue block8;
                }
                case 6: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Update max header list size to {}", value);
                    }
                    this.generator.setMaxHeaderListSize(value);
                    continue block8;
                }
            }
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Unknown setting {}:{}", key, value);
        }
        this.notifySettings(this, frame);
        if (reply) {
            SettingsFrame replyFrame = new SettingsFrame(Collections.emptyMap(), true);
            this.settings(replyFrame, Callback.NOOP);
        }
    }

    @Override
    public void onPing(PingFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        if (frame.isReply()) {
            this.notifyPing(this, frame);
        } else {
            PingFrame reply = new PingFrame(frame.getPayload(), true);
            this.control(null, Callback.NOOP, reply);
        }
    }

    @Override
    public void onGoAway(GoAwayFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.REMOTELY_CLOSED)) continue block3;
                    this.notifyClose(this, frame);
                    this.control(null, Callback.NOOP, new DisconnectFrame());
                    return;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignored {}, already closed", frame);
        }
    }

    @Override
    public void onWindowUpdate(WindowUpdateFrame frame) {
        int streamId;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", frame);
        }
        if ((streamId = frame.getStreamId()) > 0) {
            IStream stream = this.getStream(streamId);
            if (stream != null) {
                stream.process(frame, Callback.NOOP);
                this.onWindowUpdate(stream, frame);
            }
        } else {
            this.onWindowUpdate(null, frame);
        }
    }

    @Override
    public void onConnectionFailure(int error, String reason) {
        this.close(error, reason, Callback.NOOP);
        this.notifyFailure(this, new IOException(String.format("%d/%s", error, reason)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void newStream(HeadersFrame frame, Promise<Stream> promise, Stream.Listener listener) {
        boolean queued;
        HTTP2Session hTTP2Session = this;
        synchronized (hTTP2Session) {
            IStream stream;
            int streamId = frame.getStreamId();
            if (streamId <= 0) {
                streamId = this.streamIds.getAndAdd(2);
                PriorityFrame priority = frame.getPriority();
                priority = priority == null ? null : new PriorityFrame(streamId, priority.getParentStreamId(), priority.getWeight(), priority.isExclusive());
                frame = new HeadersFrame(streamId, frame.getMetaData(), priority, frame.isEndStream());
            }
            if ((stream = this.createLocalStream(streamId, promise)) == null) {
                return;
            }
            stream.setListener(listener);
            ControlEntry entry = new ControlEntry(frame, stream, new PromiseCallback(promise, stream));
            queued = this.flusher.append(entry);
        }
        if (queued) {
            this.flusher.iterate();
        }
    }

    @Override
    public int priority(PriorityFrame frame, Callback callback) {
        int streamId = frame.getStreamId();
        IStream stream = (IStream)this.streams.get(streamId);
        if (stream == null) {
            streamId = this.streamIds.getAndAdd(2);
            frame = new PriorityFrame(streamId, frame.getParentStreamId(), frame.getWeight(), frame.isExclusive());
        }
        this.control(stream, callback, frame);
        return streamId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void push(IStream stream, Promise<Stream> promise, PushPromiseFrame frame, Stream.Listener listener) {
        boolean queued;
        HTTP2Session hTTP2Session = this;
        synchronized (hTTP2Session) {
            int streamId = this.streamIds.getAndAdd(2);
            frame = new PushPromiseFrame(frame.getStreamId(), streamId, frame.getMetaData());
            IStream pushStream = this.createLocalStream(streamId, promise);
            if (pushStream == null) {
                return;
            }
            pushStream.setListener(listener);
            ControlEntry entry = new ControlEntry(frame, pushStream, new PromiseCallback(promise, pushStream));
            queued = this.flusher.append(entry);
        }
        if (queued) {
            this.flusher.iterate();
        }
    }

    @Override
    public void settings(SettingsFrame frame, Callback callback) {
        this.control(null, callback, frame);
    }

    @Override
    public void ping(PingFrame frame, Callback callback) {
        if (frame.isReply()) {
            callback.failed(new IllegalArgumentException());
        } else {
            this.control(null, callback, frame);
        }
    }

    protected void reset(ResetFrame frame, Callback callback) {
        this.control(this.getStream(frame.getStreamId()), callback, frame);
    }

    @Override
    public boolean close(int error, String reason, Callback callback) {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.LOCALLY_CLOSED)) continue block3;
                    byte[] payload = null;
                    if (reason != null) {
                        reason = reason.substring(0, Math.min(reason.length(), 32));
                        payload = reason.getBytes(StandardCharsets.UTF_8);
                    }
                    GoAwayFrame frame = new GoAwayFrame(this.lastStreamId.get(), error, payload);
                    this.control(null, callback, frame);
                    return true;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignoring close {}/{}, already closed", error, reason);
        }
        callback.succeeded();
        return false;
    }

    @Override
    public boolean isClosed() {
        return this.closed.get() != CloseState.NOT_CLOSED;
    }

    private void control(IStream stream, Callback callback, Frame frame) {
        this.frames(stream, callback, frame, Frame.EMPTY_ARRAY);
    }

    @Override
    public void frames(IStream stream, Callback callback, Frame frame, Frame ... frames) {
        int length = frames.length;
        if (length == 0) {
            this.frame(new ControlEntry(frame, stream, callback), true);
        } else {
            callback = new CountingCallback(callback, 1 + length);
            this.frame(new ControlEntry(frame, stream, callback), false);
            for (int i = 1; i <= length; ++i) {
                this.frame(new ControlEntry(frames[i - 1], stream, callback), i == length);
            }
        }
    }

    @Override
    public void data(IStream stream, Callback callback, DataFrame frame) {
        this.frame(new DataEntry(frame, stream, callback), true);
    }

    private void frame(HTTP2Flusher.Entry entry, boolean flush) {
        boolean queued;
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} {}", flush ? "Sending" : "Queueing", entry.frame);
        }
        boolean bl = queued = entry.frame.getType() == FrameType.PING ? this.flusher.prepend(entry) : this.flusher.append(entry);
        if (queued && flush) {
            if (entry.stream != null) {
                entry.stream.notIdle();
            }
            this.flusher.iterate();
        }
    }

    protected IStream createLocalStream(int streamId, Promise<Stream> promise) {
        int localCount;
        do {
            localCount = this.localStreamCount.get();
            int maxCount = this.getMaxLocalStreams();
            if (maxCount < 0 || localCount < maxCount) continue;
            promise.failed(new IllegalStateException("Max local stream count " + maxCount + " exceeded"));
            return null;
        } while (!this.localStreamCount.compareAndSet(localCount, localCount + 1));
        IStream stream = this.newStream(streamId, true);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created local {}", stream);
            }
            return stream;
        }
        promise.failed(new IllegalStateException("Duplicate stream " + streamId));
        return null;
    }

    protected IStream createRemoteStream(int streamId) {
        int remoteCount;
        do {
            remoteCount = this.remoteStreamCount.get();
            int maxCount = this.getMaxRemoteStreams();
            if (maxCount < 0 || remoteCount < maxCount) continue;
            this.reset(new ResetFrame(streamId, ErrorCode.REFUSED_STREAM_ERROR.code), Callback.NOOP);
            return null;
        } while (!this.remoteStreamCount.compareAndSet(remoteCount, remoteCount + 1));
        IStream stream = this.newStream(streamId, false);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            this.updateLastStreamId(streamId);
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created remote {}", stream);
            }
            return stream;
        }
        this.close(ErrorCode.PROTOCOL_ERROR.code, "duplicate_stream", Callback.NOOP);
        return null;
    }

    protected IStream newStream(int streamId, boolean local) {
        return new HTTP2Stream(this.scheduler, this, streamId, local);
    }

    @Override
    public void removeStream(IStream stream) {
        IStream removed = (IStream)this.streams.remove(stream.getId());
        if (removed != null) {
            assert (removed == stream);
            boolean local = stream.isLocal();
            if (local) {
                this.localStreamCount.decrementAndGet();
            } else {
                this.remoteStreamCount.decrementAndGet();
            }
            this.flowControl.onStreamDestroyed(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Removed {} {}", local ? "local" : "remote", stream);
            }
        }
    }

    @Override
    public Collection<Stream> getStreams() {
        ArrayList<Stream> result = new ArrayList<Stream>();
        result.addAll(this.streams.values());
        return result;
    }

    @ManagedAttribute(value="The number of active streams")
    public int getStreamCount() {
        return this.streams.size();
    }

    @Override
    public IStream getStream(int streamId) {
        return (IStream)this.streams.get(streamId);
    }

    @ManagedAttribute(value="The flow control send window", readonly=true)
    public int getSendWindow() {
        return this.sendWindow.get();
    }

    @ManagedAttribute(value="The flow control receive window", readonly=true)
    public int getRecvWindow() {
        return this.recvWindow.get();
    }

    @Override
    public int updateSendWindow(int delta) {
        return this.sendWindow.getAndAdd(delta);
    }

    @Override
    public int updateRecvWindow(int delta) {
        return this.recvWindow.getAndAdd(delta);
    }

    @Override
    public void onWindowUpdate(IStream stream, WindowUpdateFrame frame) {
        this.flusher.window(stream, frame);
    }

    @Override
    @ManagedAttribute(value="Whether HTTP/2 push is enabled", readonly=true)
    public boolean isPushEnabled() {
        return this.pushEnabled;
    }

    @Override
    public void onShutdown() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Shutting down {}", this);
        }
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Abrupt close for {}", this);
                }
                this.abort(new ClosedChannelException());
                break;
            }
            case LOCALLY_CLOSED: {
                this.control(null, Callback.NOOP, new DisconnectFrame());
                break;
            }
            case REMOTELY_CLOSED: {
                break;
            }
        }
    }

    @Override
    public boolean onIdleTimeout() {
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                long elapsed = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.idleTime);
                if (elapsed < this.endPoint.getIdleTimeout()) {
                    return false;
                }
                return this.notifyIdleTimeout(this);
            }
            case LOCALLY_CLOSED: 
            case REMOTELY_CLOSED: {
                this.abort(new TimeoutException("Idle timeout " + this.endPoint.getIdleTimeout() + " ms"));
                return false;
            }
        }
        return false;
    }

    private void notIdle() {
        this.idleTime = System.nanoTime();
    }

    @Override
    public void onFrame(Frame frame) {
        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "upgrade");
    }

    public void disconnect() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Disconnecting {}", this);
        }
        this.endPoint.close();
    }

    private void terminate() {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: 
                case LOCALLY_CLOSED: 
                case REMOTELY_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.CLOSED)) continue block3;
                    this.flusher.terminate();
                    for (IStream stream : this.streams.values()) {
                        stream.close();
                    }
                    this.streams.clear();
                    this.disconnect();
                    return;
                }
            }
            break;
        }
    }

    protected void abort(Throwable failure) {
        this.terminate();
        this.notifyFailure(this, failure);
    }

    public boolean isDisconnected() {
        return !this.endPoint.isOpen();
    }

    private void updateLastStreamId(int streamId) {
        Atomics.updateMax(this.lastStreamId, streamId);
    }

    protected Stream.Listener notifyNewStream(Stream stream, HeadersFrame frame) {
        try {
            return this.listener.onNewStream(stream, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
            return null;
        }
    }

    protected void notifySettings(Session session, SettingsFrame frame) {
        try {
            this.listener.onSettings(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyPing(Session session, PingFrame frame) {
        try {
            this.listener.onPing(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyReset(Session session, ResetFrame frame) {
        try {
            this.listener.onReset(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyClose(Session session, GoAwayFrame frame) {
        try {
            this.listener.onClose(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected boolean notifyIdleTimeout(Session session) {
        try {
            return this.listener.onIdleTimeout(session);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
            return true;
        }
    }

    protected void notifyFailure(Session session, Throwable failure) {
        try {
            this.listener.onFailure(session, failure);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    public String toString() {
        return String.format("%s@%x{l:%s <-> r:%s,queueSize=%d,sendWindow=%s,recvWindow=%s,streams=%d,%s}", this.getClass().getSimpleName(), this.hashCode(), this.getEndPoint().getLocalAddress(), this.getEndPoint().getRemoteAddress(), this.flusher.getQueueSize(), this.sendWindow, this.recvWindow, this.streams.size(), this.closed);
    }

    private static class PromiseCallback<C>
    implements Callback {
        private final Promise<C> promise;
        private final C value;

        private PromiseCallback(Promise<C> promise, C value) {
            this.promise = promise;
            this.value = value;
        }

        @Override
        public void succeeded() {
            this.promise.succeeded(this.value);
        }

        @Override
        public void failed(Throwable x) {
            this.promise.failed(x);
        }
    }

    private class DataEntry
    extends HTTP2Flusher.Entry {
        private int remaining;
        private int generated;

        private DataEntry(DataFrame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
            this.remaining = frame.remaining();
        }

        @Override
        public int dataRemaining() {
            return this.remaining;
        }

        @Override
        protected boolean generate(ByteBufferPool.Lease lease) {
            int toWrite = this.dataRemaining();
            int sessionSendWindow = HTTP2Session.this.getSendWindow();
            int streamSendWindow = this.stream.updateSendWindow(0);
            int window = Math.min(streamSendWindow, sessionSendWindow);
            if (window <= 0 && toWrite > 0) {
                return false;
            }
            int length = Math.min(toWrite, window);
            int generated = HTTP2Session.this.generator.data(lease, (DataFrame)this.frame, length);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Generated {}, length/window/data={}/{}/{}", this.frame, generated, window, toWrite);
            }
            this.generated += generated;
            this.remaining -= generated;
            HTTP2Session.this.flowControl.onDataSending(this.stream, generated);
            return true;
        }

        @Override
        public void succeeded() {
            HTTP2Session.this.flowControl.onDataSent(this.stream, this.generated);
            this.generated = 0;
            DataFrame dataFrame = (DataFrame)this.frame;
            if (this.dataRemaining() == 0) {
                if (this.stream.updateClose(dataFrame.isEndStream(), true)) {
                    HTTP2Session.this.removeStream(this.stream);
                }
                super.succeeded();
            }
        }
    }

    private class ControlEntry
    extends HTTP2Flusher.Entry {
        private ControlEntry(Frame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
        }

        @Override
        protected boolean generate(ByteBufferPool.Lease lease) {
            HTTP2Session.this.generator.control(lease, this.frame);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Generated {}", this.frame);
            }
            this.prepare();
            return true;
        }

        private void prepare() {
            switch (this.frame.getType()) {
                case SETTINGS: {
                    SettingsFrame settingsFrame = (SettingsFrame)this.frame;
                    Integer initialWindow = settingsFrame.getSettings().get(4);
                    if (initialWindow == null) break;
                    HTTP2Session.this.flowControl.updateInitialStreamWindow(HTTP2Session.this, initialWindow, true);
                    break;
                }
            }
        }

        @Override
        public void succeeded() {
            switch (this.frame.getType()) {
                case HEADERS: {
                    HeadersFrame headersFrame = (HeadersFrame)this.frame;
                    if (!this.stream.updateClose(headersFrame.isEndStream(), true)) break;
                    HTTP2Session.this.removeStream(this.stream);
                    break;
                }
                case RST_STREAM: {
                    if (this.stream == null) break;
                    this.stream.close();
                    HTTP2Session.this.removeStream(this.stream);
                    break;
                }
                case PUSH_PROMISE: {
                    this.stream.updateClose(true, false);
                    break;
                }
                case GO_AWAY: {
                    HTTP2Session.this.getEndPoint().shutdownOutput();
                    break;
                }
                case WINDOW_UPDATE: {
                    HTTP2Session.this.flowControl.windowUpdate(HTTP2Session.this, this.stream, (WindowUpdateFrame)this.frame);
                    break;
                }
                case DISCONNECT: {
                    HTTP2Session.this.terminate();
                    break;
                }
            }
            super.succeeded();
        }
    }
}

