/*
 * Copyright 2011 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.artificer.repository.test;

import org.apache.commons.io.IOUtils;
import org.artificer.common.ArtifactContent;
import org.artificer.common.ArtificerException;
import org.artificer.common.ontology.ArtificerOntology;
import org.artificer.common.ontology.ArtificerOntologyClass;
import org.artificer.common.query.ArtifactSummary;
import org.artificer.repository.query.ArtificerQuery;
import org.artificer.repository.query.PagedResult;
import org.junit.Assert;
import org.junit.Test;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.BaseArtifactEnum;
import org.oasis_open.docs.s_ramp.ns.s_ramp_v1.Document;

import java.io.InputStream;
import java.util.List;


/**
 * Tests that classifications can be queried.
 *
 * @author eric.wittmann@redhat.com
 */
public class ClassificationQueryTest extends AbstractNoAuditingPersistenceTest {

	@Test
	public void testDerivedRelationshipQueries() throws Exception {
		createOntology();

        Document doc = addDocument("no-classifications");
        Document docChina = addDocument("one-classification: china", "China");
        Document docJapan = addDocument("one-classification: japan", "Japan");
        Document docGermany = addDocument("one-classification: germany", "Germany");

		// Verify that the docs are available
		ArtificerQuery query = queryManager.createQuery("/s-ramp/core/Document");
        PagedResult<ArtifactSummary> artifactSet = query.executeQuery();
		Assert.assertNotNull(artifactSet);
		Assert.assertEquals(4, artifactSet.getTotalSize());

		// Make sure there's only one with the given name
		query = queryManager.createQuery("/s-ramp/core/Document[@name = ?]");
		query.setString("no-classifications");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, doc);

		// Should get just the one classified by China
		query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:exactlyClassifiedByAllOf(., 'China')]");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, docChina);

		// Should get zero artifacts
        query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:exactlyClassifiedByAllOf(., 'Asia')]");
		artifactSet = query.executeQuery();
		Assert.assertNotNull(artifactSet);
		Assert.assertEquals(0, artifactSet.getTotalSize());

		// Should get just the one classified by Germany
		query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:exactlyClassifiedByAllOf(., 'Germany')]");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, docGermany);

		// Should get zero artifacts
        query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:exactlyClassifiedByAllOf(., 'China', 'Germany')]");
		artifactSet = query.executeQuery();
		Assert.assertNotNull(artifactSet);
		Assert.assertEquals(0, artifactSet.getTotalSize());

		// Should get all classified artifacts
		query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:classifiedByAllOf(., 'World')]");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, docJapan, docChina, docGermany);

        // Should get two artifacts - japan and china
		query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:classifiedByAllOf(., 'Asia')]");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, docJapan, docChina);

		// Should get two artifacts - japan and china
		query = queryManager.createQuery("/s-ramp/core/Document[s-ramp:classifiedByAnyOf(., 'Japan', 'China')]");
		artifactSet = query.executeQuery();
        assertResults(artifactSet, docJapan, docChina);

        // Test not()
        query = queryManager.createQuery("/s-ramp/core/Document[xp2:not(s-ramp:classifiedByAnyOf(., 'Japan', 'China'))]");
        artifactSet = query.executeQuery();
        assertResults(artifactSet, doc, docGermany);
	}

    private void assertResults(PagedResult<ArtifactSummary> artifactSet, Document... docs) throws Exception {
        Assert.assertNotNull(artifactSet);
        Assert.assertEquals(docs.length, artifactSet.getTotalSize());
        List<ArtifactSummary> artifacts = artifactSet.getResults();
        for (ArtifactSummary artifact : artifacts) {
            boolean found = false;
            for (Document doc : docs) {
                if (doc.getUuid().equals(artifact.getUuid())) {
                    found = true;
                }
            }
            Assert.assertTrue(found);
        }
    }

	/**
	 * @throws org.artificer.common.ArtificerException
	 */
	private ArtificerOntology createOntology() throws ArtificerException {
		ArtificerOntology ontology = new ArtificerOntology();
		ontology.setBase("urn:example.org/test2");
		ontology.setLabel("Test Ontology #2");
		ontology.setAnnotation("This is my second test ontology.");

		ArtificerOntologyClass world = createClass(ontology, null, "World", "World", "The entire world");
		ArtificerOntologyClass asia = createClass(ontology, world, "Asia", "Asia", null);
		ArtificerOntologyClass europe = createClass(ontology, world, "Europe", "Europe", "Two world wars");
		ArtificerOntologyClass japan = createClass(ontology, asia, "Japan", "Japan", "Samurai *and* ninja?  Not fair.");
		ArtificerOntologyClass china = createClass(ontology, asia, "China", "China", "Gunpowder!");
		ArtificerOntologyClass uk = createClass(ontology, europe, "UnitedKingdom", "United Kingdom", "The food could be better");
		ArtificerOntologyClass germany = createClass(ontology, europe, "Germany", "Germany", "The fatherland");

		ontology.getRootClasses().add(world);

		world.getChildren().add(asia);
		world.getChildren().add(europe);
		asia.getChildren().add(japan);
		asia.getChildren().add(china);
		europe.getChildren().add(uk);
		europe.getChildren().add(germany);

		return persistenceManager.persistOntology(ontology);
	}

	/**
	 * Creates a test class.
	 * @param ontology
	 * @param parent
	 * @param id
	 * @param label
	 * @param comment
	 */
	private ArtificerOntologyClass createClass(ArtificerOntology ontology, ArtificerOntologyClass parent, String id, String label, String comment) {
		ArtificerOntologyClass rval = ontology.createClass(id);
		rval.setParent(parent);
		rval.setAnnotation(comment);
		rval.setLabel(label);
		return rval;
	}

	/**
	 * @throws org.artificer.common.ArtificerException
	 */
	private Document addDocument(String name, String ... classifications) throws Exception {
		InputStream contentStream = null;
		try {
			String artifactFileName = "s-ramp-press-release.pdf";
			contentStream = this.getClass().getResourceAsStream("/sample-files/core/" + artifactFileName);
			Document document = new Document();
			document.setName(name);
			document.setArtifactType(BaseArtifactEnum.DOCUMENT);
			for (String classification : classifications) {
				document.getClassifiedBy().add(classification);
			}
			return (Document) persistenceManager.persistArtifact(document, new ArtifactContent(artifactFileName, contentStream));
		} finally {
			IOUtils.closeQuietly(contentStream);
		}
	}

}
