/*
  * JBoss, Home of Professional Open Source
  * Copyright 2005, JBoss Inc., and individual contributors as indicated
  * by the @authors tag. See the copyright.txt in the distribution for a
  * full listing of individual contributors.
  *
  * This is free software; you can redistribute it and/or modify it
  * under the terms of the GNU Lesser General Public License as
  * published by the Free Software Foundation; either version 2.1 of
  * the License, or (at your option) any later version.
  *
  * This software is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  * Lesser General Public License for more details.
  *
  * You should have received a copy of the GNU Lesser General Public
  * License along with this software; if not, write to the Free
  * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
  * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
  */
package org.jboss.aspects.security;

import java.lang.reflect.UndeclaredThrowableException;
import java.security.AccessController;
import java.security.Principal;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;

import javax.security.auth.Subject;
import javax.security.jacc.PolicyContext;
import javax.security.jacc.PolicyContextException;

import org.jboss.logging.Logger;
import org.jboss.security.SecurityContext;
import org.jboss.security.SecurityContextAssociation;
import org.jboss.security.SecurityContextFactory;
import org.jboss.security.SimplePrincipal;
import org.jboss.security.identity.Identity;
import org.jboss.security.identity.plugins.SimpleIdentity;

/** A collection of privileged actions for this package
 * @author Scott.Stark@jboss.org
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 * @author Anil.Saldhana@redhat.com
 * @version $Revison: $
 */

@SuppressWarnings({"unchecked", "unused"}) 
class SecurityActions
{
   private static final Logger log = Logger.getLogger(SecurityActions.class);
 
   interface PolicyContextActions
   {
      /** The JACC PolicyContext key for the current Subject */
      static final String SUBJECT_CONTEXT_KEY = "javax.security.auth.Subject.container";
      PolicyContextActions PRIVILEGED = new PolicyContextActions()
      {
         private final PrivilegedExceptionAction exAction = new PrivilegedExceptionAction()
         {
            public Object run() throws Exception
            {
               return (Subject) PolicyContext.getContext(SUBJECT_CONTEXT_KEY);
            }
         };
         public Subject getContextSubject()
            throws PolicyContextException
         {
            try
            {
               return (Subject) AccessController.doPrivileged(exAction);
            }
            catch(PrivilegedActionException e)
            {
               Exception ex = e.getException();
               if( ex instanceof PolicyContextException )
                  throw (PolicyContextException) ex;
               else
                  throw new UndeclaredThrowableException(ex);
            }
         }
      };

      PolicyContextActions NON_PRIVILEGED = new PolicyContextActions()
      {
         public Subject getContextSubject()
            throws PolicyContextException
         {
            return (Subject) PolicyContext.getContext(SUBJECT_CONTEXT_KEY);
         }
      };

      Subject getContextSubject()
         throws PolicyContextException;
   }
   
   static ClassLoader getContextClassLoader()
   {
      return TCLAction.UTIL.getContextClassLoader();
   }

   static void setContextClassLoader(ClassLoader loader)
   {
      TCLAction.UTIL.setContextClassLoader(loader);
   }

   static Principal getCallerPrincipal()
   {
      return AccessController.doPrivileged(new PrivilegedAction<Principal>()
      {
         public Principal run()
         {
            org.jboss.security.RunAs runas = getSecurityContext().getIncomingRunAs();
            if(runas != null)
               return new SimplePrincipal(runas.getName());
            return getSecurityContext().getUtil().getUserPrincipal(); 
         }
      }); 
   }

   static Principal getPrincipal()
   {
      return AccessController.doPrivileged(new PrivilegedAction<Principal>()
      {
         public Principal run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            return sc != null ? sc.getUtil().getUserPrincipal() : null; 
         }
      });
   }

   static void setPrincipal(final Principal principal)
   {
      AccessController.doPrivileged(new PrivilegedAction<Object>()
      {
         public Object run()
         {
            Identity identity = new SimpleIdentity(principal.getName());

            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            sc.getUtil().addIdentity(identity); 
            return null;
         }
     }); 
   }

   static Object getCredential()
   {
      return AccessController.doPrivileged(new PrivilegedAction<Object>()
      {
         public Object run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            return sc.getUtil().getCredential(); 
         }
      });
   }

   static void pushSubjectContext(final Principal principal, final Object credential,
      final Subject subject)
   {
      AccessController.doPrivileged(new PrivilegedAction<Object>()
      {
         public Object run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            sc.getUtil().createSubjectInfo(principal, 
                  credential, subject);
            return null; 
         }
      });
   }
   static void popSubjectContext()
   {
      AccessController.doPrivileged(new PrivilegedAction<Object>()
      {
         public Object run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            sc.setSubjectInfo(null);
            return null; 
         }
      });
   }

   static org.jboss.security.RunAs peekRunAsIdentity()
   {
      return AccessController.doPrivileged(new PrivilegedAction<org.jboss.security.RunAs>()
      {
         public org.jboss.security.RunAs run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            
            return sc.getIncomingRunAs(); 
         }
      });
   }

   static void pushRunAsIdentity(final org.jboss.security.RunAs runas)
   {     
      AccessController.doPrivileged(
            new PrivilegedAction<Object>()
      {
         public Object run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
            sc.setIncomingRunAs(runas);
            return null;
         }
      });
   }

   static org.jboss.security.RunAs popRunAsIdentity()
   {
      return AccessController.doPrivileged(
            new PrivilegedAction<org.jboss.security.RunAs>()
      {
         public org.jboss.security.RunAs run()
         {
            SecurityContext sc = getSecurityContext();
            if(sc == null)
               throw new RuntimeException("No Security Context");
           
            return sc.getIncomingRunAs(); 
         }
      });       
   }

   static Exception getContextException()
   {
      return AccessController.doPrivileged(
            new PrivilegedAction<Exception>()
      {
         public Exception run()
         {
            return (Exception) getSecurityContext().getData().get("CONTEXTEXCEPTION"); 
         }
      }); 
   }

   static Subject getContextSubject()
      throws PolicyContextException
   {
      if(System.getSecurityManager() == null)
      {
         return PolicyContextActions.NON_PRIVILEGED.getContextSubject();
      }
      else
      {
         return PolicyContextActions.PRIVILEGED.getContextSubject();
      }      
   }
   
   interface TCLAction
   {
      class UTIL
      {
         static TCLAction getTCLAction()
         {
            return System.getSecurityManager() == null ? NON_PRIVILEGED : PRIVILEGED;
         }

         static ClassLoader getContextClassLoader()
         {
            return getTCLAction().getContextClassLoader();
         }

         static ClassLoader getContextClassLoader(Thread thread)
         {
            return getTCLAction().getContextClassLoader(thread);
         }

         static void setContextClassLoader(ClassLoader cl)
         {
            getTCLAction().setContextClassLoader(cl);
         }

         static void setContextClassLoader(Thread thread, ClassLoader cl)
         {
            getTCLAction().setContextClassLoader(thread, cl);
         }
      }

      TCLAction NON_PRIVILEGED = new TCLAction()
      {
         public ClassLoader getContextClassLoader()
         {
            return Thread.currentThread().getContextClassLoader();
         }

         public ClassLoader getContextClassLoader(Thread thread)
         {
            return thread.getContextClassLoader();
         }

         public void setContextClassLoader(ClassLoader cl)
         {
            Thread.currentThread().setContextClassLoader(cl);
         }

         public void setContextClassLoader(Thread thread, ClassLoader cl)
         {
            thread.setContextClassLoader(cl);
         }
      };

      TCLAction PRIVILEGED = new TCLAction()
      {
         private final PrivilegedAction getTCLPrivilegedAction = new PrivilegedAction()
         {
            public Object run()
            {
               return Thread.currentThread().getContextClassLoader();
            }
         };

         public ClassLoader getContextClassLoader()
         {
            return (ClassLoader)AccessController.doPrivileged(getTCLPrivilegedAction);
         }

         public ClassLoader getContextClassLoader(final Thread thread)
         {
            return (ClassLoader)AccessController.doPrivileged(new PrivilegedAction()
            {
               public Object run()
               {
                  return thread.getContextClassLoader();
               }
            });
         }

         public void setContextClassLoader(final ClassLoader cl)
         {
            AccessController.doPrivileged(
               new PrivilegedAction()
               {
                  public Object run()
                  {
                     Thread.currentThread().setContextClassLoader(cl);
                     return null;
                  }
               }
            );
         }

         public void setContextClassLoader(final Thread thread, final ClassLoader cl)
         {
            AccessController.doPrivileged(
               new PrivilegedAction()
               {
                  public Object run()
                  {
                     thread.setContextClassLoader(cl);
                     return null;
                  }
               }
            );
         }
      };

      ClassLoader getContextClassLoader();

      ClassLoader getContextClassLoader(Thread thread);

      void setContextClassLoader(ClassLoader cl);

      void setContextClassLoader(Thread thread, ClassLoader cl);
   }
   
   
   private static class GetSecurityContextAction implements PrivilegedAction
   {  
      GetSecurityContextAction()
      { 
      }
      public Object run()
      { 
         return SecurityContextAssociation.getSecurityContext(); 
      }
   }
   
   private static class SetSecurityContextAction implements PrivilegedAction
   { 
      private SecurityContext securityContext; 
      SetSecurityContextAction(SecurityContext sc)
      {
         this.securityContext = sc; 
      }
      
      public Object run()
      {
         SecurityContextAssociation.setSecurityContext(securityContext);
         return null;
      }
   }
   
   private static class ClearSecurityContextAction implements PrivilegedAction
   {  
      ClearSecurityContextAction()
      { 
      }
      public Object run()
      {
         SecurityContextAssociation.clearSecurityContext();
         return null;
      }
   }

   static void clearSecurityContext()
   {
      ClearSecurityContextAction action = new ClearSecurityContextAction();
      AccessController.doPrivileged(action);
   }
   
   static SecurityContext createSecurityContext() throws PrivilegedActionException
   {
      return (SecurityContext) AccessController.doPrivileged(new PrivilegedExceptionAction()
      {

         public Object run() throws Exception
         { 
            return SecurityContextFactory.createSecurityContext("CLIENT");
         }
      });
   }
   
   static SecurityContext getSecurityContext()
   {
      GetSecurityContextAction action = new GetSecurityContextAction();
      return (SecurityContext)AccessController.doPrivileged(action);
   }
   
   static void setSecurityContext(SecurityContext sc)
   {
      SetSecurityContextAction action = new SetSecurityContextAction(sc);
      AccessController.doPrivileged(action);
   }
   
   static void establishSecurityContext(String domain, Principal p, Object cred,
         Subject subject) throws Exception
   { 
      SecurityContext sc = SecurityContextFactory.createSecurityContext(p, 
            cred, subject, domain); 
      SecurityActions.setSecurityContext(sc);
   }  
}