package org.jboss.fresh.vfs.impl.disk;

import org.jboss.fresh.vfs.FileName;
import org.jboss.fresh.vfs.FileOpInfo;
import org.jboss.fresh.vfs.FileReadInfo;
import org.jboss.fresh.vfs.VFSException;
import org.jboss.fresh.vfs.VFSMeta;
import org.jboss.fresh.vfs.VFSStore;
import org.jboss.fresh.vfs.VFSStoreCacheUpdater;

import org.apache.log4j.Logger;

import java.io.File;
import java.io.RandomAccessFile;

public class DiskVFSStore implements VFSStore {

	private VFSMeta meta;
	private VFSStoreCacheUpdater cup;
	private FileName root;
	private File fsroot;

	private static Logger log = Logger.getLogger(DiskVFSStore.class);

	public DiskVFSStore(VFSMeta meta, String rootpath) throws Exception {
		this.meta = meta;
		fsroot = new File(rootpath);
		root = new FileName("/");
	}

	public void setCacheUpdater(VFSStoreCacheUpdater updater) {
		cup = updater;
	}

	public VFSStoreCacheUpdater getCacheUpdater() {
		return cup;
	}

	
	/** FileName must not be null, it can point to inexisting file.
	 */
	public boolean hasContent(FileName filename) throws Exception {

		FileName adjusted = root.absolutize(filename);
		File thefile = new File(fsroot, adjusted.toString());

		return thefile.length() != 0;
	}


	/** Writes a chunk of content data.
	 *	FileOpInfo must be filled filename (file doesn't need to exist), filepos, and len.
	 */
	public void writeContent(FileOpInfo info) throws Exception {
		FileName adjusted = root.absolutize(info.filename);
		File thefile = new File(fsroot, adjusted.toString());

		RandomAccessFile raf = new RandomAccessFile(thefile, "rw");
		try {
			if(!info.append)
				raf.setLength(0);
			else
				raf.seek(info.offset);
			raf.write(info.buf);

		} finally {
			raf.close();
		}
	}


	/** Reads a chunk of content data and returns it as FileRetInfo object.
	 *	FileOpInfo must be filled filename (file must exist), filepos and buf.
	 */
	public FileReadInfo readContent(FileOpInfo info) throws Exception {

		FileName adjusted = root.absolutize(info.filename);
		File thefile = new File(fsroot, adjusted.toString());

		RandomAccessFile raf = new RandomAccessFile(thefile, "r");
		try {
			int free = (int) (raf.length()-info.offset);
			if(free <= 0)
				throw new VFSException("Trying to read beyond the end of file");

			raf.seek(info.offset);
			
			int bufsize = 8192;
			boolean more = free > bufsize;
			if(free < bufsize)
				bufsize = free;

			FileReadInfo ri = new FileReadInfo();
			ri.buf = new byte[bufsize];

			int rc = raf.read(ri.buf);
			if(rc != ri.buf.length) {
				byte [] nubuf = new byte[rc];
				System.arraycopy(ri.buf, 0, nubuf, 0, rc);
				ri.buf = nubuf;
			}
			ri.tag = String.valueOf(raf.length());
			ri.more = more; 
			
			return ri;
		} finally {
			raf.close();
		}
	}


	/** Removes content data for specified file from persistence.
	 */
	public void removeContent(FileName name) throws Exception {

		// ves content je v podatkovni strukturi, ki bo odstranjena 
		// skupaj z metainfom
		FileName adjusted = root.absolutize(name);
		File thefile = new File(fsroot, adjusted.toString());

		if(thefile.exists() && !thefile.delete())
			throw new VFSException("Could not delete: " + name);
	}


	public void rename(FileName oldPath, FileName newPath) throws Exception {
		// v content delu ni ni� za postoriti na to temo
		FileName adjusted = root.absolutize(oldPath);
		File thefile = new File(fsroot, adjusted.toString());

		FileName adjDest = root.absolutize(newPath);
		File destFile = new File(fsroot, adjDest.toString());

		if(!thefile.renameTo(destFile))
			throw new VFSException("Could not move " + oldPath + " to " + newPath);

	}



}