package org.jboss.fresh.vfs.impl.mem;

import org.jboss.fresh.vfs.FileInfo;
import org.jboss.fresh.vfs.FileName;
import org.jboss.fresh.vfs.VFSException;
import org.jboss.fresh.vfs.VFSMeta;
import org.jboss.fresh.vfs.VFSMetaCacheUpdater;
import org.jboss.fresh.registry.RegistryContext;

import org.apache.log4j.Logger;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NotContextException;

import java.util.Collection;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;

public class MemVFSMeta implements VFSMeta {

	private static final String INTERNAL_FILE = "==__node__";
	private String sfJndiName;
	private VFSMetaCacheUpdater cup;
	RegistryContext fsroot;

	private static Logger log = Logger.getLogger(MemVFSMeta.class);

	public MemVFSMeta(String sfJndiName, String fsname) throws Exception {
		this.sfJndiName = sfJndiName;
		fsroot = new RegistryContext(fsname);

		FileInfo inf = new FileInfo(new FileName("/"));
		inf.setCreateDate(new Date());
		inf.setFileType(FileInfo.TYPE_DIR);
		inf.setLastModified(inf.getCreateDate());
		inf.setLength(0);

		Context rootCtx = (Context) fsroot.lookup("/");
		try {
			rootCtx.lookup(INTERNAL_FILE);
		} catch(NameNotFoundException ex) {
			rootCtx.bind(INTERNAL_FILE, fileInfoToFNode(inf));
		}
	}


	//***** VFSMeta METHODS *****

	public void setCacheUpdater(VFSMetaCacheUpdater updater) {
		cup = updater;
	}

	public VFSMetaCacheUpdater getCacheUpdater() {
		return cup;
	}

	public Object getResolvedNode(String name) throws VFSException, NamingException {
		Object node = null;

		if("/".equals(name)) {
			name = "";
		}
		
		for(int i=0; i<256; i++) {
			try {
				node = fsroot.lookup(name);
			} catch(NameNotFoundException ex) {
			}

			if(node instanceof FNode) {
				FNode fnode = (FNode) node;
				if(fnode.getType() == FNode.LINK) {
					name = fnode.getLinkRef();
				} else {
					return node;
				}
			} else {
				return node;
			}
		}

		throw new VFSException("Link resolution infinitely looped: " + name);
	}

	private FNode fileInfoToFNode(FileInfo finfo) {
		FNode node = new FNode();
		node.setName(finfo.getFileName().toString());
		node.setAttrs(finfo.getAttributes());
		node.setCreateDate( finfo.getCreateDate());
		node.setLastModified( finfo.getLastModified());
		node.setType(finfo.getFileType() == FileInfo.TYPE_FILE ? FNode.FILE : finfo.getFileType() == FileInfo.TYPE_DIR ? FNode.DIR : FNode.LINK);
		node.setMime(finfo.getMime());
		node.setFileContent(new MemFile());
		node.setLength(finfo.getLength());
		node.setTag(finfo.getTag());
		return node;
	}

	private FileInfo fnodeToFileInfo(FileName name, FNode node) throws VFSException {
		FileInfo inf = new FileInfo(name);
		inf.setAttributes(node.getAttrs());
		inf.setCreateDate(node.getCreateDate());
		inf.setFileType(node.getType() == FNode.FILE ? FileInfo.TYPE_FILE : node.getType() == FNode.DIR ? FileInfo.TYPE_DIR : FileInfo.TYPE_LINK);
		inf.setLastModified(node.getLastModified());
		inf.setLength(node.getLength());
		inf.setMime(node.getMime());
		inf.setTag(node.getTag());
		if(node.getLink() != null) {
			inf.setTarget(new FileName(node.getLink()));
		}
		return inf;
	}

	private void updateFNodeWithFileInfo(FNode node, FileInfo finfo) {
		node.setAttrs(finfo.getAttributes());
		node.setCreateDate( finfo.getCreateDate());
		node.setLastModified( finfo.getLastModified());
		node.setType(finfo.getFileType() == FileInfo.TYPE_FILE ? FNode.FILE : finfo.getFileType() == FileInfo.TYPE_DIR ? FNode.DIR : FNode.LINK);
		node.setMime(finfo.getMime());
		node.setLength(finfo.getLength());
		node.setTag(finfo.getTag());
	}

	private FileInfo childToFileInfo(FileName name, Object child) throws VFSException, NamingException {
		if(child instanceof FNode) {
			return fnodeToFileInfo(name, (FNode)child);
		} else if(child instanceof Context) {
			Context ctx = (Context) child;
			try {
				FNode node = (FNode) ctx.lookup(INTERNAL_FILE);
				return fnodeToFileInfo(name, node);
			} catch(NameNotFoundException ex) {
				throw new RuntimeException("Internal error - missing child node: " + INTERNAL_FILE + "  (for key: " + name + ")");				
			}
		} else {
			throw new RuntimeException("Internal error: child of illegal type: " + child.getClass().getName());
		}
	}

	/** Name must be valid FileName object.
	 */
	public boolean exists(FileName name) throws Exception {

		// Convert to CompositeName 
		// lookup RegistryContext
		// if NameNotFoundException it doesn't exist
		
		Object node = null;

		try {
			node = getResolvedNode(name.toString());
		} catch(NameNotFoundException ex) {
		}

		if(node == null)
			return false;

		return true;
	}


	public int countNamingEnum(NamingEnumeration enom) throws NamingException {
		int i = 0;
		for(;enom.hasMore();i++) {
			enom.next();
		}
		return i;
	}


	/** Path must be valid FileName object.
	 */
	public int countChildren(FileName path) throws Exception {
		// Convert to CompositeName 
		// get NameEnumeration
		// return count
		NamingEnumeration names = null;
		Object node = null;
		try {
			node = getResolvedNode(path.toString());
		} catch(NameNotFoundException ex) {}

		if(node == null) 
			throw new VFSException("File not found: " + path);

		if(!(node instanceof Context))
			throw new VFSException("File is not a directory");

		try {
			names = ((Context) node).list("");
		} catch(NotContextException ex) {
			throw new VFSException("File is not a directory");
		}

		return countNamingEnum(names);
	}


	/**	Path must point to existing file or dir and must not be null.
	 */
	public List resolvePath(FileName path) throws Exception {
		// Convert to CompositeName 
		// get FileInfo
		// as long as fileinfo is link keep looking up reference to next fileinfo
		// and adding FileInfo for every next component to return list.
		
		// if found broken link return whatever you have in list
		// first item in a list is always passed path param
		LinkedList ls = new LinkedList();
		
		if(path == null) {
			ls.add(new FileName("/"));
			return ls;
		}

		//ls.add(path);

		Object node = null;
		String name = path.toString();

		for(int i=0; i<256; i++) {
			try {
				node = fsroot.lookup(name);
			} catch(NameNotFoundException ex) {
				node = null;
			}

			if(node instanceof FNode) {
				ls.add(new FileName(name));
				FNode fnode = (FNode) node;
				if(fnode.getType() == FNode.LINK) {
					name = fnode.getLinkRef();
				} else {
					return ls;
				}
			} else {
				ls.add(new FileName(name));
				return ls;
			}
		}

		throw new VFSException("Link resolution infinitely looped: " + name);

	}


	public FileInfo getFileInfo(FileName name) throws Exception {
		// Convert to CompositeName 
		// get FileInfo
		// return null if no info for name

		Object node = getResolvedNode(name.toString());
		
		if(node == null)
			return null;
		
		if(node instanceof FNode) {
			return fnodeToFileInfo(name, (FNode) node);
		} else if(node instanceof Context) {
			Context ctx = (Context) node;
			try {
				node = ctx.lookup(INTERNAL_FILE);
			} catch(NameNotFoundException ex) {
				throw new RuntimeException("Internal error - missing child node: " + INTERNAL_FILE + "  (for key: " + name + ")");
			}
		}
		
		if(node == null)
			throw new RuntimeException("Internal error: node should not be null for key: " + name);

		return fnodeToFileInfo(name, (FNode) node);
	}


	public List list(FileName path) throws Exception {

		// Convert to CompositeName
		// enumerate values for the name
		// return a list of them

		NamingEnumeration names = null;
		Object node = null;
		try {
			node = getResolvedNode(path.toString());
		} catch(NameNotFoundException ex) {}

		if(node == null) 
			throw new VFSException("File not found: " + path);

		if(!(node instanceof Context))
			throw new VFSException("File is not a directory");

		try {
			names = ((Context) node).listBindings("");
		} catch(NotContextException ex) {
			throw new VFSException("File is not a directory");
		}

		LinkedList ls = new LinkedList();
		while(names.hasMore()) {
			Binding binding = (Binding) names.next();
			if(! INTERNAL_FILE.equals(binding.getName()))
				ls.add(childToFileInfo(path.absolutize(binding.getName().toString()), binding.getObject()));
		}

		return ls;
	}

	public Collection getLinks(FileName target) throws Exception {

		// return all fileinfos that link to this one
		// why would we need that?
		// it means we need to keep reverse references - a pain
		throw new RuntimeException();
	}

	public void create(FileInfo fi) throws Exception {
		// save fi under CompositeName
		// what if it exists already ?
		// Error, it must not exist

		try {
			// check father
			// Noro - v primeru direktorija moramo narediti nov context !!!
			if(fi.getFileType() == FileInfo.TYPE_DIR) {

				try {
					fsroot.lookup(fi.getFileName().toString());
					throw new VFSException("Directory exists already: " + fi.getFileName());
				} catch(NameNotFoundException ex) {
				}

				Context nuctx = fsroot.createSubcontext(fi.getFileName().toString());
				nuctx.bind(INTERNAL_FILE, fileInfoToFNode(fi));
			} else {
				fsroot.bind(fi.getFileName().toString(), fileInfoToFNode(fi));
			}
		} catch(NamingException ex) {
			throw new VFSException("Could not create file: " + fi.getFileName(), ex);
		}
	}


	// lahko spremeni katerikoli metainfo, samo filename-a ne (na ravni tabel se torej ne spremeni filename, parent in name),
	// ker je to edini primary key, ki ga imam na voljo iz parametrov;
	// problem pri primary key-ih pa je v drugih tabelah, kjer bo treba najprej vse vrstice pobrisat in nato
	// novo vnest... (ni druge; ce bi hoteli ugotoviti, kaj ekzactly se updata, bi bilo se bolj potratno... odvisno od primera)
	public void update(FileInfo fi) throws Exception {
		// look up info for CompositeName
		// apply new info to it
		Object node = null;
		try {
			node = fsroot.lookup(fi.getFileName().toString());
		} catch(NameNotFoundException ex) {
		}
		
		if(node == null)
			throw new VFSException("File does not exist: " + fi.getFileName());
		
		if(node instanceof FNode) {
			updateFNodeWithFileInfo((FNode) node, fi);
		} else if(node instanceof Context) {
			node = ((Context) node).lookup(INTERNAL_FILE);
			updateFNodeWithFileInfo((FNode) node, fi);
		} else {
			throw new RuntimeException("Internal error - Node is of wrong type: " + node.getClass().getName());
		}
	}


	// ko klices getInt na result setu, pa da je ta int nastavljen na NULL, dobis nazaj 0! Zato ne sme biti noben index 0!
	public void remove(FileName name) throws Exception {
		// remove entry
		// if directory you need to remove context
		Object node = null;
		try {
			node = fsroot.lookup(name.toString());
		} catch(NameNotFoundException ex) {
		}
		
		if(node == null)
			throw new VFSException("File does not exist: " + name);
		
		if(node instanceof FNode) {
			fsroot.unbind(name.toString());
		} else if(node instanceof Context) {
			// remove INTERNAL_FILE first
			String nnode = name.toString() + INTERNAL_FILE;
			Object co = fsroot.lookup(nnode);
			fsroot.unbind(nnode);
			try {
				fsroot.destroySubcontext(name.toString());
			} catch (Exception ex) {
				fsroot.bind(nnode, co);
				throw ex;
			} catch (Throwable t) {
				fsroot.bind(nnode, co);
				throw new RuntimeException("Error while removing directory: ", t);
			}
		} else {
			throw new RuntimeException("Internal error - Node is of wrong type: " + node.getClass().getName());
		}
		
	}


	public void rename(FileName oldPath, FileName newPath) throws Exception {
		// lookup structure and change the name
		// rename is same as move - should be called move
		Object node = null;

		try {
			node = fsroot.lookup(oldPath.toString());
		} catch(NameNotFoundException ex) {
		}

		if(node == null)
			throw new RuntimeException("Internal error - Node does not exist: " + oldPath.toString());

		if(node instanceof FNode) {
			fsroot.rename(oldPath.toString(), newPath.toString());
		} else if(node instanceof Context) {
			// rename is already covered in naming
			// we have to move it first
			fsroot.rename(oldPath.toString(), newPath.toString());
			if(!newPath.getName().equals(oldPath.getName())) {
				try {
					node = fsroot.lookup(newPath.toString() + INTERNAL_FILE);
				} catch(NameNotFoundException ex) {
				}

				if(node == null)
					throw new RuntimeException("Internal error - Node does not exist: " + newPath.toString() + INTERNAL_FILE);
				
				((FNode) node).setName(newPath.getName());
			}
		} else {
			throw new RuntimeException("Internal error - Node is of wrong type: " + node.getClass().getName());
		}

	}


	public float moveBefore(FileName currFile, FileName nextFile) throws Exception {
		// this operation moves a file whithin it's parent. Native FSs usually don't support this
		throw new RuntimeException("Method not implemented");
	}

}